package xnap.gui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;

/**
 * ValidatedTextField is a subclass of JTextField that validates input as its
 * typed into the field.
 */
public class ValidatedTextField extends JTextField {

    // ------------------------------------------------------------------------
    // Public Static Constants
    // ------------------------------------------------------------------------

    public static final String ALPHABET_LOWER = "abcdefghijklmnopqrstuvwxyz";
    public static final String ALPHABET_UPPER = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    public static final String NUMBERS_INT = "0123456789";
    public static final String NUMBERS_DECIMAL = NUMBERS_INT + ".";
    public static final String MONEY_USD = "$" + NUMBERS_DECIMAL;
    public static final String ALPHA_NUM 
	= ALPHABET_LOWER + ALPHABET_UPPER + NUMBERS_INT;
    public static final String EMAIL = ALPHA_NUM + ".@_-&";
    public static final String ANYTHING = null;

    // ------------------------------------------------------------------------
    // Private Instance Data
    // ------------------------------------------------------------------------

    private ValidatedDocument _textDocument;

    // ------------------------------------------------------------------------
    // Public Constructors
    // ------------------------------------------------------------------------

    /**
     * Constructs a new text field object initialized with the specified
     * text and columns.
     *
     * @param text The text to be display, or null
     * @param columns The number of columns to use to calculate the preferred
     *        width. If columns is set to zero, the preferred width will be
     *        whatever naturally results from the component implementation.
     * @param validChars The list of valid (allowable) characters, or null if
     *        anything is allowed.
     */
    public ValidatedTextField(String text, int columns, String validChars) 
    {
        super(text, columns);

	initialize(validChars);

        setText(text);
    }

    /**
     * Constructs a new text field object initialized with the specified
     * text.
     *
     * @param text The text to be display, or null
     * @param validChars The list of valid (allowable) characters, or null if
     *        anything is allowed.
     */
    public ValidatedTextField(String text, String validChars)
    {
        super(text);

	initialize(validChars);

        setText( text );
    }

    /**
     * Constructs a new empty text field object initialized with the specified
     * columns.
     *
     * @param columns The number of columns to use to calculate the preferred
     *        width. If columns is set to zero, the preferred width will be
     *        whatever naturally results from the component implementation.
     * @param validChars The list of valid (allowable) characters, or null if
     *        anything is allowed.
     */
    public ValidatedTextField(int columns, String validChars) 
    {
        super(columns);

	initialize(validChars);
    }

    /**
     * Constructs a new empty text field object.
     *
     * @param validChars The list of valid (allowable) characters, or null if
     *        anything is allowed.
     */
    public ValidatedTextField(String validChars) 
    {
	initialize(validChars);
    }

    /**
     * Constructs a new empty text field object.
     */
    public ValidatedTextField()
    {
	initialize(null);
    }

    // ------------------------------------------------------------------------
    // Public Methods
    // ------------------------------------------------------------------------

    public void initialize(String validChars)
    {
	if (validChars == NUMBERS_INT) {
	    setHorizontalAlignment(JTextField.RIGHT);
	}
        _textDocument = new ValidatedDocument(validChars);
        setDocument( _textDocument );
    }

    /**
     * Gets the string of characters used to validate user-input against.
     */
    public String getValidChars() {
        return _textDocument.getValidChars();
    }

    /**
     * Sets the string of characters used to validate user-input against.
     *
     * @param validChars Any string of characters, or null if anything is allowed.
     */
    public void setValidChars( String validChars ) {
        _textDocument.setValidChars( validChars );
    }

    public int getIntValue()
    {
	try {
	    return Integer.parseInt(getText());
	}
	catch (NumberFormatException e) {
	    return 0;
	}  
    }
}


/**
 * ValidatedDocument is a private subclass of PlainDocument.  It intercepts the
 * "insertString()" message and validates the text.  The portion of text that
 * passes the validation test gets passed to super.insertString() for normal
 * processing.
 *
 * @author  Andrew H. Blanchard
 *          Portions of this code were taken from
 *     http://java.sun.com/docs/books/tutorial/uiswing/components/textfield.html
 */
class ValidatedDocument extends PlainDocument {

    // --------------------------------------------------------------------------
    // Private Instance Data
    // --------------------------------------------------------------------------

    private String _validChars = null;

    // --------------------------------------------------------------------------
    // Public Constructors
    // --------------------------------------------------------------------------

    /**
     * Constructs a new ValidatedDocument.
     *
     * @param validChars The string of characters to validate user-input against.
     */
    public ValidatedDocument( String validChars ) {
        super();
        _validChars = validChars;
    }

    // --------------------------------------------------------------------------
    // Public Methods
    // --------------------------------------------------------------------------

    /**
     * When new data is placed in the text-field, insertString() is called.
     * This method validates the string to insert.  The portion of the string
     * that passes the validation is forwarded to super.insertString() for actual
     * insertion into the document model.  This is a call-back method and
     * shouldn't normally be used directly.
     */
    public void insertString( int offs, String str, AttributeSet a )
	throws BadLocationException {
        char[] source = str.toCharArray();
        char[] result = new char[source.length];
        int j = 0;

        for (int i = 0; i < result.length; i++) {
            if (isValidChar(source[i])) {
                result[j++] = source[i];
            }
        }

        super.insertString(offs, new String(result, 0, j), a);
    }

    /**
     * Gets the string of characters used to validate user-input against.
     */
    public String getValidChars() {
        return _validChars;
    }

    /**
     * Sets the string of characters used to validate user-input against.
     *
     * @param validChars Any string of characters, or null if anything is
     *        allowed.
     */
    public void setValidChars( String validChars ) {
        _validChars = validChars;
    }

    // ------------------------------------------------------------------------
    // Private Methods
    // ------------------------------------------------------------------------

    /**
     * Determines if a characters is valid, based on the string of valid
     * characters.
     */
    private boolean isValidChar( char ch ) {
        if ( _validChars == null ) {
            return true;
        } else {
            return ( _validChars.indexOf( ch ) >= 0 );
        }
    }

}
