/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.menu;

import xnap.XNap;
import xnap.gui.table.AbstractDynamicTableModel;
import xnap.gui.table.Column;
import xnap.gui.table.LibraryTableModel;
import xnap.util.Preferences;
import xnap.util.StringHelper;
import xnap.util.TablePreferencesSupport;

import java.awt.event.*;
import java.beans.*;
import java.util.*;
import javax.swing.*;

/**
 * Used for menus that control table column visibility.
 */
public class TableColumnsMenu extends JMenu
    implements ActionListener, PropertyChangeListener
{
    
    //--- Constant(s) ---
    
    //--- Data field(s) ---
    
    private String table;
    private TablePreferencesSupport tps;
    private JCheckBoxMenuItem jcbSortorder;
    private JCheckBoxMenuItem[] buttons = new JCheckBoxMenuItem[0];

    //--- Constructor(s) ---

    public TableColumnsMenu(String table, String[] columns, 
			    TablePreferencesSupport tps)
    {
	if (tps == null) {
	    tps = Preferences.getInstance();
	}

	this.table = table;
	this.tps = tps;

	SortOrderAction acSortOrder = new SortOrderAction();
	jcbSortorder = new JCheckBoxMenuItem(acSortOrder);
	jcbSortorder.setSelected(tps.getTableMaintainSortOrder(table));
	add(jcbSortorder);
	addSeparator();

	buttons = new JCheckBoxMenuItem[columns.length];
	for (int i = 0; i < buttons.length; i++) {
	    JCheckBoxMenuItem jcb = new JCheckBoxMenuItem(columns[i]);
	    jcb.addActionListener(this);
	    add(jcb);
	    buttons[i] = jcb;
	}

	setVisibleColumns(tps.getTableColumnsArray(table));
	tps.addTableListener(table, this);
    }

    public TableColumnsMenu(String table, String[] columns)
    {
	this(table, columns, null);
    }

    public TableColumnsMenu(AbstractDynamicTableModel model)
    {
	this(model.getTableIdentifier(), model.getColumnNames(), 
	     model.getPreferencesSupport());

	setText(model.getTableName());
	setToolTipText(XNap.tr("Show selected columns"));
    }

    //--- Method(s) ---

    public void actionPerformed(ActionEvent event)
    {
	if (event.getSource() instanceof JCheckBoxMenuItem) {
	    for (int i = 0; i < buttons.length; i++) {
		if (buttons[i] == event.getSource()) {
		    int[] cols = tps.getTableColumnsArray(table);
		    if (buttons[i].isSelected()) {
			cols = add(cols, i);
		    }
		    else {
			cols = remove(cols, i);
		    }
		    tps.setTableColumns(table, cols);
		    return;
		}
	    }
	}
    }

    public int[] getVisibleColumns()
    {
	int[] visibleColumns = new int[buttons.length]; 
	int count = 0;
	for (int i = 0; i < buttons.length; i++) {
	    if (buttons[i].isSelected()) {
		visibleColumns[count] = i;
		count++;
	    }
	}
	int[] array = new int[count];
	System.arraycopy(visibleColumns, 0, array, 0, array.length);
	return array;
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	setVisibleColumns(tps.getTableColumnsArray(table));
	jcbSortorder.setSelected(tps.getTableMaintainSortOrder(table));
    }

    /**
     * Updates buttons.
     */
    public void setVisibleColumns(int[] visibleColumns)
    {
	for (int i = 0; i < buttons.length; i++) {
	    buttons[i].setSelected(false);
	}
	for (int i = 0; i < visibleColumns.length; i++) {
	    int c = visibleColumns[i];
	    if (c >= 0 && c < buttons.length) {
		buttons[c].setSelected(true);
	    }
	}
    }

    /**
     * 
     */
    public static int[] add(int[] oldArray, int element)
    {
	int[] newArray = new int[oldArray.length + 1];
	int count = 0;
	boolean added = false;
	for (int i = 0; i < oldArray.length; i++) {
	    added = added | (oldArray[i] == element);
	    newArray[count] = oldArray[i];
	    count++;
	}

	if (!added) {
	    newArray[count] = element;
	    count++;
	}
	    
	int[] array = new int[count];
	System.arraycopy(newArray, 0, array, 0, array.length);
	return array;
    }

    /**
     * This is O(n^2).
     */
    public static int[] remove(int[] oldArray, int element)
    {
	int[] newArray = new int[oldArray.length];
	int count = 0;
	for (int i = 0; i < oldArray.length; i++) {
	    if (oldArray[i] != element) {
		newArray[count] = oldArray[i];
		count++;
	    }
	}
	int[] array = new int[count];
	System.arraycopy(newArray, 0, array, 0, array.length);
	return array;
    }

    protected class SortOrderAction extends AbstractAction {
	
	public SortOrderAction()
	{
	    putValue(Action.NAME, XNap.tr("Maintain Sort Order"));
	    putValue(Action.SHORT_DESCRIPTION, 
		     XNap.tr("If enabled, the sort order will be maintained as new items are added to the table."));
	}

	public void actionPerformed(ActionEvent event)
	{
	    if (event.getSource() instanceof JCheckBoxMenuItem) {
		JCheckBoxMenuItem jcb = (JCheckBoxMenuItem)event.getSource();
		tps.setTableMaintainSortOrder(table, jcb.isSelected());
	    }
	}

    }

}
