/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.prefs;

import xnap.XNap;
import xnap.gui.*;
import xnap.gui.table.*;
import xnap.gui.theme.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

public class LookAndFeelPrefsPanel extends AbstractPreferencesPanel 
    implements ListSelectionListener {
    
    //--- Data field(s) ---

    private DefaultListModel dlmLafs;
    private JList jlLafs;
    private DefaultListModel dlmThemes;
    private JList jlThemes;
    private JPanel jpPreview;
    private int lastIndex = 0;
    private ConfigureThemeAction acConfigureTheme = new ConfigureThemeAction();

    private Object[] data = {
	"Berlin", "Bremen", "Hamburg", "Muenchen", "Stuttgart"
    };

    //--- Constructor(s) ---

    public LookAndFeelPrefsPanel()
    {
	setLayout(new GridBagLayout());

	// look and feel list
	JPanel jpLafs = new JPanel(new BorderLayout());
	jpLafs.add(new JLabel(XNap.tr("Look and Feel")), BorderLayout.NORTH);
	dlmLafs = new DefaultListModel();
	jlLafs = new JList(dlmLafs);
	jlLafs.setVisibleRowCount(4);
	JScrollPane jspLafs = new JScrollPane(jlLafs);
	jpLafs.add(jspLafs, BorderLayout.CENTER);

	// theme list
	JPanel jpThemes = new JPanel(new BorderLayout());
	jpThemes.add(new JLabel(XNap.tr("Theme")), BorderLayout.NORTH);
	dlmThemes = new DefaultListModel();
	jlThemes = new JList(dlmThemes);
	jlThemes.setVisibleRowCount(4);
	JScrollPane jspThemes = new JScrollPane(jlThemes);
	jpThemes.add(jspThemes, BorderLayout.CENTER);

	// look and feel panel
	JPanel jpLists = new JPanel();
	jpLists.setLayout(new BoxLayout(jpLists, BoxLayout.X_AXIS));
	jpLists.add(jpLafs);
	jpLists.add(Box.createHorizontalStrut(5));
	jpLists.add(jpThemes);
	GridBagHelper.add(this, jpLists);
	
	// configure button
	JPanel jpConfigure = new JPanel(new FlowLayout(FlowLayout.RIGHT, 5, 0));
	JButton jbConfigure = new JButton(acConfigureTheme);
	jpConfigure.add(jbConfigure);
	GridBagHelper.add(this, jpConfigure);

	// preview
	JPanel jpControls = new JPanel();
	jpControls.setLayout(new BoxLayout(jpControls, BoxLayout.Y_AXIS));
	jpControls.add(new JLabel("I am a Label"));
	jpControls.add(new JButton("I am a button"));
	jpControls.add(new JCheckBox("I am a check box"));
	jpControls.add(new JRadioButton("I am a radio"));

	JPanel jpSlider = new JPanel();
	jpSlider.setLayout(new BoxLayout(jpSlider, BoxLayout.Y_AXIS));
	JSlider js = new JSlider(JSlider.HORIZONTAL, 0, 10, 3);
	js.setSnapToTicks(true);
	js.setPreferredSize(new Dimension(50, 50));
	jpSlider.add(js);
	JScrollBar jsb = new JScrollBar(JScrollBar.HORIZONTAL, 5, 2, 0, 10);
	jpSlider.add(jsb);

	JTabbedPane jtp = new JTabbedPane();
	jtp.addTab("Controls", jpControls);
	jtp.addTab("Slider", jpSlider);

	JTable jtTable = (new MyTableModel()).createJTable();
	jtTable.setPreferredScrollableViewportSize(new Dimension(50, 50));
	JScrollPane jspTable = new JScrollPane(jtTable);

	JTree jtTree = new JTree();
	jtTree.setVisibleRowCount(4);
	JScrollPane jspTree = new JScrollPane(jtTree);

	jpPreview = new JPanel(new GridLayout(1, 3, 10, 10));
	jpPreview.add(jtp);
	jpPreview.add(jspTable);
	jpPreview.add(jspTree);

	JPanel jpWrap = new JPanel(new BorderLayout());
	jpWrap.setBorder(new TitledBorder(XNap.tr("Preview", 1)));
	jpWrap.add(jpPreview, BorderLayout.CENTER);
	GridBagHelper.add(this, jpWrap);

	GridBagHelper.addVerticalSpacer(this);

	initializeLists();
    }

    public void initializeLists()
    {
	UIManager.LookAndFeelInfo[] info 
	    = ThemeManager.getInstalledLookAndFeels();
	String currentLaf = UIManager.getLookAndFeel().getClass().getName();
	for (int i = 0; i < info.length; i++) {
	    dlmLafs.addElement(new MyLookAndFeelInfo(info[i]));
	    if (currentLaf.equals(info[i].getClassName())) {
		jlLafs.setSelectedIndex(i);
		jlLafs.ensureIndexIsVisible(i);
		lastIndex = i;
	    }
	}
	jlLafs.addListSelectionListener(this);

	Theme[] themes = ThemeManager.getThemes();
	String currentTheme = prefs.getInstance().getTheme();
	for (int i = 0; i < themes.length; i++) {
	    dlmThemes.addElement(themes[i]);
	    if (currentTheme.equals(themes[i].getClassName())) {
		jlThemes.setSelectedIndex(i);
		jlThemes.ensureIndexIsVisible(i);
	    }
	}
	jlThemes.addListSelectionListener(this);
	acConfigureTheme.setEnabled(getSelectedTheme().isConfigurable());
    }

    public void apply() 
    {
	prefs.setLookAndFeel(getSelectedLaf().getClassName());
	prefs.setTheme(getSelectedTheme().getClassName());
    }

    public Icon getIcon()
    {
	return null;
    }

    public UIManager.LookAndFeelInfo getSelectedLaf()
    {
	return ((MyLookAndFeelInfo)jlLafs.getSelectedValue()).info;
    }

    public Theme getSelectedTheme()
    {
	return (Theme)jlThemes.getSelectedValue();
    }

    public String getTitle()
    {
	return XNap.tr("Look & Feel");
    }

    public void updateTheme()
    {
	// save current theme
	Object[] savedDefaults = ThemeManager.getDefaults();
	
	// set theme
	ThemeManager.setTheme(getSelectedTheme());
	SwingUtilities.updateComponentTreeUI(jpPreview);

	// reset theme
	ThemeManager.setDefaults(savedDefaults);

	// enable configuration dialog
	acConfigureTheme.setEnabled(getSelectedTheme().isConfigurable());
    }

    /**
     * Sets the selected look and feel for the preview panel.
     */
    public void valueChanged(ListSelectionEvent event) 
    {
	if (event.getSource() == jlLafs) {
	    String currentLaf = UIManager.getLookAndFeel().getClass().getName();

	    // set laf
	    try {
		UIManager.setLookAndFeel(getSelectedLaf().getClassName());
	    }
	    catch (Exception e) {
		xnap.util.Debug.log("could not load look and feel");
		e.printStackTrace(System.out);
		JOptionPane.showMessageDialog
		  (this, XNap.tr("Sorry, not supported or not installed"),
		     "Look and Feel", JOptionPane.ERROR_MESSAGE);
		int i = jlLafs.getSelectedIndex();
		jlLafs.setSelectedIndex((i == dlmLafs.size() - 1) ? 0 : i + 1);
		return;
	    }

	    // redraw component
	    SwingUtilities.updateComponentTreeUI(jpPreview);

	    // reset laf
	    lastIndex = jlLafs.getSelectedIndex();
	    try {
		UIManager.setLookAndFeel(currentLaf);
	    }
	    catch (Exception e) {
	    }
	}
	else if (event.getSource() == jlThemes) {
	    updateTheme();
	}
    }

    /**
     * Wraps toString() for UIManager.LookAndFeelInfo class.
     */
    class MyLookAndFeelInfo  {

	public UIManager.LookAndFeelInfo info;
	
	public MyLookAndFeelInfo(UIManager.LookAndFeelInfo info)
	{
	    this.info = info;
	}

	public String getClassName()
	{
	    return info.getClassName();
	}

	public String toString()
	{
	    return info.getName();
	}

    }

    private class MyTableModel extends AbstractDynamicTableModel {

	//--- Data field(s) ---

	private Column columns[] = new Column[] {
	    new Column("Operating System", String.class),
	    new Column("License", String.class),
	};

	private Object[][] data = {
	    { "Linux"  , "GPL"         },
	    { "FreeBSD", "BSD License" },
	};

	//--- Constructor(s) ---

	public MyTableModel() 
	{
	    setColumns(columns);
	    fireTableDataChanged();
	}

	//--- Method(s) ---

	public Object get(int i, int j) 
	{
	    return data[i][j];
        }

	public int getRowCount() 
	{
	    return data.length;
	}

	public String getTableName()
	{
	    return "";
	}

	public void set(Object o, int i, int j)
	{
	}

    }

    private class ConfigureThemeAction extends AbstractAction
    {
	public ConfigureThemeAction()
	{
	    putValue(Action.NAME, XNap.tr("Configure Theme") + "...");
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Opens the theme configuration dialog"));
            putValue(Action.MNEMONIC_KEY, new Integer(KeyEvent.VK_C));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    if (getSelectedTheme().showConfigurationDialog
		(LookAndFeelPrefsPanel.this)) {
		updateTheme();
	    }
        }

    }
 
}
