/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.util;

import xnap.util.Preferences;

import java.awt.*;
import java.util.*;
import javax.swing.*;
import org.apache.log4j.Logger;

/**
 * Helps with gui related tasks. We can not have <code>Preferences.java</code>
 * depend on java.awt.
 */
public class GUIHelper
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    protected static Preferences prefs = Preferences.getInstance();
    //protected static Logger logger = Logger.getLogger(GUIHelper.class);

    //--- Constructor(s) ---

    //--- Method(s) ---

    public static void bindToKey(JButton jb, KeyStroke ks, String name)
    {
	InputMap inputMap = jb.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
	ActionMap actionMap = jb.getActionMap();
	if (inputMap != null && actionMap != null) {
	    inputMap.put(ks, name);
	    actionMap.put(name, jb.getAction());
	}
    }

    public static Color getColor(String colorPref)
    {
	return decodeColor(prefs.getColor(colorPref));
    }

    /**
     * Returns a font from the preferences.
     */
    public static Font getFont(String fontPref)
    {
	return decodeFont(prefs.getFont(fontPref));
    }

    public static Color decodeColor(int color)
    {
	return new Color(color & 0xFF, (color >> 8) & 0xFF, 
			 (color >> 16) & 0xFF);
    }

    public static int encodeColor(int red, int green, int blue)
    {
	return red + 256 * green + (256 * 256) * blue;
    }

    public static int encodeColor(Color c)
    {
	return encodeColor(c.getRed(), c.getGreen(), c.getBlue());
    }

    /**
     * Always returns a valid font.
     */
    public static Font decodeFont(String fontString)
    {
	StringTokenizer t = new StringTokenizer(fontString, 
						Preferences.ARRAY_SEPARATOR);

	if (t.countTokens() >= 3) {
	    try {
		String name = t.nextToken();
		int style = Integer.parseInt(t.nextToken());
		int size = Integer.parseInt(t.nextToken());

		return new Font(name, style, size);
	    }
	    catch (NumberFormatException e) {
	    }
	}

	return new Font("Monospaced", 0, 12);
    }

    public static String encodeFont(Font font)
    {
	StringBuffer sb = new StringBuffer();
	sb.append(font.getName());
	sb.append(Preferences.ARRAY_SEPARATOR);
	sb.append(font.getStyle());
	sb.append(Preferences.ARRAY_SEPARATOR);
	sb.append(font.getSize());

	return sb.toString();
    }

    public static void setAccelerator(JMenuItem jmi, int keyCode)
    {
 	int mask = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
	jmi.setAccelerator(KeyStroke.getKeyStroke(keyCode, mask));
    }

    public static void setColor(String colorPref, Color color)
    {
	prefs.setColor(colorPref, encodeColor(color));
    }

    /**
     * Saves a font to the preferecnes.
     */
    public static void setFont(String fontPref, Font font)
    {
	prefs.setFont(fontPref, encodeFont(font));
    }

    public static void setMnemonics(Container c)
    {
	HashSet letters = new HashSet();

	for (int i = 0; i < c.getComponentCount(); i++) {
	    Component component = c.getComponent(i);

	    if (component instanceof AbstractButton) {
		AbstractButton ab = (AbstractButton)component;
		if (ab.getMnemonic() == 0) {
		    setMnemonics(ab, letters);
		}
		else {
		    letters.add(new Integer(ab.getMnemonic()));
		}
	    }

	    if (component instanceof JMenu) {
		setMnemonics(((JMenu)component).getPopupMenu());
	    }

	    // recurse
	    if (component instanceof Container) {
		setMnemonics((Container)component);
	    }
	}
    }

    private static boolean setMnemonics(AbstractButton ab, HashSet letters)
    {
	String text = ab.getText().toUpperCase();

	// try first letters of words first
	StringTokenizer t = new StringTokenizer(text);
	while (t.hasMoreTokens()) {
	    Integer character = new Integer((int)t.nextToken().charAt(0));
	    if (!letters.contains(character)) {
		letters.add(character);
		ab.setMnemonic(character.intValue());
		return true;
	    }
	}

	// pick any character, start with the second one
	// the first one has already been checked
	for (int i = 1; i < text.length(); i++) {
	    Integer character = new Integer((int)text.charAt(i));
	    if (text.charAt(i) != ' ' && !letters.contains(character)) {
		letters.add(character);
		ab.setMnemonic(character.intValue());
		return true;
	    }
	}

	return false;
    }

    /**
     * Wraps HTML tags around <code>text</code> so the maximum width
     * is limited to a senseful value.
     *
     * @return text, enclosed in table html tags
     */
    public static String tt(String text)
    {
	StringBuffer sb = new StringBuffer(33 + text.length() + 25);

	sb.append("<html><table><tr><td width=\"300\">");
	sb.append(text);
	sb.append("</td></tr></table></html>");

	return sb.toString();
    }

    public static void limitSize(JComponent c)
    {
	c.setMaximumSize(new Dimension(c.getPreferredSize().width,
				       c.getMaximumSize().height));

    }

}
