/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.io;

import xnap.*;
import xnap.cmdl.*;
import xnap.util.*;
import xnap.util.event.*;

import java.beans.*;
import java.io.*;
import java.util.*;
import org.apache.log4j.Logger;

public class Repository extends AbstractRepository
    implements PropertyChangeListener {

    //--- Constant(s) ---

    public static int VERSION = 2;
    
    // --- Data Field(s) ---

    protected static Logger logger = Logger.getLogger(Repository.class);
    private static Repository singleton = new Repository();

    protected Node rootNode = new Node();
    private Preferences prefs = Preferences.getInstance();
    private StatusListener listener;
    private boolean autoAddToSearchTree = true; //prefs.getBuildSearchTreeOnStartup();
    
    // --- Constructor(s) ---
    
    private Repository()
    {
	super(FileHelper.getHomeDir() + "shared", VERSION);

	readAndUpdateLater();

	Preferences.getInstance().addPropertyChangeListener(this);

	Executer.addCommand(new ListCmd());
	Executer.addCommand(new UpdateCmd());
    }

    // --- Method(s) ---

    public static Repository getInstance()
    {
	return singleton;
    }

    /**
     * Blocks until the search tree is blocked.
     */
    public void buildSearchTree()
    {
	if (!autoAddToSearchTree) {
	    autoAddToSearchTree = true;
	    
	    int count = size();
	    for (int i = 0; i < count; i++) {
		File f = getFile(i);
		if (f != null) {
		    addToSearchTree(f);
		}
	    }
	}
    }

    public String getDirs()
    {
	return prefs.getUploadDirs();
    }

    public RepositoryFile getFile(int i)
    {
	return (RepositoryFile)super.get(i);
    }

    public synchronized void updateStatus()
    {
	if (listener != null) {
	    if (isUpdateRunning()) {
		listener.setStatus(getNonNullSize() + 
				   XNap.tr("updating", 1, 0));
	    }
	    else {
		listener.setStatus(getNonNullSize() + 
				   XNap.tr("shared", 1, 0));
	    }
	}
    }

    public synchronized void setStatusListener(StatusListener newValue) {
	listener = newValue;
	updateStatus();
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	String p = e.getPropertyName();

	if (e.getPropertyName().equals("uploadDirs")) {
	    updateLater();
	}
    }

    public File[] search(String searchText)
    {
	buildSearchTree();

	logger.info("searching for " + searchText);

	HashSet results = new HashSet();
	searchText = StringHelper.stripExtra(searchText);
	StringTokenizer t = new StringTokenizer(searchText.toLowerCase(), " ");
	boolean firstRun = true;
	while (t.hasMoreTokens()) {
	    Node node = rootNode;
	    String s = t.nextToken();
	    for (int i = 0; i < s.length(); i++) {
		Character c = new Character(s.charAt(i));
		node = (Node)node.children.get(c);
		if (node == null) {
		    return null;
		}
	    }
	    if (firstRun) {
		for (Iterator i = node.files.iterator(); i.hasNext();) {
		    results.add(i.next());
		}
	    }
	    else {
		// tokens are AND related
		// build average between results of last run and results of
		// this run 
		HashSet newResults = new HashSet();
		for (Iterator i = node.files.iterator(); i.hasNext();) {
		    Object o = i.next();
		    if (results.contains(o)) {
			newResults.add(o);
		    }
		}
		results = newResults;

		if (results.size() == 0) {
		    return null;
		}
	    }
	    firstRun = false;
	}

	File[] array = new File[results.size()];
	System.arraycopy(results.toArray(), 0, array, 0, array.length);
	return array;
    }

    public File[] search(String searchText, long filesize)
    {
	File[] results = search(searchText);
	LinkedList filtered = new LinkedList();

	if (results == null)
	    return null;

	for (int i = 0; i < results.length; i++) {
	    if (results[i].length() == filesize) {
		filtered.add(results[i]);
	    }
	}
	File[] array = new File[filtered.size()];
	System.arraycopy(filtered.toArray(), 0, array, 0, array.length);
	return array;
    }

    /**
     * Get file at <code>index</code> and compare it to the given filename. If
     * they match return the file otherwise null.
     */
    public File getRepositoryFile(String filename, int index)
    {
	if (index >= 0 && index < size()) {
	    File file = (File)get(index);

	    if (file != null && file.getName().equals(filename)) {
		return file;
	    }
	} 

	return null;
    }

    protected void addNewFile(File f)
    {
	if (!(f instanceof RepositoryFile)) {
	    f = FileHandler.handle(f, true);
	}

	super.addNewFile(f);

	if (getNonNullSize() % 16 == 0) {
	    updateStatus();
	}

	addToSearchTree(f);
    }

    /**
     * Adds <code>f</code> to search tree.
     */
    private void addToSearchTree(File f)
    {
	if (!autoAddToSearchTree) {
	    return;
	}

	String name = StringHelper.stripExtra(f.getAbsolutePath());
	StringTokenizer t = new StringTokenizer(name.toLowerCase(), " ");
	while (t.hasMoreTokens()) {
	    Node node = rootNode;
	    String s = t.nextToken();
	    //logger.debug("adding " + s + " to search tree");
	    for (int i = 0; i < s.length(); i++) {
		Character c = new Character(s.charAt(i));
		Node child = (Node)node.children.get(c);
		if (child == null) {
		    child = new Node();
		    node.children.put(c, child);
		}
		node = child;
	    }
	    node.files.add(f);
	}
    }

    /**
     * Does not share incomplete folder and files starting with a dot.
     */
    protected boolean isPartOfRepository(File f)
    {
	if (f.getName().startsWith(".")) {
	    return false;
	}
	else if (f.isDirectory()) {
	    return !f.equals(new File(prefs.getIncompleteDir()));
	}
	return true;
    }
    
    protected void preRead()
    {
	updateStatus();
    }

    protected void postRead(boolean success, String response) 
    {
	// don't care if we can't read repository, just rebuild it
	allowWrite = true;
    }

    protected void preUpdate()
    {
	updateStatus();
    }

    protected void postUpdate() 
    {
	updateStatus();
    }

    protected class ListCmd extends AbstractCommand
    {
	public ListCmd()
	{
	    putValue(CMD, new String[] {"listrepository"});
	    putValue(SHORT_HELP, "Shows the repository.");
	}

	public boolean execute(String[] parms)
	{
  	    int i = 0; 

	    for (Iterator it = Repository.this.iterator(); it.hasNext(); 
		 i++) {
		RepositoryFile f = (RepositoryFile)it.next();
		if (f != null) {
		    Console.getInstance().println(i + ": " + f.getName());
		}
  	    }

	    return true;
  	}
    }

    protected class UpdateCmd extends AbstractCommand
    {
	public UpdateCmd()
	{
	    putValue(CMD, new String[] {"updaterepository"});
	    putValue(SHORT_HELP, "Updates the repository.");
	}

	public boolean execute(String[] parms)
	{
  	    updateLater();
	    return true;
  	}
    }

    protected class Node {
	
	public HashSet files = new HashSet();
	public Hashtable children = new Hashtable();

    }

}
