/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.io;

import xnap.util.*;

import java.util.*;
import org.apache.log4j.Logger;

public class ThrottleSupport
{

    //--- Constant(s) ---

    public static final long TICK_LENGTH = 250;

    //--- Data field(s) ---

    private static Logger logger = Logger.getLogger(ThrottleSupport.class);

    /**
     * bytes per tick
     */
    protected long bandwidth = 0;
    protected long allocated = 0;
    protected long tickStart = 0;
    protected Object lock = new Object();

    //--- Constructor(s) ---

    public ThrottleSupport()
    {
    }

    //--- Method(s) ---

    public int getPreferredBlocksize()
    {
	synchronized (lock) {
	    long bs = bandwidth / 4;
	    return bs == 0 ? 512 : bs < 64 ? 64 : bs > 512 ? 512 : (int)bs;
	}
    }

    public void setBandwidth(long newValue)
    {
	synchronized (lock) {
	    bandwidth = (newValue * TICK_LENGTH) / 1000;
	}
    }

    /**
     * Returns the number of bytes that the calling thread is allowed to 
     * send. Blocks until at least one byte could be allocated.
     *
     * @return -1, if interrupted;
     */
    public int allocate(int bytes)
    {
	synchronized (lock) {
	    while (true) {
		if (System.currentTimeMillis() - tickStart >= TICK_LENGTH) {
		    //logger.warn("* new tick: " + bandwidth + " to allocate *");
		    tickStart = System.currentTimeMillis();
		    allocated = 0;
		    //lock.notifyAll();
		}

		//Debug.log("*** allocated " + allocated + " / " + bandwidth);

		if (bandwidth == 0) {
		    // no limit
		    return bytes;
		}
		else if (bytes < bandwidth - allocated) {
		    // we still have some bandwidth left
		    allocated += bytes;
		    //logger.warn("returning " + bytes);
		    return bytes;
		}
		else if (bandwidth - allocated > 0) {
		    // don't have enough, but return all we have left
		    bytes = (int)(bandwidth - allocated);
		    allocated = bandwidth;
		    //logger.warn("returning " + bytes);
		    return bytes;
		}
		
		// we could not allocate any bandwidth, wait until the next
		// tick is started

		// this is a bit too long
		long t
		    = (TICK_LENGTH - (System.currentTimeMillis() - tickStart));
		if (t > 0) {
		    try {
			//logger.warn("waiting for " + t);
			lock.wait(t);
		    }
		    catch (InterruptedException e) {
			return -1;
		    }
		}
	    }
	}
    }	    

}
