/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.net;

import xnap.util.*;

import java.beans.*;
import java.io.*;
import java.net.*;
import java.util.*;
import org.apache.log4j.Logger;

public abstract class AbstractTransferContainer extends AbstractRunnable
    implements ITransferContainer, Runnable {

    //--- Constant(s) ---

    public static final int RECALC_INTERVAL = 5000;
    /**
     * Wait 50 milli seconds before calculating rate for the first time.
     * Otherwise we will get fantastically high ones.
     */
    public static final int MIN_RECALC_INTERVAL = 50;

    //--- Data field(s) ---

    protected static Logger logger = Logger.getLogger("xnap.net.transfer");

    protected long bytesTransferred = 0;
    protected long totalBytesTransferred = 0;
    protected File file = null;

    private long lastBytesTransferred = 0;
    private long lastElapsedTime = 0;
    private long lastRate = 0;

    protected long totalRate = -1;
    protected long transferStartTime = -1;

    //--- Constructor(s) ---

    protected AbstractTransferContainer()
    {
	status = STATUS_NOT_STARTED;
    }

    //--- Method(s) ---

    public boolean isDone()
    {
	return ((status == STATUS_SUCCESS) 
		|| (status == STATUS_FAILED)
		|| (status == STATUS_ABORTED) 
		|| (status == STATUS_DELETED)
		|| (status == STATUS_ERROR));
    }

    public boolean isResumable() 
    {
	return (!isAborting()
		&& ((status == STATUS_NOT_STARTED)
		    || (status == STATUS_FAILED) 
		    || (status == STATUS_ABORTED)
		    || (status == STATUS_LOCALLY_QUEUED)));
    }

    public boolean isRunning()
    {
	return ((status == STATUS_CONNECTING) 
		|| (status == STATUS_LOCALLY_QUEUED)
		|| (status == STATUS_DOWNLOADING)
		|| (status == STATUS_UPLOADING)
		|| (status == STATUS_SEARCHING)
		|| (status == STATUS_WAITING));
    }

    public long getAverageRate()
    {
	if (totalRate != -1) {
	    return totalRate;
	}
	else if (getStatus() != STATUS_DOWNLOADING 
		 && getStatus() != STATUS_UPLOADING) {
	    return -1;
	}
	long elapsedTime = getElapsedTime();
	if (elapsedTime > 0) {
	    return (getBytesTransferred() * 1000) / elapsedTime;
	}
	else {
	    return getBytesTransferred();
	}
    }

    public long getBytesTransferred() 
    {
	return bytesTransferred;
    }

    public long getTotalBytesTransferred()
    {
	return totalBytesTransferred;
    }

    /**
     * Returns the current download rate.
     *
     * @return byte / s
     */
    public long getCurrentRate() 
    {
	if (totalRate != -1) {
	    return totalRate;
	}
	else if (status != STATUS_DOWNLOADING && status != STATUS_UPLOADING) {
	    return -1;
	}
	
	// Try to take at least a RECALC_INTERVAL into account to
	// avoid jumping values all the time.
	
	long deltaTime = System.currentTimeMillis() - lastElapsedTime;
	if (deltaTime > MIN_RECALC_INTERVAL) {
	    long deltaBytes = getBytesTransferred() - lastBytesTransferred;
	    if (getElapsedTime() < RECALC_INTERVAL) {
		lastRate = (deltaBytes * 1000) / deltaTime;
	    }
	    else {
		lastBytesTransferred += deltaBytes;
		lastElapsedTime += deltaTime;

		long t = RECALC_INTERVAL - deltaTime;
		if (t > 0) {
		    // approximate how many bytes were transfered in the 
		    // last RECALC_INTERVAL
		    deltaBytes += (t * lastRate) / 1000;
		    deltaTime = RECALC_INTERVAL;
		}
		lastRate = (deltaBytes * 1000) / deltaTime;
	    }
	}

	return lastRate;
    }

    /**
     * Returns the number of milli seconds that have passed since the 
     * transfer has been started.
     */
    protected long getElapsedTime() 
    {
        return System.currentTimeMillis() - transferStartTime;
    }

    public File getFile()
    {
	return file;
    }

    public void setFile(File newValue)
    {
        file = newValue;
    }

    public String getFilename()
    {
	File f = getFile();
	return (f != null) ? f.getName() : "";
    }

    public long getFilesize()
    {
	File f = getFile();
	return (f != null) ? f.length() : 0;
    }

    public abstract IUser getUser();

    public void setStatus(int newValue, String newResponse)
    {
//  	logger.debug("AbstractTransferContainer: setStatus "
//  		     + status + " -> " + newValue + " " + newResponse);
	
	super.setStatus(newValue, newResponse);

	if (isDone()) {
	    // display overall average at last
	    long t = getElapsedTime();
	    if  (t > 0) {
		totalRate = (getBytesTransferred() * 1000) / t;
	    }
	    else {
		totalRate = getBytesTransferred();
	    }
	    totalRate = (totalRate > 0) ? totalRate : -1;
	}
    }

    public String getStatusMsg(int status) 
    {
	return STATUS_MSGS[status];
    }

    public void abort() 
    {
	if (!isDone()) {
	    logger.debug("abort " + getFile());
	    die(STATUS_ABORTED);
	}
    }

    public void delete()
    {
	// I would switch those two lines
	getFile().delete();
	die(STATUS_DELETED);
    }

    public void fail(String msg)
    {
	die(STATUS_FAILED, msg);
    }

    public void locallyQueued()
    {
	setStatus(STATUS_LOCALLY_QUEUED);
    }

    public abstract void run();

    public void start() 
    {
	if (!canStart()) {
	    // already running
	    logger.error("already running");
	    return;
	}

	die = false;
	setStatus(STATUS_NOT_STARTED);

	runner = new Thread(this, "Transfer " + getFilename());
	runner.start();
    }

    public String toString()
    {
	return getFilename() + ", " + getFilesize();
    }

    protected void startTransfer()
    {
	bytesTransferred = 0;
	lastBytesTransferred = 0;
        transferStartTime = System.currentTimeMillis();
	lastElapsedTime = transferStartTime;
	totalRate = -1;
    }

}
