/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.gift;

import java.text.MessageFormat;

import xnap.XNap;
import xnap.cmdl.AbstractCommand;
import xnap.cmdl.CommandLineSupport;
import xnap.cmdl.Executer;
import xnap.net.ISearch;
import xnap.plugin.AbstractPlugin;
import xnap.plugin.INetworkPlugin;
import xnap.plugin.gift.net.Engine;
import xnap.plugin.gift.net.Search;
import xnap.plugin.gift.net.event.ErrorEvent;
import xnap.plugin.gift.net.event.OfflineEvent;
import xnap.plugin.gift.net.event.OnlineEvent;
import xnap.plugin.gift.net.event.StatsEvent;
import xnap.plugin.gift.net.event.listener.ErrorEventListener;
import xnap.plugin.gift.net.event.listener.NetworkEventListener;
import xnap.plugin.gift.util.GiftPreferences;
import xnap.util.Debug;
import xnap.util.Preferences;
import xnap.util.SearchFilter;
import xnap.util.SearchManager;


/**
 * Plugin
 *
 * @author $author$
 * @version $Revision: 1.19 $
 */
public class Plugin extends AbstractPlugin implements INetworkPlugin,
    NetworkEventListener, ErrorEventListener, CommandLineSupport {
    //~ Static fields/initializers ---------------------------------------------

    private static final AbstractCommand[] commands = {  };
    public static final String VERSION = "2.0dev"; //XNap.VERSION;
    private static Plugin singleton;

    //~ Instance fields --------------------------------------------------------

    private GiftPreferences giftPrefs = GiftPreferences.getInstance();
    private Preferences prefs = Preferences.getInstance();
    private String statusMessage;

    //~ Constructors -----------------------------------------------------------

    /**
     * Creates a new Plugin object.
     */
    public Plugin() {
        singleton = this;
    }

    //~ Methods ----------------------------------------------------------------

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public AbstractCommand[] getCommands() {
        return commands;
    }

    /**
     * Singleton getInstance
     *
     * @return Plugin
     */
    public static Plugin getInstance() {
        return singleton;
    }

    /**
     * @see xnap.plugin.IPlugin#getDescription()
     */
    public String getDescription() {
        return XNap.tr("Support for the OpenFT network through giFT.");
    }

    /**
     * @see xnap.plugin.IPlugin#getName()
     */
    public String getName() {
        return "giFT";
    }

    /**
     * @see xnap.plugin.INetworkPlugin#getStatus()
     */
    public int getStatus() {
        if (Engine.getInstance().isConnected()) {
            return STATUS_GREEN;
        } else {
            return STATUS_RED;
        }
    }

    /**
     * @see xnap.plugin.INetworkPlugin#getStatusMessage()
     */
    public String getStatusMessage() {
        return statusMessage;
    }

    /**
     * @see xnap.plugin.IPlugin#getVersion()
     */
    public String getVersion() {
        return VERSION;
    }

    /**
     * @see xnap.plugin.gift.net.event.listener.NetworkEventListener#attached(xnap.plugin.gift.net.event.OnlineEvent)
     */
    public void attached(OnlineEvent evt) {
        SearchManager.getInstance().readyToSearch(true);
        statusMessage = MessageFormat.format(XNap.tr("connected to {0}"),
                new Object[] { Engine.getInstance().getServerInfo() });
    }

    /**
     * @see xnap.plugin.gift.net.event.listener.NetworkEventListener#detached(xnap.plugin.gift.net.event.OfflineEvent)
     */
    public void detached(OfflineEvent evt) {
        SearchManager.getInstance().readyToSearch(false);
        statusMessage = MessageFormat.format(XNap.tr("disconnected from {0}"),
                new Object[] { Engine.getInstance().getServerInfo() });
    }

    /**
     * @see xnap.plugin.gift.net.event.listener.ErrorEventListener#onError(xnap.plugin.gift.net.event.ErrorEvent)
     */
    public void onError(ErrorEvent evt) {
        System.out.println(evt.getErrorMessage());
        evt.getException().printStackTrace();
    }

    /**
     * @see xnap.plugin.INetworkPlugin#search(xnap.util.SearchFilter, int)
     */
    public ISearch[] search(SearchFilter filter, int priority) {
        Debug.log("gift: search");

        Search search = new Search(filter, priority, prefs.getMaxSearchResults());
		Engine.getInstance().addEventListener(search);
        return new ISearch[] { search };
    }

    /**
     * @see xnap.plugin.IPlugin#start()
     */
    public void start() {
        Debug.log("gift: start");
        statusMessage = MessageFormat.format(XNap.tr("connecting to {0}"),
                new Object[] { Engine.getInstance().getServerInfo() });
        Engine.getInstance().setHost(giftPrefs.getGiftHost());
        Engine.getInstance().setPort(giftPrefs.getGiftPort());
        Engine.getInstance().setUserName(Preferences.getInstance().getUsername());
        Engine.getInstance().addEventListener(this);
        Engine.getInstance().start();

        Executer.installHandler(this);
        super.start();
    }

    /**
     * @see xnap.plugin.gift.net.event.listener.NetworkEventListener#statsUpdate(xnap.plugin.gift.net.event.StatsEvent)
     */
    public void statsUpdate(StatsEvent evt) {
        statusMessage = MessageFormat.format(XNap.tr("connected to {0}"),
                new Object[] { Engine.getInstance().getServerInfo() }) + " " +
            Engine.getInstance().getStats();
    }

    /**
     * @see xnap.plugin.IPlugin#stop()
     */
    public void stop() {
        Engine.getInstance().stop();
        Executer.removeHandler(this);
        Debug.log("gift: stop");
        super.stop();
    }
}
