/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.gift.net.lexer;

import java.text.CharacterIterator;
import java.text.StringCharacterIterator;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;


/**
 * Command  Represents a GiFT-Command
 *
 * @author <a href="mailto:tvanlessen@taval.de">Tammo van Lessen</a>
 * @version CVS $Id: Command.java,v 1.3 2003/01/17 02:03:36 taval Exp $
 */
public class Command
{
    //~ Instance fields --------------------------------------------------------

    private Hashtable keys;
    private Hashtable modifiers;
    private Vector subCommands;
    private String cmdArgument;
    private String cmdModifier;
    private String command;

    //~ Constructors -----------------------------------------------------------

    /**
     * Creates a new <code>cmd</code>-Command object.
     *
     * @param cmd the command's name
     */
    public Command(String cmd)
    {
        this();
        command = cmd;
    }

    /**
     * Creates an empty Command object.
     */
    public Command()
    {
        modifiers = new Hashtable();
        keys = new Hashtable();
        subCommands = new Vector();
        command = null;
        cmdModifier = null;
        cmdArgument = null;
    }

    //~ Methods ----------------------------------------------------------------

    /**
     * Method getAllSubCommandsByName.
     */
    public Vector getAllSubCommandsByName(String key)
    {
		Command cmd;
		Vector cmdList = new Vector();
		for (int i = 0; i < subCommands.size(); i++) {
			cmd =(Command)subCommands.get(i); 
			if (cmd.getCommand().equalsIgnoreCase(key)) {
				cmdList.add(cmd);
			}
		}
		return cmdList;
    }

    /**
     * returns the command
     *
     * @return the command's name
     */
    public String getCommand()
    {
        return command;
    }

    /**
     * Sets the commands argument
     *
     * @param arg the argument
     */
    public void setCommandArgument(String arg)
    {
        cmdArgument = arg;
    }

    /**
     * Returns the commands argument
     *
     * @return the argument
     */
    public String getCommandArgument()
    {
        return cmdArgument;
    }

    /**
     * Sets the modifier for the command
     *
     * @param arg the modifier
     */
    public void setCommandModifier(String arg)
    {
        cmdModifier = arg;
    }

    /**
     * Returns the commands modifier
     *
     * @return the modifier
     */
    public String getCommandModifier()
    {
        return cmdModifier;
    }

    /**
     * Returns the argument corresponding to the given key
     *
     * @param key the key
     *
     * @return the key's argument
     */
    public String getKey(String key)
    {
        return (String) keys.get(key);
    }

    /**
     * Sets the argument for the given key
     *
     * @param key the key
     * @param arg the argument
     */
    public void setKeyArgument(String key, String arg)
    {
        keys.put(key, arg);
    }

    /**
     * Sets the modifier for the specified key.
     *
     * @param key the key
     * @param mod the modifier
     */
    public void setKeyModifier(String key, String mod)
    {
        modifiers.put(key, mod);
    }

    /**
     * Returns the modifier for the given key.
     *
     * @param key the key
     *
     * @return the modifier
     */
    public String getKeyModifier(String key)
    {
        return (String) modifiers.get(key);
    }

    /**
     * Returns all keys
     *
     * @return all keys
     */
    public Enumeration getKeys()
    {
        return keys.keys();
    }

    /**
     * Returns the subcommand by its name
     * If more subcommands with the same name exists, the first will be returned
     *
     * @param key the subcommands name
     *
     * @return the subcommand
     */
    public Command getSubCommandByName(String key)
    {
        Command cmd;
        for (int i = 0; i < subCommands.size(); i++) {
        	cmd =(Command)subCommands.get(i); 
        	if (cmd.getCommand().equalsIgnoreCase(key)) {
        		return cmd;
        	}
        }
        return null;
    }

    /**
     * Returns all subcommands (its names)
     *
     * @return the subcommands
     */
    public Vector getSubCommands()
    {
        return subCommands;
    }

    /**
     * Adds a key with its argument
     *
     * @param key the key
     * @param arg the argument
     */
    public void addKey(String key, String arg)
    {
        keys.put(key, arg);
    }

    /**
     * Adds a key without its argument
     *
     * @param key the key
     */
    public void addKey(String key)
    {
        keys.put(key, "empty");
    }

    /**
     * Adds a subcommand
     *
     * @param command the subcommand
     */
    public void addSubCommand(Command command)
    {
        subCommands.add(command);
    }

    /**
     * Returns if command has keys
     *
     * @return hasKeys
     */
    public boolean hasKeys()
    {
        return !((keys.size() == 0) ||
        ((keys.size() == 1) && (keys.get("COMMAND") != null)));
    }

    /**
     * Parses the command from a String (GiFT-UI-protocol)
     *
     * @param cmd the String
     */
    public void parse(String cmd)
    {
        parseRecursive(this, new StringCharacterIterator(cmd));
    }

    /**
     * Returns the command as String (GiFT-UI-protocol)
     *
     * @return the commandstr
     */
    public String print()
    {
        return print(true);
    }

    /**
     * Debugdump
     *
     * @return String
     */
    public String toString()
    {
        String str = "GiFTCommand " + command + "\n";
        Enumeration en = keys.keys();
        String key;

        while (en.hasMoreElements()) {
            key = (String) en.nextElement();
            str += ("  + " + key + " = " + keys.get(key) + "\n");
        }

		for (int i = 0; i < subCommands.size(); i++) {
            str += "--- sub ---\n";
            str += subCommands.get(i);
            str += "--- /sub ---\n";
		}

        return str;
    }

    private String escape(String s)
    {
        StringBuffer buffer = new StringBuffer(s.length());

        for (int i = 0; i < s.length(); i++) {
            char ch = s.charAt(i);

            switch (ch) {
            case '(':
                buffer.append('\\');
                buffer.append(ch);

                break;

            case ')':
                buffer.append('\\');
                buffer.append(ch);

                break;

            case '{':
                buffer.append('\\');
                buffer.append(ch);

                break;

            case '}':
                buffer.append('\\');
                buffer.append(ch);

                break;

            case '[':
                buffer.append('\\');
                buffer.append(ch);

                break;

            case ']':
                buffer.append('\\');
                buffer.append(ch);

                break;

            case ';':
                buffer.append('\\');
                buffer.append(ch);

                break;

            case '\\':
                buffer.append('\\');
                buffer.append(ch);

                break;

            default:
                buffer.append(ch);
            }
        }

        return buffer.toString();
    }

    private boolean parseRecursive(Command cmd, StringCharacterIterator sci)
    {
        String currKey = null;
        char curr;

        while (sci.getIndex() < sci.getEndIndex()) {
            switch (curr = sci.current()) {
            case ' ':
            case '\t':
            case '\r':
            case '\n':
                sci.next();

                continue;

            case '(':

                StringBuffer argBuf = new StringBuffer();

                for (char ch = sci.next();
                        (ch != ')') && (ch != CharacterIterator.DONE);
                        ch = sci.next()) {
                    if (ch == '\\') {
                        sci.next();
                    }

                    argBuf.append(ch);
                }

                if (currKey == null) {
                    cmd.setCommandArgument(argBuf.toString());
                } else {
                    cmd.addKey(currKey, argBuf.toString());
                    currKey = null;
                }

                sci.next(); // skip ')'

                continue;

            case '{':
                sci.next(); // skip '{'

                Command subCommand = new Command(currKey);

                if (this.parseRecursive(subCommand, sci) == false) {
                    return false;
                }

                sci.next(); // skip '}'
                cmd.addSubCommand(subCommand);

                continue;

            case '}':
            case ';':
                return true;

            default:

                StringBuffer buf = new StringBuffer();

                for (char ch = sci.current(); ch != CharacterIterator.DONE;
                        ch = sci.next()) {
                    if (ch == ';') {
                        break;
                    }

                    if (ch == '(') {
                        break;
                    }

                    if (ch == '{') {
                        break;
                    }

                    if (Character.isWhitespace(ch)) {
                        break;
                    }

                    if (ch == '\\') {
                        sci.next();
                    }

                    buf.append(ch);
                }

                if (cmd.command == null) {
                    cmd.command = buf.toString();
                } else {
                    currKey = buf.toString();
                }

                continue;
            }
        }

        return false;
    }

    private String print(boolean root)
    {
        String str = "\n";

        if (root) {
            str += escape(command.toUpperCase());
        } else {
            str += escape(command);
        }

        if (getCommandArgument() != null) {
            str += ("(" + escape(getCommandArgument()) + ")");
        }

        if (!root && keys.isEmpty()) {
            return str;
        }

        if (!root) {
            str += "{\n";
        }

        Enumeration en = keys.keys();

        while (en.hasMoreElements()) {
            String key = (String) en.nextElement();
            String keymod = (getKeyModifier(key) == null) ? ""
                                                          : (" [" +
                getKeyModifier(key) + "] ");
            str += (" " + escape(key) + escape(keymod) + "(" +
            escape((String) keys.get(key)) + ")\n");
        }

		for (int i = 0; i < subCommands.size(); i++) {
            str += ((Command) subCommands.get(i)).print(false);
		}

        str += (root ? ";\n" : "}");

        return str;
    }
}
