/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.gnutella.gui;

import xnap.*;
import xnap.gui.*;
import xnap.gui.event.PopupListener;
import xnap.gui.table.*;
import xnap.io.*;
import xnap.plugin.gnutella.gui.table.*;
import xnap.plugin.gnutella.net.*;
import xnap.plugin.gnutella.util.*;
import xnap.util.*;

import java.awt.*;
import java.awt.event.*;
import java.lang.Math;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.table.*;
import java.beans.*;

public class ConnectionsPanel extends AbstractPanel implements ActionListener
{
  
    // --- Data Field(s) ---

    private JScrollPane jsp;
    private JTable jta;
    private ConnectionsTableModel ctm;

    private JCheckBox jcbAutoConnect;
    private JCheckBox jcbCleanDisconnected;

    private ConnectAction connectAction = new ConnectAction();
    private DisconnectAction disconnectAction = new DisconnectAction();
    private AddServentAction addServentAction = new AddServentAction();

    private GnuPreferences gnuPrefs = GnuPreferences.getInstance();
    private Preferences prefs = Preferences.getInstance();

    // --- Constructor(s) ---

    public ConnectionsPanel()
    {
	initialize();
    }

    // --- Method(s) ---

    private void initialize()
    {
        setLayout(new java.awt.BorderLayout());

	JPopupMenu popup = new JPopupMenu();
	popup.add(connectAction);
	popup.add(disconnectAction);
	popup.addSeparator();
	popup.add(addServentAction);
	popup.pack();
	
        ctm = new ConnectionsTableModel(Connections.getInstance());
        jta = ctm.createJTable();
        jta.setShowGrid(false);
	
	MouseListener popupListener = new PopupListener(popup);
	jta.addMouseListener(popupListener);
	jta.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0), 
  			      connectAction);
	jta.getActionMap().put(connectAction, connectAction);

        // table scroll panel
        jsp = new JScrollPane();
        jsp.setViewportView(jta);


	// cache hosts


        // button panel
	JPanel buttonPanel = new JPanel();
        buttonPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
	
        jcbAutoConnect = new JCheckBox("Use AutoConnector", 
				       gnuPrefs.getUseAutoconnector());
        jcbAutoConnect.addActionListener(this);
        buttonPanel.add(jcbAutoConnect);

	jcbCleanDisconnected = new JCheckBox("Clean Disconnected",
					     gnuPrefs.getCleanDisconnected());
	jcbCleanDisconnected.addActionListener(this);
	buttonPanel.add(jcbCleanDisconnected);


        add(jsp, "Center");
	add(buttonPanel, "South");
    }
    
    public AbstractAction[] getActions()
    {
	return ( new AbstractAction[] {
	    new OpenHostsFileAction(), connectAction, disconnectAction, 
	    addServentAction
	});
    }

    public void actionPerformed(ActionEvent e)
    {
	Connections.getInstance().setEnabled(jcbAutoConnect.isSelected());
	gnuPrefs.setCleanDisconnected(jcbCleanDisconnected.isSelected());
    }

    public void doLoadServentsFile(final String filename) {
        (new Thread("LoadServersFile") 
	    {
		public void run() 
		{
		    Connections.getInstance().addFromFile(filename);
		}
	    }
	 ).start();
    }

    protected Servent[] getSelectedServents()
    {
	int[] rows = jta.getSelectedRows();
	Servent[] servents = new Servent[rows.length];
	for (int i = 0; i < rows.length; i++) {
	    servents[i] = ctm.get(rows[i]);
	}

	return servents;
    }

    /**
     * ConnectAction class handles the command to connect to the selected
     * servent(s).  For each selected server it calls
     * Connections.doConnect(servent).
     */
    private class ConnectAction extends AbstractAction {

        public ConnectAction() 
	{
            putValue( Action.NAME, "Connect" );
            putValue( Action.SHORT_DESCRIPTION,
                      "Connect to the selected servent(s)" );
	    putValue(Action.SMALL_ICON, 
		     XNapFrame.getIcon("connect_established.png"));
            putValue( Action.MNEMONIC_KEY, new Integer('C') );
        }

        public void actionPerformed( ActionEvent event ) 
	{
	    Servent[] servents = getSelectedServents();
            for (int i = 0; i < servents.length; i++) {
		Connections.getInstance().doConnect(servents[i]);
            }
        }

    } // class ConnectAction

    /**
     * DisconnectAction class handles the command to disconnect from the
     * selected server(s).  For each selected server it calls
     * from the Napigator services.  It calls ServerPanel.doAskNapigator().
     */
    private class DisconnectAction extends AbstractAction 
    {
        public DisconnectAction() {
            putValue( Action.NAME, "Disconnect" );
            putValue( Action.SHORT_DESCRIPTION,
                      "Disconnect from the selected servent(s)" );
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("connect_no.png"));
            putValue( Action.MNEMONIC_KEY, new Integer('D') );
        }

        public void actionPerformed( ActionEvent event ) 
	{
	    Servent[] servents = getSelectedServents();
            for (int i = 0; i < servents.length; i++) {
		Connections.getInstance().doDisconnect(servents[i]);
            }
        }

    } // class DisconnectAction

    /**
     * AddServentAction class handles the command to add a user defined
     * server to the list.  It calls ServerPanel.addRow().
     */
    private class AddServentAction extends AbstractAction {

        public AddServentAction() {
            putValue( Action.NAME, "Add Servent..." );
            putValue( Action.SHORT_DESCRIPTION, "Add a server to the list" );
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("filenew.png"));
            putValue( Action.MNEMONIC_KEY, new Integer('A') );
        }

        public void actionPerformed( ActionEvent event ) {
            String url = JOptionPane.showInputDialog(ConnectionsPanel.this,
                                    "Servent (hostname:port)",
                                    "Add Servent",
                                    JOptionPane.QUESTION_MESSAGE);

	    if (url != null)
		Connections.getInstance().addServent(url, false);
        }

    } // class AddServentAction

    /**
     * OpenHostsFileAction class handles the command to open a user-specified
     * file containing a list of hosts.  Once the user selects a file the
     * ServerPanel.doLoadServersFile() method is called.
     */
    private class OpenHostsFileAction extends AbstractAction {

        public OpenHostsFileAction() {
            putValue( Action.NAME, "Open Hosts File..." );
            putValue( Action.SHORT_DESCRIPTION,
                      "Open a file with the servents you wish to use" );
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("fileopen.png"));
	    putValue( Action.MNEMONIC_KEY, new Integer('O') );
        }

        public void actionPerformed( ActionEvent event ) 
	{
            JFileChooser chooser = new JFileChooser();
            chooser.setSelectedFile(new File(FileHelper.getHomeDir() 
					     + "gnutella-hosts"));

            if (chooser.showOpenDialog(ConnectionsPanel.this)
                        == JFileChooser.APPROVE_OPTION) {
                String filename = chooser.getSelectedFile().getAbsolutePath();
                doLoadServentsFile(filename);
            }
        }

    } 
}
