/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.util;

import xnap.net.AbstractChannel;
import xnap.plugin.nap.net.Server;
import xnap.plugin.nap.net.msg.ExceptionListener;
import xnap.plugin.nap.net.msg.MessageHandler;
import xnap.plugin.nap.net.msg.client.ChannelTopicMessage;
import xnap.plugin.nap.net.msg.client.JoinChannelMessage;
import xnap.plugin.nap.net.msg.client.PartChannelMessage;
import xnap.plugin.nap.net.msg.client.PublicMessage;
import xnap.plugin.nap.net.msg.server.MessageListener;
import xnap.plugin.nap.net.msg.server.ServerMessage;

import java.io.IOException;

/**
 * Represents a single channel on a server. The MessageHandler pushes the 
 * messages, so we do not need to listen to anybody.
 * All messages are sent asynchronously to avoid blocking the UI.
 */
public class Channel extends AbstractChannel implements ExceptionListener
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    protected boolean joined = false;
    protected Server server;

    //--- Constructor(s) ---

    public Channel(Server server, String name, String topic, int userCount)
    {
	super(server, name, topic, userCount);

	this.server = server;
    }

    //--- Method(s) ---

    public boolean canChangeTopic()
    {
	return true;
    }

    public void changeTopic(String newValue) throws IOException
    {
	ChannelTopicMessage m = new ChannelTopicMessage(name, newValue);
	m.setExceptionListener(this);
	MessageHandler.send(server, m);
    }

    public void close()
    {
	if (joined) {
	    joined = false;
	    MessageHandler.send(server, new PartChannelMessage(name));
	}

	super.close();
    }

    public void exceptionThrown(Exception e) 
    {
	errorReceived(e.getMessage());
    }

    public boolean isJoined()
    {
	return joined;
    }

    public void join() throws IOException
    {
	JoinChannelMessage m = new JoinChannelMessage(name);
	m.setExceptionListener(this);
  	MessageHandler.send(server, m);
	joined = true;
    }

    public void sendMessage(String message) throws IOException
    {
	// split message if it's too large
	for (int i = 0; i < message.length(); i += 174) {
	    int end = Math.min(i + 174, message.length());
	    PublicMessage m = new PublicMessage(name, 
						message.substring(i, end));
	    m.setExceptionListener(this);
	    MessageHandler.send(server, m);
	}
	
    }

    public boolean equals(Object o)
    {
	if (o instanceof Channel) {
	    Channel c = (Channel)o;
	    return (getServer().equals(c.getServer()) 
		    && getName().equalsIgnoreCase(c.getName()));
	}
	return false;
    }

}
