/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.util;

import java.util.ArrayList;

/**
 * Provides a string tokenizer that respects quotes. If a separator is followed
 * by a quote the next token will extent to the next quote even if there are
 * separators in between.
 */
public class QuotedStringTokenizer 
{

    //--- Data field(s) ---
    
    String tokens[] = new String[0];
    int curToken = 0;
    int count;
    
    //--- Constructor(s) ---

    /**
     * Constructs a <code>QuotedStringTokenizer</code>.
     *
     * @see java.util.StringTokenizer
     */
    public QuotedStringTokenizer(String s, String separators) 
    {
        parse(s, separators);
    }

    /**
     * Constructs a <code>QuotedStringTokenizer</code>.
     *
     * @see java.util.StringTokenizer
     */
    public QuotedStringTokenizer(String s)
    {
	this(s, " ");
    }
    
    //--- Method(s) ---

    /**
     * Return the remaining tokens.
     */
    public String allNextTokens()
    {
	StringBuffer sb = new StringBuffer();
	while (hasMoreTokens()) {
	    sb.append(nextToken());
	    if (hasMoreTokens()) {
		sb.append(" ");
	    }
	}
	return sb.toString();
    }

    public int countTokens() 
    {
        return tokens.length - curToken;
    }

    public boolean hasMoreTokens() 
    {
        return curToken < tokens.length;
    }

    public String nextToken() 
    {
        return tokens[curToken++];
    }

    public String[] getTokens()
    {
	return tokens;
    }

    protected void parse(String s, String separators) 
    {
        int i = 0;
        boolean inQuotes = false;
        boolean gettingToken = true;
        StringBuffer token = new StringBuffer("");
        ArrayList result = new ArrayList();

        while(true) {
	    if(s.length() <= i) {
		if (token.toString().length() > 0) {
		    result.add(token.toString());
		}
		break;
	    }

	    char c = s.charAt(i++);
	    if (c == '"') {
		if (inQuotes) {
		    result.add(token.toString());
		    token.setLength(0);
		}

		inQuotes = !inQuotes;
		gettingToken = inQuotes;
	    } 
	    else if (separators.indexOf(c) != -1) {
		if (inQuotes) {
		    token.append(c);
		}
		else if (gettingToken) {
		    // don't add zero length tokens
		    if (token.length() > 0) {
			result.add(token.toString());
			// initialize new token
			token.setLength(0);
			gettingToken = false;
		    }
		}
	    } 
	    else {
		token.append(c);
		gettingToken = true;
	    }
	} 
	
        tokens = (String[])result.toArray(tokens);
    }
}
