/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.util;

import xnap.net.*;

import java.util.*;

/**
 * Search results are added in 3 layers. 
 *
 * 1: data
 * 2: filterData
 * 3: groupedData
 *
 * FIX: This class is not synchronized.
 */
public class SearchResultCollector implements ISearchResultCollector {

    //--- Constant(s) ---

    //--- Data Field(s) ---

    /**
     * Contains all collected <code>ISearchResult</code> objects.
     */
    protected EventVector data = new EventVector();
    protected SearchFilter filter;
    /**
     * Contains <code>ISearchResult</code> objects that were allowed by 
     * <code>filter</code>.
     */
    protected Vector filteredData = new Vector();
    protected Grouper grouper;
    /**
     * Contains <code>ISearchResultContainer</code> objects that were grouped 
     * by <code>grouper</code>.
     */
    protected EventVector groupedData = new EventVector();

    //--- Constructor(s) ---

    public SearchResultCollector(SearchFilter filter, Grouper grouper)
    {
	this.filter = filter;
	this.grouper = grouper;
	this.grouper.setData(groupedData);
    }

    public SearchResultCollector(Grouper grouper)
    {
	this(null, grouper);
    }

    //--- Method(s) ---

    public synchronized boolean add(ISearchResult result)
    {
	int i = data.indexOf(result);
	if (i != -1 && Preferences.getInstance().getRemoveDuplicateResults()) {
	    // check score
	    if (result.getScore() > ((ISearchResult)data.get(i)).getScore()) {
		// replace
		Debug.log("SearchResultCollector: replacing " + result);
		data.set(i, result);
		filteredData.remove(result);
		filterAdd(result);
		// FIX: we need to somehow assure consistency
		//rebuildGroupedData();
	    }
	    else {
		Debug.log("SearchResultCollector: already contains " + result);
		return false;
	    }
	}
	else {
	    data.add(result);
	    filterAdd(result);
	}

	return true;
    }

    public synchronized void clear()
    {
	data.clear();
	rebuildFilteredData();
    }

    public EventVector getData()
    {
	return data;
    }

    public EventVector getGroupedData()
    {
	return groupedData;
    }

    public SearchFilter getFilter()
    {
	return filter;
    }

    public synchronized void setFilter(SearchFilter newValue)
    {
	filter = newValue;
	rebuildFilteredData();
    }

    public synchronized void setGrouper(Grouper newValue)
    {
	grouper = newValue;
	grouper.setData(groupedData);
	rebuildGroupedData();
    }

    public ISearchResult[] getResults()
    {
	ISearchResult[] array = new ISearchResult[data.size()];
	System.arraycopy(data.toArray(), 0, array, 0, array.length);
	return array;
    }

    public SearchResultContainer[] getGroupedResults()
    {
	SearchResultContainer[] array 
	    = new SearchResultContainer[groupedData.size()];
	System.arraycopy(groupedData.toArray(), 0, array, 0, array.length);
	return array;	
    }

    public void remove(ISearchResult result)
    {
	data.remove(result);	
    }

    protected void filterAdd(ISearchResult sr)
    {
	if (filter == null || filter.matches(sr)) {
	    filteredData.add(sr);
	    grouperAdd(sr);
	}
    }

    protected void grouperAdd(ISearchResult sr)
    {
	grouper.add(sr);
    }

    protected void rebuildFilteredData()
    {
	filteredData.clear();
	grouper.clear();

 	for (Iterator i = data.iterator(); i.hasNext();) {
	    filterAdd((ISearchResult)i.next());
	}
    }

    protected void rebuildGroupedData()
    {
	grouper.clear();
 	for (Iterator i = filteredData.iterator(); i.hasNext();) {
	    grouperAdd((ISearchResult)i.next());
	}
    }
	
}

