/*
 * gnuavi.c
 *
 * Copyright (C) 1998 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include "gnuavi.h"

#ifdef DEBUG
#define printid(s,x) printf("%s=%c%c%c%c\n",s,\
			((char*)&x)[3],((char*)&x)[2],((char*)&x)[1],((char*)&x)[0])
#else
#define printid(s,x)
#endif
/*
 * read 4 byte chars as id into the variable 'n'
 */
int
gv_read_id(gv_u32 *n, FILE *fp)
{
	int rc;
	rc = fread (n, 1, 4, fp);
#ifdef LSB
	gv_swap4byte(*n);
#endif
	return (rc);
}

/*
 * read 4 byte and swap if needed
 */
int
gv_read4byte(gv_u32 *n, FILE *fp)
{
	int rc;
	rc = fread (n, 1, 4, fp);
#ifndef LSB
	gv_swap4byte(*n);
#endif
	return (rc);
}

/*
 * read 2 byte and swap if needed
 */
int
gv_read2byte(gv_u16 *n, FILE *fp)
{
	int rc;
	rc = fread (n, 1, 2, fp);
#ifndef LSB
	gv_swap2byte(*n);
#endif
	return (rc);
}

/*
 */
int
gv_read_snd_strf (gv_avi_snd_strf *strf, FILE *fp)
{
	int in = 0;
	in += gv_read_id (&strf->id, fp);
	in += gv_read4byte (&strf->size, fp);
	in += gv_read2byte (&strf->format, fp);
	in += gv_read2byte (&strf->channels, fp);
	in += gv_read4byte (&strf->rate, fp);
	in += gv_read4byte (&strf->average_bps, fp);
	in += gv_read2byte (&strf->blockalign, fp);
	in += gv_read2byte (&strf->sample_size, fp);
	printid (" strf id", strf->id);
#ifdef DEBUG
	printf (" snd format=%d channels=%d rate=%d sample_size=%d\n",
			strf->format, strf->channels, strf->rate, strf->sample_size);
#endif
	fseek (fp, strf->size - (in-8), SEEK_CUR);
	return (strf->size+8);
}

/*
 */
int
gv_read_vid_strf (gv_avi_vid_strf *strf, FILE *fp)
{
	int in = 0;
	in += gv_read_id (&strf->id, fp);
	in += gv_read4byte (&strf->size, fp);
	in += gv_read4byte (&strf->q_size, fp);
	in += gv_read4byte (&strf->width, fp);
	in += gv_read4byte (&strf->height, fp);
	in += gv_read2byte (&strf->planes, fp);
	in += gv_read2byte (&strf->bit_cnt, fp);
	in += gv_read4byte (&strf->comp, fp);
	in += gv_read4byte (&strf->image_size, fp);
	in += gv_read4byte (&strf->xpels_meter, fp);
	in += gv_read4byte (&strf->ypels_meter, fp);
	in += gv_read4byte (&strf->num_colors, fp);
	in += gv_read4byte (&strf->imp_colors, fp);
	printid (" strf id", strf->id);
#ifdef DEBUG
	printf (" vid q_size=%d planes=%d bit_cnt=%d image_size=%d comp=%d\n",
		strf->q_size, strf->planes, strf->bit_cnt, strf->image_size,strf->comp);
	printf (" --- xpels=%d ypels=%d\n", strf->xpels_meter, strf->ypels_meter);
#endif
	fseek (fp, strf->size - (in-8), SEEK_CUR);
	return (strf->size+8);
}

/*
 */
int
gv_read_strh (gv_avi_strh *strh, FILE *fp)
{
	int in = 0;
	in += gv_read_id (&strh->id, fp);
	in += gv_read4byte (&strh->size, fp);
	in += gv_read_id (&strh->type, fp);
	in += gv_read4byte (&strh->fcc_handler, fp);
	in += gv_read4byte (&strh->priority, fp);
	in += gv_read4byte (&strh->init_fr, fp);
	in += gv_read4byte (&strh->scale, fp);
	in += gv_read4byte (&strh->rate, fp);
	in += gv_read4byte (&strh->start, fp);
	in += gv_read4byte (&strh->length, fp);
	in += gv_read4byte (&strh->buf_size, fp);
	in += gv_read4byte (&strh->quality, fp);
	in += gv_read4byte (&strh->sample_size, fp);
	printid (" strh id", strh->id);
	printid (" strh type", strh->type);
#ifdef DEBUG
	printf ("  scale=%d buf_size=%d smpl size=%d\n",
				strh->scale, strh->buf_size, strh->sample_size);
#endif
	fseek (fp, strh->size - (in-8), SEEK_CUR);
	return (strh->size+8);
}

/*
 */
int
gv_read_avi_header (gv_avi_hdrl *hdrl, FILE *fp, int size)
{
	int in=0, i;
	gv_u32 streams = 0;
	gv_avi_vid_strf *vid_strf;
	gv_avi_snd_strf *snd_strf;
	gv_u32 list_id, list_size, list_type;

	in += gv_read_id (&hdrl->avih.id, fp);
	in += gv_read4byte (&hdrl->avih.size, fp);
	printid ("avih",hdrl->avih.id);
#ifdef DEBUG
	printf ("%d: size=%d (%d)\n",__LINE__,hdrl->avih.size, sizeof(hdrl->avih));
#endif
	in += gv_read4byte (&hdrl->avih.us_pf, fp);
	in += gv_read4byte (&hdrl->avih.bps, fp);
	in += gv_read4byte (&hdrl->avih.pad_gran, fp);
	in += gv_read4byte (&hdrl->avih.flags, fp);
	in += gv_read4byte (&hdrl->avih.frames, fp);
	in += gv_read4byte (&hdrl->avih.init_fr, fp);
	in += gv_read4byte (&hdrl->avih.streams, fp);
	in += gv_read4byte (&hdrl->avih.buf_size, fp);
	in += gv_read4byte (&hdrl->avih.width, fp);
	in += gv_read4byte (&hdrl->avih.height, fp);
	in += gv_read4byte (&hdrl->avih.scale, fp);
	in += gv_read4byte (&hdrl->avih.rate, fp);
	in += gv_read4byte (&hdrl->avih.start, fp);
	in += gv_read4byte (&hdrl->avih.length, fp);

	if (hdrl->avih.streams > 0) {
		streams = hdrl->avih.streams;
		hdrl->str = malloc (sizeof (gv_stream_hdr) * streams);
		if (!hdrl->str)
			return (0);
	}
	for (i = 0; i < streams; i++) {
		in += gv_read_id (&list_id, fp);
		in += gv_read4byte (&list_size, fp);
		in += gv_read_id (&list_type, fp);
		if (list_type != GV_STRL) {
			fseek (fp, list_size-4, SEEK_CUR);
#ifdef DEBUG
			printf ("%s: skipping ..\n", __FILE__);
#endif
			continue;
		}
		in += gv_read_strh (&hdrl->str[i].strh, fp);
		switch (hdrl->str[i].strh.type) {
			case GV_VIDS:
				vid_strf = malloc (sizeof (gv_avi_vid_strf));
				hdrl->str[i].strf = vid_strf;
				in += gv_read_vid_strf (vid_strf, fp);
				break;
			case GV_AUDS:
				snd_strf = malloc (sizeof (gv_avi_snd_strf));
				hdrl->str[i].strf = snd_strf;
				in += gv_read_snd_strf (snd_strf, fp);
				break;
			default:
				printf ("oops: file %s, line %d\n", __FILE__, __LINE__);
				break;
		}
	}
	fseek (fp, size-(in+4), SEEK_CUR);
#ifdef DEBUG
	printf ("%d: size=%d in=%d pos=%ld\n", __LINE__, size, in, ftell(fp));
#endif
	return (1);
}

/*
 * read the avi header
 */
gv_avi *
gv_read_header (FILE *fp, FILE *fp_snd)
{
	gv_avi *avi;
	gv_u32 list_id, list_size, list_type;

	avi = malloc (sizeof (gv_avi));
	if (!avi)
		return (NULL);

	avi->fp = fp;
	avi->fp_snd = fp_snd;
	avi->d_pos = 0;

	gv_read_id (&avi->id, fp);
	if (avi->id != GV_RIFF)
		goto hdr_err;
	gv_read4byte (&avi->size, fp);
	gv_read_id (&avi->type, fp);
	if (avi->type != GV_AVI)
		goto hdr_err;

again:
	gv_read_id (&list_id, fp);
	gv_read4byte (&list_size, fp);
	gv_read_id (&list_type, fp);
	printid ("list id", list_id);
	printid ("list type", list_type);

	if (list_id != GV_LIST)
		goto hdr_err;
	switch (list_type) {
		case GV_hdrl:
			gv_read_avi_header (&avi->hdrl, fp, list_size);
			goto again;
			break;
		case GV_movi:
			avi->d_pos = ftell (fp);
			fseek (fp_snd, avi->d_pos, SEEK_SET);
			printf (" d_pos=%ld\n", avi->d_pos);
			break;
		default:
			if (feof(fp))
				goto hdr_err;
			fseek (fp, list_size, SEEK_SET);
			break;
	}
	return (avi); 
hdr_err:
	free (avi);
	return (NULL);
}


/*
 * convert from rgb555 to rgb565
 */
int
gv_get_frame_15to16 (gv_avi *avi, gv_byte *buf, int pad)
{
	gv_u32 atom_id, atom_size;
	register int in = 0, y, x;
	register int width = avi->hdrl.avih.width;
	register int height = avi->hdrl.avih.height;
	gv_byte *ptr;
	gv_stream_hdr *str = gv_video_stream (avi, 0);
	int bytes_per_pixel = ((gv_avi_vid_strf *)str->strf)->bit_cnt;
	int out_line_size = width * 2;
	int line_size = width * 2;
	int image_size = width * height * bytes_per_pixel /8;
	gv_byte *line;
	gv_u16 *u16, *rgb;

m15to16:
	in  = gv_read_id (&atom_id, avi->fp);
	in += gv_read4byte (&atom_size, avi->fp);
	if (in != 8) {/* eof? */
		return (0);
	}
	switch (atom_id) {
		case GV_FRAM:
			line = malloc (line_size);
			for (y = 0; y < height; y++) {
				ptr = buf+(height - y - 1)*(out_line_size + pad);
				if (fread (line, 1, line_size, avi->fp) < line_size) {
					free (line);
					return (0);
				}
				rgb = (gv_u16 *)line;
				u16 = (gv_u16 *)ptr;
				for (x = 0; x < width; x++) {
					*u16++ = ((*rgb & 0xFFE0)<<1) | (*rgb & 0x001F);
					rgb ++;
				}
			}
			free (line);
			if (atom_size - image_size > 0)
				fseek (avi->fp, atom_size - image_size, SEEK_CUR);
			return (1);
			break;
		case GV_SMPL:
			/* skip sound sample */
			fseek (avi->fp, atom_size, SEEK_CUR);
			goto m15to16;
			break;
		default:
			printf ("oops: id=%d\n", atom_id);
			break;
	} 
	return (0);
}

/*
 * convert from rgb555 to rgb888
 */
int
gv_get_frame_15to24 (gv_avi *avi, gv_byte *buf, int pad)
{
	gv_u32 atom_id, atom_size;
	register int in = 0, y, x;
	register int width = avi->hdrl.avih.width;
	register int height = avi->hdrl.avih.height;
	int out_line_size = width * 3;
	int line_size = width * 2;
	int image_size = line_size * height;
	gv_byte *line, *ptr;
	gv_u16 *rgb;

m15to24:
	in  = gv_read_id (&atom_id, avi->fp);
	in += gv_read4byte (&atom_size, avi->fp);
	if (in != 8) /* eof? */
		return (0);

	switch (atom_id) {
		case GV_FRAM:
			line = malloc (line_size);
			for (y = 0; y < height; y++) {
				ptr = buf+(height - y - 1)*(out_line_size + pad);
				if (fread (line, 1, line_size, avi->fp) < line_size) {
					free (line);
					return (0);
				}
				rgb = (gv_u16 *)line;
				for (x = 0; x < width; x++) {
					*ptr++ = ((*rgb) & 0x001F) << 3;
					*ptr++ = ((*rgb) & 0x03E0) >> 2;
					*ptr++ = ((*rgb) & 0x7C00) >> 7;
					rgb++;
				}
			}
			free (line);
			if (atom_size - image_size > 0)
				fseek (avi->fp, atom_size - image_size, SEEK_CUR);
			return (1);
			break;
		case GV_SMPL:
			/* skip sound sample */
			fseek (avi->fp, atom_size, SEEK_CUR);
			goto m15to24;
			break;
		default:
			printf ("oops: id=%d\n", atom_id);
			break;
	} 
	return (0);
}

/*
 * convert to rgb555 or rgb565
 */
int
gv_get_frame_24toX (gv_avi *avi, gv_byte *buf, int pad, int bits)
{
	gv_u32 atom_id, atom_size;
	register int in = 0, y, x;
	register int width = avi->hdrl.avih.width;
	register int height = avi->hdrl.avih.height;
	gv_byte *rgb, *ptr;
	gv_stream_hdr *str = gv_video_stream (avi, 0);
	int bytes_per_pixel = ((gv_avi_vid_strf *)str->strf)->bit_cnt;
	int out_line_size = width * 2;
	int line_size = width * 3;
	int image_size = width * height * bytes_per_pixel /8;
	register int gshift = 0, bshift = 0;
	gv_byte *line;
	gv_u16 *u16;

m24toX:
	in  = gv_read_id (&atom_id, avi->fp);
	in += gv_read4byte (&atom_size, avi->fp);
	if (in != 8) /* eof? */
		return (0);

	switch (atom_id) {
		case GV_FRAM:
			line = malloc (line_size);
			if (bits == 15) {
				gshift = 3;
				bshift = 10;
			} else {
				gshift = 2;
				bshift = 11;
			}
			for (y = 0; y < height; y++) {
				ptr = buf+(height - y - 1)*(out_line_size + pad);
				if (fread (line, 1, line_size, avi->fp) < line_size) {
					free (line);
					return (0);
				}
				rgb = line;
				u16 = (gv_u16 *)ptr;
				for (x = 0; x < width; x++) {
					*u16++ = (rgb[2] >> 3) | ((rgb[1]>>gshift)<<5) |
								((rgb[0]>>3)<<bshift);
					rgb += 3;
				}
			}
			free (line);
			if (atom_size - image_size > 0)
				fseek (avi->fp, atom_size - image_size, SEEK_CUR);
			return (1);
			break;
		case GV_SMPL:
			/* skip sound sample */
			fseek (avi->fp, atom_size, SEEK_CUR);
			goto m24toX;
			break;
		default:
			printf ("oops: id=%d\n", atom_id);
			break;
	} 
	return (0);
}

/*
 * returns image in native format
 */
int
gv_get_frame_raw (gv_avi *avi, gv_byte *buf, int pad)
{
	gv_u32 atom_id, atom_size;
	register int in = 0, y, i;
	register int width = avi->hdrl.avih.width;
	register int height = avi->hdrl.avih.height;
	gv_stream_hdr *str = gv_video_stream (avi, 0);
	int bytes_per_pixel = ((gv_avi_vid_strf *)str->strf)->bit_cnt / 8;
	int line_size = width * bytes_per_pixel;
	int image_size = width * height * bytes_per_pixel;
	gv_byte *ptr, t;

mraw:
	in  = gv_read_id (&atom_id, avi->fp);
	in += gv_read4byte (&atom_size, avi->fp);
	if (in != 8) /* eof? */
		return (0);

	switch (atom_id) {
		case GV_FRAM:
#ifdef DEBUG
			printf ("%s: size=%d\n", __FILE__, atom_size);
#endif
			for (y = 0; y < height; y++) {
				ptr = buf+(height - y - 1)*(line_size + pad);
				if (fread (ptr, 1, line_size, avi->fp) < line_size)
					return (0);
				if (bytes_per_pixel == 3) {
					for (i = 0; i < line_size; i+=3) {
						t = ptr[i];
						ptr[i] = ptr[i+2];
						ptr[i+2] = t;
					}
				}
			}
			if (atom_size - image_size > 0)
				fseek (avi->fp, atom_size - image_size, SEEK_CUR);
			return (1);
			break;
		case GV_SMPL:
			/* skip sound sample */
			fseek (avi->fp, atom_size, SEEK_CUR);
			goto mraw;
			break;
		default:
			printf ("oops: id=%d\n", atom_id);
			break;
	} 
	return (0);
}

/*
 */
gv_stream_hdr *
gv_video_stream (gv_avi *avi, int num) {
	int i, vidnum = 0;

	for (i = 0; i < avi->hdrl.avih.streams; i++) {
		if (avi->hdrl.str[i].strh.type == GV_VIDS) {
			if (vidnum == num)
				return (&avi->hdrl.str[i]);
			vidnum++;
		}
	}
	return (NULL);
}

/*
 */
gv_stream_hdr *
gv_sound_stream (gv_avi *avi, int num) {
	int i, sndnum = 0;

	for (i = 0; i < avi->hdrl.avih.streams; i++) {
		if (avi->hdrl.str[i].strh.type == GV_AUDS) {
			if (sndnum == num)
				return (&avi->hdrl.str[i]);
			sndnum++;
		}
	}
	return (NULL);
}

/*
 */
void
gv_reset (gv_avi *avi)
{
#ifdef DEBUG
	printf ("%s: gv_reset() d_pos=%ld\n", __FILE__, avi->d_pos);
#endif
	if (avi->d_pos) {
		fseek (avi->fp, avi->d_pos, SEEK_SET);
		fseek (avi->fp_snd, avi->d_pos, SEEK_SET);
	}
}

/*
 */
void
gv_skip_frame (gv_avi *avi)
{
	int in = 0;
	gv_u32 atom_id, atom_size;

#ifdef DEBUG
	printf ("%s: gv_skip_frame()\n", __FILE__);
#endif
	while (1) {
		in  = gv_read_id (&atom_id, avi->fp);
		in += gv_read4byte (&atom_size, avi->fp);
		if (in != 8) /* eof? */
			return;

		switch (atom_id) {
			case GV_FRAM:
				fseek (avi->fp, atom_size, SEEK_CUR);
				return;
				break;
			default:
				fseek (avi->fp, atom_size, SEEK_CUR);
				break;
		}
	}
}

