/*
 * video.c
 *
 * Copyright (C) 1998 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include "video.h"
#include "gnuavi.h"
#include "gt.h"

#define read_msb4(p,f)	gt_read4byte(p,f)
#define read_lsb4(p,f)	gv_read4byte(p,f)

/*
 * read some informations to fill up the 'video'structure
 */
int
vid_read_header (video *vid)
{
	gv_avi *avi;
	gv_stream_hdr *str;
	switch (vid->type) {
		case VT_AVI:
			avi = gv_read_header (vid->fp, vid->fp_snd);
			vid->hdr = avi;
			if (!avi) {
				perror ("vid_read_header()");
				return (0);
			}
			vid->width  = avi->hdrl.avih.width;
			vid->height = avi->hdrl.avih.height;
			vid->frames = avi->hdrl.avih.frames;
			vid->tpf = avi->hdrl.avih.us_pf / 1000;
			vid->fps = 1000 / vid->tpf;
			vid->vid_no = 0;
			str = gv_video_stream (avi, 0);
			if (!str) {
				printf ("no video stream?\n");
				break;
			}
			switch (((gv_avi_vid_strf *) str->strf)->bit_cnt) {
				case 16:
					vid->vid_bpp = 16;
					vid->vid_depth = 15;
					vid->vid_codec = VV_COD_RGB555;
					break;
				case 24:
					vid->vid_bpp = 24;
					vid->vid_depth = 24;
					vid->vid_codec = VV_COD_RGB888;
					break;
				default:
					printf ("oops?\n");
					break;
			}
			str = gv_sound_stream (avi, 0);
			if (!str)
				vid->snd_codec = 0;
			else {
				vid->snd_codec =((gv_avi_snd_strf *)str->strf)->format;
				vid->snd_rate = ((gv_avi_snd_strf *)str->strf)->rate;
				vid->snd_size = ((gv_avi_snd_strf *)str->strf)->sample_size;
				vid->snd_chnl = ((gv_avi_snd_strf *)str->strf)->channels;
			}
			vid->snd_buf_size = 0;
			break;
		case VT_QT:
			break;
		default:
			return (0);
			break;
	}
	return (1);
}

/*
 */
video *
vid_open (char *file)
{
	video *vid = NULL;
	vid = malloc (sizeof(video));
	if (!vid) {
		perror ("malloc()");
		return (NULL);
	}
	vid->fp = fopen (file, "rb");
	if (!vid->fp) {
		perror (file);
		goto err;
	}
	vid->fp_snd = fopen (file, "rb");
	if (!vid->fp_snd) {
		perror (file);
		goto err;
	}

	if (!gt_read4byte (&vid->type, vid->fp))
		goto err;
	if (vid->type == GV_RIFF) {
		if (!read_lsb4 (&vid->size, vid->fp))
			goto err;
		if (!read_msb4 (&vid->type, vid->fp))
			goto err;
		if (vid->type != GV_AVI)
			goto err;
		vid->type = VT_AVI;
	} else {
		/* quicktime? */
		vid->size = vid->type;
		if (!read_msb4 (&vid->type, vid->fp))
			goto err;
		switch (vid->type) {
			case GTA_movie:
			case GTA_movie_data:
			case GTA_skip:
				vid->type = VT_QT;
				break;
			default:
				goto err;
				break;
		}
	}
	fseek (vid->fp, 0, SEEK_SET);
	switch (vid->type) {
		case VT_QT:
		case VT_AVI:
			vid_read_header (vid);
			break;
		default:
			goto err;
			break;
	}
	return (vid);
err:
	fclose (vid->fp);
	fclose (vid->fp_snd);
	free (vid);
	return (NULL);
}

/*
 */
void
vid_close (video *vid)
{
	fclose (vid->fp);
	fclose (vid->fp_snd);
}

/*
 * returns the next frame in native format
 */
int
vid_get_frame (video *vid, gv_byte *buf)
{
	gv_avi *avi = (gv_avi *)vid->hdr;
	if (!buf)
		return (0);

#ifdef DEBUG2
	printf ("%s: vid_get_frame()\n", __FILE__);
#endif
	switch (vid->type) {
		case VT_AVI:
			vid->vid_no++;
			if (vid->vid_codec == vid->out_codec)
				return gv_get_frame_raw (avi, buf, vid->out_pad);
			switch (vid->out_codec) {
				case VV_COD_RGB555:
					if (vid->vid_codec == VV_COD_RGB888)
						return gv_get_frame_24to15 (avi,buf, vid->out_pad);
					printf ("codec not supported!\n");
					return (0);
					break;
				case VV_COD_RGB565:
					if (vid->vid_codec == VV_COD_RGB888)
						return gv_get_frame_24to16 (avi,buf, vid->out_pad);
					else if (vid->vid_codec == VV_COD_RGB555)
						return gv_get_frame_15to16 (avi,buf, vid->out_pad);
					printf ("codec not supported!\n");
					return (0);
					break;
				case VV_COD_RGB888:
					if (vid->vid_codec == VV_COD_RGB555)
						return gv_get_frame_15to24 (avi,buf, vid->out_pad);
					printf ("%s: codec not supported!\n", __FILE__);
					return (0);
					break;
				default:
					printf ("%s: codec not supported!\n", __FILE__);
					return (0);
					break;
			}
			break;
		default:
			return(0);
			break;
	}
	return (1);
}

/*
 * set parameters for the returned image format
 */
int
vid_set_video_parms (video *vid, int codec, int pad)
{
	if (!vid)
		return (0);
	vid->out_pad = pad;
	vid->out_codec = codec;
	return (1);
}

/*
 */
void
vid_reset (video *vid)
{
	switch (vid->type) {
		case VT_AVI:
			gv_reset((gv_avi *)vid->hdr);
			break;
		default:
			break;
	}
}

/*
 */
void
vid_skip_frame (video *vid)
{
	switch (vid->type) {
		case VT_AVI:
			gv_skip_frame((gv_avi *)vid->hdr);
			break;
		default:
			break;
	}
}

/*
 */
void
vid_skip_obj (video *vid)
{
	gv_u32 id, size;
	switch (vid->type) {
		case VT_AVI:
			gv_read_id (&id, vid->fp);
			gv_read4byte (&size, vid->fp);
			fseek (vid->fp, size, SEEK_CUR);
			break;
		default:
			break;
	}
}


/*
 * what comes next, audio or video frame?
 */
int
vid_next_in (video *vid)
{
	int vin = 0, in;
	gv_u32 id;

	switch (vid->type) {
		case VT_AVI:
			in = gv_read_id (&id, vid->fp);
			if (in < 4)
				return (vin);
			fseek (vid->fp, -4, SEEK_CUR);
			if (id == GV_FRAM)
				vin = VNI_VIDEO;
			else if (id == GV_SMPL)
				vin = VNI_SOUND;
			break;
		default:
			break;
	}
	return (vin);
}

/*
 */
int
vid_get_sound (video *vid, unsigned char **buf, int *size)
{
	gv_u32 atom_id, atom_size;
	int in = 0;

	switch (vid->type) {
		case VT_AVI:
			in = gv_read_id (&atom_id, vid->fp);
			in+= gv_read4byte (&atom_size, vid->fp);
			if (in < 8)
				return (0);
			if (vid->snd_buf_size < 1) {
				vid->snd_buf = malloc (atom_size);
				vid->snd_buf_size = atom_size;
			}
			if (atom_size > vid->snd_buf_size) {
				vid->snd_buf = realloc (vid->snd_buf, atom_size);
				vid->snd_buf_size = atom_size;
			}
			*size = atom_size;
			*buf = vid->snd_buf;
			if (fread (vid->snd_buf, 1, atom_size, vid->fp) < atom_size)
				return (0);
			return (1);
			break;
		default:
			break;
	}
	return (0);
}

