/****************************************************************************
 *                                                                          *
 * Copyright 1999-2005 ATI Technologies Inc., Markham, Ontario, CANADA.     *
 * All Rights Reserved.                                                     *
 *                                                                          *
 * Your use and or redistribution of this software in source and \ or       *
 * binary form, with or without modification, is subject to: (i) your       *
 * ongoing acceptance of and compliance with the terms and conditions of    *
 * the ATI Technologies Inc. software End User License Agreement; and (ii)  *
 * your inclusion of this notice in any version of this software that you   *
 * use or redistribute.  A copy of the ATI Technologies Inc. software End   *
 * User License Agreement is included with this software and is also        *
 * available by contacting ATI Technologies Inc. at http://www.ati.com      *
 *                                                                          *
 ****************************************************************************/

#ifndef _FIREGL_PUBLIC_H_
#define _FIREGL_PUBLIC_H_

#include <stdarg.h>
#include "kcl_pci.h"
#include "kcl_io.h"

#define FIREGL_USWC_SUPPORT     1

#define FGL_DEVICE_SIGNATURE    0x10020000

#ifndef KERNEL_VERSION
#define KERNEL_VERSION(a,b,c) (((a) << 16) + ((b) << 8) + (c))
#endif

#if defined(__i386__) || defined(__x86_64__)
#define LITTLEENDIAN_CPU 1
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,9)

#define REMAP_PAGE_RANGE_FN remap_pfn_range
#define REMAP_PAGE_RANGE_STR "remap_pfn_range"
#define REMAP_PAGE_RANGE_OFF(offset) ((offset) >> PAGE_SHIFT)

#else /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,9) */

#define REMAP_PAGE_RANGE_FN remap_page_range
#define REMAP_PAGE_RANGE_STR "remap_page_range"
#define REMAP_PAGE_RANGE_OFF(offset) (offset)

#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,9) */

#define REMAP_PAGE_RANGE(vma,offset) \
    REMAP_PAGE_RANGE_FN(FGL_VMA_API_PASS \
                        (vma)->vm_start,	\
                        REMAP_PAGE_RANGE_OFF(offset), \
                        (vma)->vm_end - (vma)->vm_start, \
                        (vma)->vm_page_prot)


/* Page table macros */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,26)
#define PAGING_FAULT_SIGBUS_INT (unsigned long)NOPAGE_SIGBUS
#else /* LINUX_VERSION_CODE < KERNEL_VERSION(2,6,26) */
#define PAGING_FAULT_SIGBUS_INT VM_FAULT_SIGBUS
#endif

#define PGD_OFFSET(mm, pgd_p, pte_linear)	\
do { \
    pgd_p = pgd_offset(mm, pte_linear); \
} while(0)

#define PGD_PRESENT(pgd_p) \
do { \
    if (!pgd_present(*(pgd_p)))	\
    { \
        __KE_ERROR("FATAL ERROR: User queue buffer not present! (pgd)\n"); \
        return PAGING_FAULT_SIGBUS_INT;   /* Something bad happened; generate SIGBUS */ \
        /* alternatively we could generate a NOPAGE_OOM "out of memory" */ \
    } \
} while(0)

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11)

#define PMD_OFFSET(pmd_p, pgd_p, pte_linear) \
do { \
    pud_t *pud_p = pud_offset(pgd_p, pte_linear); \
    if (!pud_present(*(pud_p)))	\
    { \
        __KE_ERROR("FATAL ERROR: User queue buffer not present! (pud)\n"); \
        return PAGING_FAULT_SIGBUS_INT;   /* Something bad happened; generate SIGBUS */ \
        /* alternatively we could generate a NOPAGE_OOM "out of memory" */ \
    } \
    pmd_p = pmd_offset(pud_p, pte_linear); \
} while(0)

#else /* LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11) */

#define PMD_OFFSET(pmd_p, pgd_p, pte_linear)	\
do { \
    pmd_p = pmd_offset(pgd_p, pte_linear); \
} while(0)

#endif

#define PMD_PRESENT(pmd_p) \
do { \
    if (!pmd_present(*(pmd_p)))	\
    { \
        __KE_ERROR("FATAL ERROR: User queue buffer not present! (pmd)\n"); \
        return PAGING_FAULT_SIGBUS_INT;   /* Something bad happened; generate SIGBUS */ \
        /* alternatively we could generate a NOPAGE_OOM "out of memory" */ \
    } \
} while(0)

#ifdef pte_offset_atomic
#define PTE_OFFSET(pte, pmd_p, pte_linear) \
do { \
    pte_t* pte_p; \
    pte_p = pte_offset_atomic(pmd_p, pte_linear); \
    pte = *pte_p; \
    pte_kunmap(pte_p); \
} while(0)
#else
#ifdef pte_offset_map
#define PTE_OFFSET(pte, pmd_p, pte_linear) \
do { \
    pte_t* pte_p; \
    pte_p = pte_offset_map(pmd_p, pte_linear); \
    pte = *pte_p; \
    pte_unmap(pte_p); \
} while(0)
#else
#ifdef pte_offset_kernel
#define PTE_OFFSET(pte, pmd_p, pte_linear) \
do { \
    pte_t* pte_p; \
    pte_p = pte_offset_kernel(pmd_p, pte_linear); \
    pte = *pte_p; \
} while(0)
#else
#define PTE_OFFSET(pte, pmd_p, pte_linear) \
do { \
    pte_t* pte_p; \
    pte_p = pte_offset(pmd_p, pte_linear); \
    pte = *pte_p; \
} while(0)
#endif
#endif
#endif

#define PTE_PRESENT(pte) \
do { \
    if (!pte_present(pte)) \
    { \
        __KE_ERROR("FATAL ERROR: User queue buffer not present! (pte)\n"); \
        return PAGING_FAULT_SIGBUS_INT;   /* Something bad happened; generate SIGBUS */ \
        /* alternatively we could generate a NOPAGE_OOM "out of memory" */ \
    } \
} while(0)

#ifdef pfn_to_page
#define PMD_PAGE(pmd) pmd_page(pmd)
#else /* for old 2.4 kernels */
#define PMD_PAGE(pmd) (pfn_to_page(pmd_val(pmd) >> PAGE_SHIFT))
#endif

#if !defined(CONFIG_SMP) || defined(CONFIG_SUSPEND_SMP) || defined(CONFIG_PM_SLEEP_SMP) // ACPI not working on older SMP kernel (prior to 2.6.13) 
#define FIREGL_POWER_MANAGEMENT
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,14)
#define PMSG_EVENT(pmsg_state) (pmsg_state).event
#else
#define PMSG_EVENT(pmsg_state) (pmsg_state)
#define PM_EVENT_SUSPEND 3
#endif

#if defined(CONFIG_COMPAT) && LINUX_VERSION_CODE > KERNEL_VERSION(2,6,9)
#define FIREGL_IOCTL_COMPAT
#endif

/*****************************************************************************/

struct vm_area_struct;
struct semaphore;
struct drm_device;
struct firegl_pcie_mem;

#if !defined(ATI_API_CALL)
#define ATI_API_CALL __attribute__((regparm(0)))
#endif

/*****************************************************************************/

typedef unsigned long __ke_dev_t;
typedef unsigned long __ke_off_t;
#ifdef __x86_64__
typedef long __ke_ssize_t;
typedef unsigned long __ke_size_t;
#else
typedef int __ke_ssize_t;
typedef unsigned int __ke_size_t;
#endif
typedef unsigned char __ke_u8;
typedef unsigned short __ke_u16;
typedef unsigned int __ke_u32;
typedef unsigned long long __ke_u64;
typedef unsigned long long __ke_dma_addr_t;
typedef long long __ke_loff_t;

// note: assigning uniqe types to originally non interchangeable types
typedef struct { int uniqe1; } __ke_wait_queue_head_t;
typedef struct { int uniqe2; } __ke_wait_queue_t;
typedef struct { int uniqe3; } __ke_sigset_t;
typedef struct { int uniqe7; } __ke_file_operations_t;

typedef	int (*__ke_read_proc_t)(
    char* page, char** start, __ke_off_t off, int count, int* eof, void* data);

typedef struct {
    const char*             name;
    __ke_read_proc_t        f;
    __ke_file_operations_t* fops;
} __ke_proc_list_t;

extern __ke_proc_list_t firegl_proc_list[];

typedef struct {
    unsigned long           signature;
    int                     privdevcount; // number of privdev structs
    __ke_proc_list_t *      proclist;
    const char *            name;
	unsigned int	        major_version;
	unsigned int	        minor_version;
	unsigned int            patchlevel;
	const char *	        date;
    __ke_sigset_t *         psigmask;
    void *                  privglobal; // private global context
} __ke_device_t;

/*****************************************************************************/

/** KCL declarations */

/** Type definitions for variables containing OS dependent data 
 *
 * These type definitions are used for data that need to be saved in the
 * proprietary code but is not interpreted by the proprietary code in any way.
 * To manipulate values of these types proprietary code must use only KCL functions
 *
 * To be of the safe side, propietary code has to use these type definitions to
 * declare local variables storing values received from KCL functions.  
 * However, legacy code is allowed to use compatible
 * types.  Compiler is supposed to catch dangerous type conversions
 *
 * In each implementation of KCL, it needs to make sure that the size of the KCL
 * defined type is greater or equal to the size of the corresponding OS type.
 * This can by done by checking which KCL functions use each particular type,
 * and what is the corresponding OS type
 * 
 */
typedef int             KCL_TYPE_Pid;       /** Process identifier */
typedef int             KCL_TYPE_Tgid;      /** Thread Group identifier */
typedef int             KCL_TYPE_Uid;       /** User identifier */

/** Atomic variables
 * This type is defined using struct to make sure compiled code will
 * always refer to the memory containing the atomic variable (prevent
 * compiler from aliasing this memory)
 * Since atomic variables are intended for using in the concurrent
 * environment, volatile is used
 */
typedef struct { volatile unsigned int counter; } KCL_TYPE_Atomic;

/** OS independent constant definitions */

typedef enum
{
    KCL_ERROR_DEVICE_RESOURCE_BUSY,
    KCL_ERROR_INVALID_ARGUMENT,
    KCL_ERROR_PERMISSION_DENIED,
    KCL_ERROR_INVALID_ADDRESS,
    KCL_ERROR_INPUT_OUTPUT,
    KCL_ERROR_INVALID_SLOT,
    KCL_ERROR_OUT_OF_MEMORY,
    KCL_ERROR_OPERATION_NOT_PERMITTED,
    KCL_ERROR_DEVICE_NOT_EXIST,
    KCL_ERROR_INTERRUPTED_SYSTEM_CALL,
    KCL_ERROR_SIGNAL_INTERRUPTED_SYSTEM_CALL,
    KCL_ERROR_CORRUPTED_SHARED_LIB,
    KCL_ERROR_NUM
} KCL_ENUM_ErrorCode;

typedef enum
{
    KCL_PROCESS_STATE_READY_TO_RUN,
    KCL_PROCESS_STATE_UNINTERRUPTIBLE_SLEEP,
    KCL_PROCESS_STATE_INTERRUPTIBLE_SLEEP,
    KCL_PROCESS_STATE_NUM
} KCL_ENUM_ProcessState;

typedef enum
{
    KCL_SECURITY_CAP_GENERAL_SYS_ADMIN,
    KCL_SECURITY_CAP_IPC_LOCK,
    KCL_SECURITY_CAP_NUM
} KCL_ENUM_PosixSecurityCap;

typedef enum
{
    KCL_KERNEL_CONF_PARAM_HUGE_MEM,
    KCL_KERNEL_CONF_PARAM_NUM
} KCL_ENUM_KernelConfigParam;

/** KCL function declarations */

extern void          ATI_API_CALL KCL_GlobalKernelScheduler(void);
extern int           ATI_API_CALL KCL_GetSignalStatus(void);
extern unsigned int  ATI_API_CALL KCL_CurrentProcessIsTerminating(void);
extern unsigned int  ATI_API_CALL KCL_ProcessIsTerminated(unsigned int pid);
extern void          ATI_API_CALL KCL_SetCurrentProcessState(KCL_ENUM_ProcessState state);
extern const char*   ATI_API_CALL KCL_GetModuleParamString(void);
extern KCL_TYPE_Pid  ATI_API_CALL KCL_GetPid(void);
extern KCL_TYPE_Tgid ATI_API_CALL KCL_GetTgid(void);
extern KCL_TYPE_Uid  ATI_API_CALL KCL_GetEffectiveUid(void);
extern void          ATI_API_CALL KCL_DelayInMicroSeconds(unsigned long usecs);
extern void          ATI_API_CALL KCL_DelayUseTSC(unsigned long usecs);
extern unsigned long ATI_API_CALL KCL_ConvertAddressVirtualToBus(void* address);
extern unsigned long ATI_API_CALL KCL_ConvertAddressVirtualToPhysical(void* address);
extern void*         ATI_API_CALL KCL_GetHighMemory(void);
extern int           ATI_API_CALL KCL_GetErrorCode(KCL_ENUM_ErrorCode errcode);
extern void          ATI_API_CALL KCL_GlobalKernelLock(void);
extern void          ATI_API_CALL KCL_GlobalKernelUnlock(void);
extern int           ATI_API_CALL KCL_PosixSecurityCapCheck(KCL_ENUM_PosixSecurityCap cap);
extern void          ATI_API_CALL KCL_PosixSecurityCapSetIPCLock(unsigned int lock);
extern unsigned long ATI_API_CALL KCL_GetAvailableRamPages(void);
extern void          ATI_API_CALL KCL_ReserveMemPage(void* pt);
extern void          ATI_API_CALL KCL_UnreserveMemPage(void* pt);
extern void          ATI_API_CALL KCL_LockMemPage(void* pt);
extern void          ATI_API_CALL KCL_UnlockMemPage(void* pt);
extern int           ATI_API_CALL KCL_KernelConfigParamIsDefined(KCL_ENUM_KernelConfigParam param);
extern int           ATI_API_CALL KCL_SetPageCache(void* virt, int pages,int enable);
extern void          ATI_API_CALL KCL_AtomicInc(KCL_TYPE_Atomic* v);
extern void          ATI_API_CALL KCL_AtomicDec(KCL_TYPE_Atomic* v);
extern void          ATI_API_CALL KCL_AtomicAdd(KCL_TYPE_Atomic* v, int val);
extern void          ATI_API_CALL KCL_AtomicSub(KCL_TYPE_Atomic* v, int val);
extern int           ATI_API_CALL KCL_AtomicGet(KCL_TYPE_Atomic* v);
extern void          ATI_API_CALL KCL_AtomicSet(KCL_TYPE_Atomic* v, int val);
extern int           ATI_API_CALL KCL_AtomicIncAndTest(KCL_TYPE_Atomic* v);
extern int           ATI_API_CALL KCL_AtomicDecAndTest(KCL_TYPE_Atomic* v);
extern void          ATI_API_CALL KCL_AtomicSetBit(int nr, volatile void * addr);
extern void          ATI_API_CALL KCL_AtomicClearBit(int nr, volatile void * addr);
extern void          ATI_API_CALL KCL_AtomicToggleBit(int nr, volatile void* addr);
extern int           ATI_API_CALL KCL_AtomicTestBit(int nr, volatile void* addr);
extern int           ATI_API_CALL KCL_AtomicTestAndSetBit(int nr, volatile void* addr);
extern int           ATI_API_CALL KCL_AtomicTestAndClearBit(int nr, volatile void* addr);
extern int           ATI_API_CALL KCL_AtomicTestAndToggleBit(int nr, volatile void* addr);
extern int           ATI_API_CALL KCL_PosixSecurityCapCheck(KCL_ENUM_PosixSecurityCap cap);

/*****************************************************************************/

extern int ATI_API_CALL drm_name_info(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
extern int ATI_API_CALL firegl_bios_version(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
extern int ATI_API_CALL firegl_interrupt_info(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
extern int ATI_API_CALL drm_mem_info(char* buf, char** start, __ke_off_t offset, int len, int *eof, void *data);
extern int ATI_API_CALL drm_mem_info1(char* buf, char** start, __ke_off_t offset, int len, int *eof, void *data);
extern int ATI_API_CALL drm_vm_info(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
extern int ATI_API_CALL drm_clients_info(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
extern int ATI_API_CALL firegl_lock_info(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
#ifdef DEBUG
extern int ATI_API_CALL drm_bq_info(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
#endif
extern int ATI_API_CALL firegl_interrupt_open(void* data, KCL_IO_FILE_Handle file);
extern int ATI_API_CALL firegl_interrupt_release(KCL_IO_FILE_Handle file);
extern unsigned int ATI_API_CALL firegl_interrupt_read(
                                    KCL_IO_FILE_Handle user_file, 
                                    char *user_buf, 
                                    __ke_size_t user_buf_size, 
                                    __ke_loff_t *user_file_pos);
extern unsigned int ATI_API_CALL firegl_interrupt_poll(
        KCL_IO_FILE_Handle user_file, KCL_IO_FILE_PollTableHandle pt);
extern int ATI_API_CALL firegl_interrupt_write(
                                    KCL_IO_FILE_Handle user_file,
                                    const char *user_buf, 
                                    __ke_size_t user_buf_size, 
                                    __ke_loff_t *user_file_pos);

extern int ATI_API_CALL firegl_debug_proc_read(char* buf, char** start, __ke_off_t offset, int len, int* eof, void* data);
extern int ATI_API_CALL firegl_debug_proc_write(void* file, const char *buffer, unsigned long count, void *data); 

/*****************************************************************************/

extern int ATI_API_CALL firegl_private_init (__ke_device_t *);
extern void ATI_API_CALL firegl_private_cleanup (__ke_device_t *);
extern int ATI_API_CALL firegl_init(__ke_device_t*);
extern int ATI_API_CALL firegl_open(int minor, KCL_IO_FILE_Handle filp);
extern int ATI_API_CALL firegl_release(KCL_IO_FILE_Handle filp);
extern int ATI_API_CALL firegl_ioctl(
                        KCL_IO_FILE_Handle filp,
                        unsigned int cmd,
                        unsigned long arg);

#ifdef __x86_64__
extern long ATI_API_CALL firegl_compat_ioctl(
                        KCL_IO_FILE_Handle filp,
                        unsigned int cmd,
                        unsigned long arg);
#endif

/*****************************************************************************/

extern int ATI_API_CALL firegl_mmap(KCL_IO_FILE_Handle filp, struct vm_area_struct* vma);
extern void ATI_API_CALL drm_vm_open(struct vm_area_struct* vma);
extern void ATI_API_CALL drm_vm_close(struct vm_area_struct* vma);
extern void* ATI_API_CALL firegl_get_dev_from_vm(  struct vm_area_struct* vma );
extern void* ATI_API_CALL firegl_get_pcie_from_vm(  struct vm_area_struct* vma );
extern void* ATI_API_CALL firegl_get_pciemem_from_addr( struct vm_area_struct* vma, unsigned long addr );
extern unsigned long ATI_API_CALL firegl_get_pcie_pageaddr_from_vm(  struct vm_area_struct* vma, struct firegl_pcie_mem* pciemem, unsigned long offset);
extern void* ATI_API_CALL firegl_get_pagelist_from_vm(  struct vm_area_struct* vma );
extern unsigned long ATI_API_CALL firegl_get_addr_from_vm(  struct vm_area_struct* vma);
extern unsigned long ATI_API_CALL firegl_get_pagetable_page_from_vm(struct vm_area_struct* vma);
extern unsigned long ATI_API_CALL mc_heap_get_page_addr(void *vma, unsigned long offset);

/*****************************************************************************/

extern __ke_wait_queue_head_t* ATI_API_CALL __ke_alloc_wait_queue_head_struct(void);
extern void ATI_API_CALL __ke_free_wait_queue_head_struct(__ke_wait_queue_head_t* queue_head);
extern __ke_wait_queue_t* ATI_API_CALL __ke_alloc_wait_queue_struct(void);
extern void ATI_API_CALL __ke_free_wait_queue_struct(__ke_wait_queue_t* queue);

extern void ATI_API_CALL __ke_wait_event_interruptible(__ke_wait_queue_head_t* queue_head, int condition);
extern void ATI_API_CALL __ke_wake_up_interruptible(__ke_wait_queue_head_t* queue_head);
    
extern void ATI_API_CALL __ke_add_wait_queue(__ke_wait_queue_head_t* queue_head, __ke_wait_queue_t* entry);
extern void ATI_API_CALL __ke_remove_wait_queue(__ke_wait_queue_head_t* queue_head, __ke_wait_queue_t* entry);
extern void ATI_API_CALL __ke_init_waitqueue_head(__ke_wait_queue_head_t* queue_head);

extern void ATI_API_CALL __ke_configure_sigmask(__ke_sigset_t *pSigMask);
extern int ATI_API_CALL firegl_sig_notifier(void *priv);
extern int              firegl_sig_notifier_wrap(void *priv);
extern void ATI_API_CALL __ke_block_all_signals(int (*notifier)(void *priv), void *pPriv, __ke_sigset_t *pSigMask);
extern void ATI_API_CALL __ke_unblock_all_signals(void);

extern unsigned long ATI_API_CALL __ke__cmpxchg(volatile void *ptr, unsigned long old,                      
                      unsigned long new, int size);

#define __ke_cmpxchg(ptr,o,n)                        \
  ((__typeof__(*(ptr)))__ke__cmpxchg((ptr),(unsigned long)(o),      \
                 (unsigned long)(n),sizeof(*(ptr))))
/*****************************************************************************/

extern __ke_dev_t ATI_API_CALL __ke_getdevice(__ke_device_t *dev);

extern void ATI_API_CALL __ke_mod_inc_use_count(void);
extern void ATI_API_CALL __ke_mod_dec_use_count(void);

extern void ATI_API_CALL __ke_down_struct_sem(__ke_device_t *dev, int idx);
extern void ATI_API_CALL __ke_up_struct_sem(__ke_device_t *dev, int idx);
#define __KE_MAX_SEMAPHORES 4
extern void ATI_API_CALL __ke_sema_init(struct semaphore* sem, int value);
extern __ke_size_t ATI_API_CALL __ke_sema_size(void);
extern void ATI_API_CALL __ke_down(struct semaphore* sem);
extern void ATI_API_CALL __ke_up(struct semaphore* sem);

//PPLIB adding interruptible down for semaphore
extern int ATI_API_CALL __ke_down_interruptible(struct semaphore* sem);
//PPLIB end

extern void ATI_API_CALL __ke_spin_lock(__ke_device_t *dev, int ndx);
extern void ATI_API_CALL __ke_spin_unlock(__ke_device_t *dev, int ndx);
	#define __KE_MAX_SPINLOCKS 7

int ATI_API_CALL __ke_vsprintf(char *buf, const char *fmt, va_list ap);
int ATI_API_CALL __ke_vsnprintf(char *buf, size_t size, const char *fmt, va_list ap);

extern void ATI_API_CALL __ke_printk(const char* fmt, ...);

#ifdef __x86_64__
extern void* ATI_API_CALL KCL_AMD64_COMPAT32_AllocUserSpace(long size);
#endif
extern int ATI_API_CALL KCL_CopyFromUserSpace(void* to, const void* from, __ke_size_t size);
extern int ATI_API_CALL KCL_CopyToUserSpace(void* to, const void* from, __ke_size_t size);

extern void* ATI_API_CALL __ke_malloc(__ke_size_t size);
extern void* ATI_API_CALL __ke_malloc_atomic(__ke_size_t size);
extern void ATI_API_CALL __ke_free(void* p);
extern void ATI_API_CALL __ke_free_s(void* p, __ke_size_t size);
extern void* ATI_API_CALL __ke_vmalloc(__ke_size_t size);
extern void* ATI_API_CALL __ke_vmalloc_32(__ke_size_t size);
extern void* ATI_API_CALL __ke_vmalloc_atomic(__ke_size_t size);
extern void ATI_API_CALL __ke_vfree(void* p);
extern void* ATI_API_CALL __ke_get_free_page(void);
extern void* ATI_API_CALL __ke_get_free_pages(int order);
extern void ATI_API_CALL __ke_free_page(void* pt);
extern void ATI_API_CALL __ke_free_pages(void* pt, int order);
extern void ATI_API_CALL __ke_get_page(void* pt);
extern void ATI_API_CALL __ke_put_page(void* pt);
extern int ATI_API_CALL __ke_verify_read_access(void* addr, __ke_size_t size);
extern int ATI_API_CALL __ke_verify_write_access(void* addr, __ke_size_t size);
extern unsigned long ATI_API_CALL KCL_GetPageTableByVirtAddr(unsigned long virtual_addr, unsigned long* page_addr, int* is_highpte);
extern void ATI_API_CALL __ke_put_vm_page_table(unsigned long page_addr);
extern __ke_dma_addr_t ATI_API_CALL KCL_GetDmaPhysAddr(unsigned long virtual_addr);
extern void* ATI_API_CALL __ke_memset(void* s, int c, __ke_size_t count);
extern void* ATI_API_CALL __ke_memcpy(void* d, const void* s, __ke_size_t count);
extern ATI_API_CALL __ke_size_t __ke_strlen(const char *s);
extern char* ATI_API_CALL __ke_strcpy(char* d, const char* s);
extern char* ATI_API_CALL __ke_strncpy(char* d, const char* s, __ke_size_t count);
extern int ATI_API_CALL __ke_strcmp(const char *s1, const char *s2);
extern int ATI_API_CALL __ke_strncmp(const char* str1, const char* str2, __ke_size_t count);
extern int ATI_API_CALL __ke_strnicmp(const char* str1, const char* str2, __ke_size_t count);
extern char* ATI_API_CALL __ke_strchr(const char *s, int c);
extern int ATI_API_CALL __ke_sprintf(char* buf, const char* fmt, ...);
extern int ATI_API_CALL __ke_snprintf(char* buf, size_t size, const char* fmt, ...);
extern int ATI_API_CALL KCL_TestAndClearPageDirtyFlag(unsigned long virtual_addr);
extern unsigned long ATI_API_CALL __ke_do_mmap(KCL_IO_FILE_Handle  file, unsigned long addr, unsigned long len, unsigned long pgoff);
extern int ATI_API_CALL __ke_do_munmap(unsigned long addr, unsigned long len);
extern void* ATI_API_CALL __ke_vmap(unsigned long *pagelist, unsigned int count);
#ifdef FIREGL_USWC_SUPPORT
extern void* ATI_API_CALL KCL_vmap_wc(unsigned long *pagelist, unsigned int count);
#endif
extern void ATI_API_CALL __ke_vunmap(void* addr);
extern unsigned long ATI_API_CALL KCL_GetInitKerPte(unsigned long address);


/*****************************************************************************/

extern int ATI_API_CALL __ke_flush_cache(void);

/*****************************************************************************/

extern int ATI_API_CALL __ke_config_mtrr(void);
extern int ATI_API_CALL __ke_mtrr_add_wc(unsigned long base, unsigned long size);
extern int ATI_API_CALL __ke_mtrr_del(int reg, unsigned long base, unsigned long size);

extern int ATI_API_CALL KCL_is_pat_enabled(void);

/*****************************************************************************/

extern int ATI_API_CALL KCL_InstallInterruptHandler(unsigned int irq, void (*ATI_API_CALL handler)(void*), const char *dev_name, void *context);
extern void ATI_API_CALL KCL_UninstallInterruptHandler(unsigned int irq, void *context);

/*****************************************************************************/

extern void* ATI_API_CALL __ke_vma_file_priv(struct vm_area_struct* vma);
extern unsigned long ATI_API_CALL __ke_vm_start(struct vm_area_struct* vma);
extern unsigned long ATI_API_CALL __ke_vm_end(struct vm_area_struct* vma);
extern unsigned long ATI_API_CALL __ke_vm_offset(struct vm_area_struct* vma);
enum __ke_vm_maptype
{
    __KE_ADPT,
    __KE_SHM,
    __KE_CTX,
    __KE_PCI_BQS,
    __KE_AGP_BQS,
    __KE_AGP,
    __KE_SG,
    __KE_KMAP,
    __KE_GART_USWC,
    __KE_GART_CACHEABLE,
    __KE_ADPT_REG
};
extern char* ATI_API_CALL __ke_vm_flags_str(struct vm_area_struct* vma, char* buf);
extern char* ATI_API_CALL __ke_vm_page_prot_str(struct vm_area_struct* vma, char* buf);
extern char* ATI_API_CALL __ke_vm_phys_addr_str(struct vm_area_struct* vma, 
                                   char* buf, 
                                   unsigned long linear_address, 
                                   __ke_dma_addr_t* phys_address);
extern int ATI_API_CALL __ke_vm_map(KCL_IO_FILE_Handle filp,
                                    struct vm_area_struct* vma,
                                    unsigned long offset,
                                    enum __ke_vm_maptype type,
                                    int readonly);

#ifdef __x86_64__
/*
typedef int (*__ke_ioctl_trans_handler_t)(unsigned int, unsigned int,
                                          unsigned long, KCL_IO_FILE_Handle );
*/
extern int ATI_API_CALL __ke_register_ioctl32_conversion(unsigned int cmd, int (*handler)(unsigned int, unsigned int, unsigned long, KCL_IO_FILE_Handle));
extern void ATI_API_CALL __ke_unregister_ioctl32_conversion(unsigned int cmd);
#endif

/*****************************************************************************/

#ifdef __x86_64__

int ATI_API_CALL firegl_get_user_ptr(__ke_u32 *src,   void **dst);
int ATI_API_CALL firegl_get_user_u16(__ke_u16 *src,   __ke_u16 *dst);
int ATI_API_CALL firegl_get_user_u32(__ke_u32 *src,   __ke_u32 *dst);
int ATI_API_CALL firegl_get_user_u64(__ke_u32 *src,   __ke_u64 *dst);

int ATI_API_CALL firegl_put_user_ptr(void *src,      __ke_u32 *dst);
int ATI_API_CALL firegl_put_user_u16(__ke_u16 src,   __ke_u16 *dst);
int ATI_API_CALL firegl_put_user_u32(__ke_u32 src,   __ke_u32 *dst);
int ATI_API_CALL firegl_put_user_u64(__ke_u64 src,   __ke_u32 *dst);
int ATI_API_CALL firegl_init_32compat_ioctls(void);
void ATI_API_CALL firegl_kill_32compat_ioctls(void);

#endif

/*****************************************************************************/

extern int ATI_API_CALL firegl_pci_save_state(KCL_PCI_DevHandle pdev, struct drm_device* dev);
extern int ATI_API_CALL firegl_pci_restore_state(KCL_PCI_DevHandle pdev, struct drm_device* dev);
extern int ATI_API_CALL firegl_cail_powerdown(struct drm_device* dev, __ke_u32 state);
extern int ATI_API_CALL firegl_cail_powerup(struct drm_device* dev);
extern void ATI_API_CALL firegl_pm_save_framebuffer(struct drm_device* dev);
extern void ATI_API_CALL firegl_pm_restore_framebuffer(struct drm_device* dev);
extern void ATI_API_CALL firegl_pm_save_onchip_ram(struct drm_device* dev);
extern void ATI_API_CALL firegl_pm_restore_onchip_ram(struct drm_device* dev);
extern void ATI_API_CALL firegl_pm_restore_atigart(struct drm_device* dev);
extern void ATI_API_CALL firegl_pm_disable_interrupts(struct drm_device* dev);
extern void ATI_API_CALL firegl_pm_enable_interrupts(struct drm_device* dev);

/*****************************************************************************/

extern int ATI_API_CALL fgldbg_register_sysrq(int);
extern int ATI_API_CALL fgldump_register_sysrq(int);

/*****************************************************************************/

/* globals */
extern int __ke_debuglevel;
#define DEBUGFLAG_NO_HW_ACCESS  0x00020000
#define DEBUGLEVEL_MASK         0x0000ffff
#define debuglevel              (__ke_debuglevel &  DEBUGLEVEL_MASK)

extern int __ke_moduleflags;

/* module flags*/
#define __KE_FLAG_DEBUG                     0x00000001
#define __KE_FLAG_NOAUTH                    0x00000002
#define __KE_FLAG_DISABLE_FIREGL_AGPLOCK    0x00000010
#define __KE_FLAG_DISABLE_DYNAMIC_PCIE      0x00000020
#define __KE_FLAG_DISABLE_USWC              0x00000040

/* global constants */
extern const char*          __ke_UTS_RELEASE;
extern const unsigned long  __ke_LINUX_VERSION_CODE;
extern const unsigned int   __ke_PAGE_SHIFT;
extern const unsigned int   __ke_PAGE_SIZE;
extern const unsigned long  __ke_PAGE_MASK;

extern const unsigned long  __ke_MODVERSIONS_State;
extern const unsigned long  __ke_SMP_State;
extern const unsigned long  __ke_PAE_State;

/* global vars that are in fact constants */
extern unsigned long        __ke_HZ;

/*****************************************************************************/
/* Macros to make printk easier */
#define __KE_ERROR(fmt, arg...)                                             \
    __ke_printk("<3>[fglrx:%s] *ERROR* " fmt, __FUNCTION__, ##arg)

#define __KE_INFO(fmt, arg...)                                              \
    __ke_printk("<6>[fglrx] " fmt, ##arg)

#define __KE_DEBUG0(fmt, arg...)                                            \
    do                                                                      \
    {                                                                       \
        if (debuglevel > 0)                                                 \
        {                                                                   \
            __ke_printk("<7>[fglrx:%s] " fmt , __FUNCTION__, ##arg);        \
        }                                                                   \
    } while (0)
/*****************************************************************************/

#ifndef __GFP_COMP
#define __GFP_COMP 0
#endif

#ifdef FIREGL_USWC_SUPPORT

#ifndef MSR_IA32_CR_PAT
#define MSR_IA32_CR_PAT     0x277
#endif

#ifndef cpu_has_pat
#define cpu_has_pat  test_bit(X86_FEATURE_PAT, (void *) &boot_cpu_data.x86_capability)
#endif

#ifndef cpu_has_pge
#define cpu_has_pge test_bit(X86_FEATURE_PGE, &boot_cpu_data.x86_capability)
#endif

#ifndef pgprot_writecombine
#define pgprot_writecombine(prot) __pgprot((pgprot_val(prot) & ~(_PAGE_PCD)) | _PAGE_PWT)
#endif

#ifndef pgprot_noncached
#define pgprot_noncached(prot) __pgprot(pgprot_val(prot) | _PAGE_PCD | _PAGE_PWT)
#endif

#endif //FIREGL_USWC_SUPPORT


/*****************************************************************************
*                                                                            *
* Declarations for Kernel Abstraction Services                               *
*                                                                            *
******************************************************************************/
typedef void ATI_API_CALL (*KAS_CallbackWrapper_t)(void* proutine, void* pcontext);
typedef unsigned int ATI_API_CALL (*KAS_CallbackWrapperRet_t)(void* proutine, void* pcontext);

/** \brief Type definition for KAS initialization */
typedef struct tag_KAS_Initialize_t
{
    unsigned long exec_level_invalid;
    unsigned long exec_level_init;
    unsigned long exec_level_regular;
    unsigned long exec_level_idh;
    unsigned long exec_level_ih;
    KAS_CallbackWrapper_t callback_wrapper;
    KAS_CallbackWrapperRet_t callback_wrapper_ret;
} KAS_Initialize_t;

/** \brief Type definition for Interrupt Handling Routine */
typedef unsigned int (*KAS_IhRoutine_t)(void* pIhContext);

/** \brief Types of routines */
#define KAS_ROUTINE_TYPE_INVALID    0
#define KAS_ROUTINE_TYPE_REGULAR    1
#define KAS_ROUTINE_TYPE_IDH        2
#define KAS_ROUTINE_TYPE_IH         3

/** \brief Types of spinlocks */
#define KAS_SPINLOCK_TYPE_INVALID   0
#define KAS_SPINLOCK_TYPE_REGULAR   1
#define KAS_SPINLOCK_TYPE_IDH       2
#define KAS_SPINLOCK_TYPE_IH        3

/** \brief Return codes */
#define KAS_RETCODE_OK              0
#define KAS_RETCODE_ERROR           1
#define KAS_RETCODE_TIMEOUT         2
#define KAS_RETCODE_SIGNAL          3

/** \brief Interface functions */
extern unsigned int  ATI_API_CALL KAS_Initialize(KAS_Initialize_t* pinit);
extern unsigned int  ATI_API_CALL KAS_Ih_Execute(KAS_IhRoutine_t ih_routine,
                                                 void* ih_context);
extern unsigned int  ATI_API_CALL KAS_ExecuteAtLevel(void* pSyncRoutine,
                                                     void* pContext,
                                                     unsigned long sync_level);
extern unsigned long ATI_API_CALL KAS_GetExecutionLevel(void);

extern unsigned int  ATI_API_CALL KAS_Idh_GetObjectSize(void);
extern unsigned int  ATI_API_CALL KAS_Idh_Initialize(void* hIdh,
                                                     void* pfnIdhRoutine,
                                                     void* pIdhContext);
extern unsigned int  ATI_API_CALL KAS_Idh_Queue(void* hIdh);

extern unsigned int  ATI_API_CALL KAS_Spinlock_GetObjectSize(void);
extern unsigned int  ATI_API_CALL KAS_Spinlock_Initialize(void* hSpinLock,
                                             unsigned int spinlock_type);
extern unsigned int  ATI_API_CALL KAS_Spinlock_Acquire(void* hSpinLock);
extern unsigned int  ATI_API_CALL KAS_Spinlock_Release(void* hSpinLock);

extern unsigned int  ATI_API_CALL KAS_SlabCache_GetObjectSize(void);
extern unsigned int  ATI_API_CALL KAS_SlabCache_Initialize(void* hSlabCache,
                                                    unsigned int iEntrySize,
                                                    unsigned int access_type);
extern unsigned int  ATI_API_CALL KAS_SlabCache_Destroy(void* hSlabCache);
extern void*         ATI_API_CALL KAS_SlabCache_AllocEntry(void* hSlabCache);
extern unsigned int  ATI_API_CALL KAS_SlabCache_FreeEntry(void* hSlabCache,
                                                          void* pvEntry);

extern unsigned int  ATI_API_CALL KAS_Event_GetObjectSize(void);
extern unsigned int  ATI_API_CALL KAS_Event_Initialize(void* hEvent);
extern unsigned int  ATI_API_CALL KAS_Event_Set(void* hEvent);
extern unsigned int  ATI_API_CALL KAS_Event_Clear(void* hEvent);
extern unsigned int  ATI_API_CALL KAS_Event_WaitForEvent(void* hEvent,
                                                    unsigned long long timeout,
                                                    unsigned int timeout_use);

extern unsigned int  ATI_API_CALL KAS_Mutex_GetObjectSize(void);
extern unsigned int  ATI_API_CALL KAS_Mutex_Initialize(void* hMutex);
extern unsigned int  ATI_API_CALL KAS_Mutex_Acquire(void* hMutex,
                                                    unsigned long long timeout,
                                                    unsigned int timeout_use);
extern unsigned int  ATI_API_CALL KAS_Mutex_Release(void* hMutex);

extern unsigned int  ATI_API_CALL KAS_Thread_GetObjectSize(void);
extern unsigned int  ATI_API_CALL KAS_Thread_Start(void* hThread,
                                                   void* routine,
                                                   void* pcontext);
extern unsigned int  ATI_API_CALL KAS_Thread_WaitForFinish(void* hThread);
extern unsigned int  ATI_API_CALL KAS_Thread_Finish(void* hThread);

extern unsigned int  ATI_API_CALL KAS_InterlockedList_GetListHeadSize(void);
extern unsigned int  ATI_API_CALL KAS_InterlockedList_GetListEntrySize(void);
extern unsigned int  ATI_API_CALL KAS_InterlockedList_Initialize(
                                                    void* hListHead,
                                                    unsigned int access_type);
extern unsigned int  ATI_API_CALL KAS_InterlockedList_InsertAtTail(
                                                      void* hListHead,
                                                      void* hListEntry,
                                                      void** phPrevEntry);
extern unsigned int  ATI_API_CALL KAS_InterlockedList_InsertAtHead(
                                                      void* hListHead,
                                                      void* hListEntry,
                                                      void** phPrevEntry);
extern unsigned int  ATI_API_CALL KAS_InterlockedList_RemoveAtHead(
                                                      void* hListHead,
                                                      void** phRemovedEntry);

extern unsigned int  ATI_API_CALL KAS_AtomicCompareExchangeUnsignedInt(
                                                unsigned int *puiDestination,
                                                unsigned int uiExchange,
                                                unsigned int uiComparand);

extern unsigned int  ATI_API_CALL KAS_AtomicExchangeUnsignedInt(
                                                unsigned int *puiDestination,
                                                unsigned int uiExchange);

extern unsigned int  ATI_API_CALL KAS_AtomicExchangeAddUnsignedInt(
                                                unsigned int *puiDestination,
                                                unsigned int uiAdd);

extern unsigned int  ATI_API_CALL KAS_AtomicAddInt(
                                                unsigned int *puiDestination,
                                                int iAdd);

extern void*         ATI_API_CALL KAS_AtomicCompareExchangePointer(
                                                        void* *ppvDestination,
                                                        void* pvExchange,
                                                        void* pvComparand);

extern void*         ATI_API_CALL KAS_AtomicExchangePointer(
                                                        void* *ppvDestination,
                                                        void* pvExchange);

extern unsigned long ATI_API_CALL KAS_GetTickCounter(void);
extern unsigned long ATI_API_CALL KAS_GetTicksPerSecond(void);
extern  long ATI_API_CALL KAS_ScheduleTimeout(long n_jiffies);

/******************************************************************************
**
**  Interface layer to asyncIO layer
**
*******************************************************************************/
#define FIREGL_ASYNCIO_MAX_DEV      4    /* Currently hardcode to 4  */
#define FIREGL_ASYNCIO_MAX_FILE     32   /* Currently hardcode to 32 */
#define FIREGL_ASYNCIO_MAX_SEMA     ((FIREGL_ASYNCIO_MAX_FILE+1)*FIREGL_ASYNCIO_MAX_DEV)

extern void * ATI_API_CALL __ke_asyncio_alloc_sema(void);
extern void ATI_API_CALL __ke_asyncio_free_sema(struct semaphore *pSema);
extern void ATI_API_CALL __ke_asyncio_init_sema(void);

typedef enum {
    __KE_POLLIN = 0,
    __KE_POLLRDNORM,
    __KE_EAGAIN,
    __KE_FASYNC_ON,
    __KE_FASYNC_OFF,
    __KE_SIGIO,
    __KE_ESPIPE,
    __KE_EINTR
} __ke_asynio_contant_t;

extern int ATI_API_CALL __ke_getConstant(int contant);

extern __ke_ssize_t ATI_API_CALL firegl_asyncio_read( KCL_IO_FILE_Handle filp,
                                                      char *buf, 
                                                      __ke_size_t size,
                                                      __ke_loff_t *off_ptr);

extern __ke_ssize_t ATI_API_CALL firegl_asyncio_write( KCL_IO_FILE_Handle filp,
                                                       const char *buf, 
                                                       __ke_size_t size,
                                                       __ke_loff_t *off_ptr);

extern unsigned int ATI_API_CALL firegl_asyncio_poll(
        KCL_IO_FILE_Handle filp, KCL_IO_FILE_PollTableHandle table);

extern int ATI_API_CALL firegl_asyncio_fasync(int fd, 
                                              KCL_IO_FILE_Handle filp,
                                              int mode);


extern void *ATI_API_CALL KCL_lock_init(void);
extern void ATI_API_CALL KCL_lock_deinit(void *plock);
extern void ATI_API_CALL KCL_spin_lock(void *lock);
extern void ATI_API_CALL KCL_spin_unlock(void *lock);
extern void ATI_API_CALL KCL_get_random_bytes(void *buf, int nbytes);
extern void ATI_API_CALL adapter_chain_init(void);
extern void ATI_API_CALL adapter_chain_cleanup(void);
extern void ATI_API_CALL cf_object_init(void);
extern void ATI_API_CALL cf_object_cleanup(void);
extern int ATI_API_CALL firegl_init_device_list(int num_of_devices);
extern void ATI_API_CALL firegl_cleanup_device_heads(void);
extern int ATI_API_CALL firegl_get_dev(void *pubdev, KCL_PCI_DevHandle pdev);
extern void *ATI_API_CALL firegl_find_device(int minor);
extern void* ATI_API_CALL firegl_query_pcidev(KCL_PCI_DevHandle pdev);

extern void* ATI_API_CALL KCL_get_pubdev(void);
extern void ATI_API_CALL KCL_dump_stack(void);
/*******************************************************/
extern int ATI_API_CALL firegl_dbg_stub(__ke_device_t *pubdev);
extern void ATI_API_CALL firegl_dump(__ke_device_t *pubdev);

//Add functions need to be traced
typedef enum
{
    FN_FIREGL_OPEN_RELEASE = 0,
    FN_FIREGL_IOCTL,
    FN_FIREGL_MMAP,
    FN_FIREGL_READ_WRITE,
    FN_FIREGL_FASYNC,
    FN_FIREGL_POLL,
    FN_FIREGL_LSEEK,
    FN_FIREGL_COMPAT_IOCTL,
    FN_DRM_VM_OPEN_CLOSE,
    FN_DRM_NOPAGE,
    FN_FIREGL_ACPI,
    FN_FIREGL_IRQ,
    FN_FIREGL_PROC,
    FN_FIREGL_KAS,
    FN_FIREGL_KCL,
    FN_FIREGL_INIT,
    FN_GENERIC1,
    FN_GENERIC2,
    FN_GENERIC3,                      //Added more if necessary  
    FN_FIREGL_RESERVED =0x20,         //0x20 -0x3f reserverd 
    FN_GENERIC_MAXIMUM  = 0x3f
} FN_TRACE;

typedef enum
{
    FN_DEBUG_TRACE = 0,
    FN_DEBUG_LEVEL1,
    FN_DEBUG_LEVEL2,
    FN_DEBUG_LEVEL3,
    FN_DEBUG_LEVEL4,
    FN_DEBUG_LEVEL5,
    FN_DEBUG_LEVEL6,
    FN_DEBUG_LEVEL7,
    FN_DEBUG_MAXIMUM  = 0x8
} FN_DEBUG;

#define FN_DEBUG_LEVEL_ALL 0xff

extern void ATI_API_CALL firegl_trace(unsigned int traceMask, unsigned int debugMask, void* name, int line, long param, const char* fmt, ...);

#define __KE_TRACE(_m, _p, fmt, arg...)                                                                     \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_TRACE, (void *)__FUNCTION__ , (int)(__LINE__), (long)(_p), fmt, ##arg);   \
    } while (0)


#define __KE_DEBUG(_m, fmt, arg...)                                                                         \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_LEVEL1, (void *)__FUNCTION__ , (int)__LINE__, 0, fmt, ##arg);          \
    } while (0)

#define __KE_DEBUG2(_m, fmt, arg...)                                                                        \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_LEVEL2, (void *)__FUNCTION__ , (int)__LINE__, 0, fmt, ##arg);             \
    } while (0)

#define __KE_DEBUG3(_m, fmt, arg...)                                                                        \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_LEVEL3, (void *)__FUNCTION__ , (int)__LINE__, 0, fmt, ##arg);             \
    } while (0)

#define __KE_DEBUG4(_m, fmt, arg...)                                                                        \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_LEVEL4, (void *)__FUNCTION__ , (int)__LINE__, 0, fmt, ##arg);             \
    } while (0)

#define __KE_DEBUG5(_m, fmt, arg...)                                                                        \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_LEVEL5, (void *)__FUNCTION__ , (int)__LINE__, 0, fmt, ##arg);             \
    } while (0)

#define __KE_DEBUG6(_m, fmt, arg...)                                                                        \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_LEVEL6, (void *)__FUNCTION__ , (int)__LINE__, 0, fmt, ##arg);             \
    } while (0)    

#define __KE_DEBUG7(_m, fmt, arg...)                                                                        \
    do                                                                                                      \
    {                                                                                                       \
        firegl_trace(_m, FN_DEBUG_LEVEL7, (void *)__FUNCTION__ , (int)__LINE__, 0, fmt, ##arg);             \
    } while (0)    

/***********************************************************/

#endif /* _FIREGL_PUBLIC_H_ */
