# -*- coding: UTF-8 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

'''RestrictedManagerCommon:
   this file contains common code in most of the
   restricted-manager frontends. It provides classes
   and methods that are reimplemented in the corresponding
   frontends. 
   RestrictedManagerCommon is not (yet) a CLI frontend for 
   restricted-manager.
'''

__copyright__ = 'Copyright © 2007 Canonical Ltd.'
__author__    = 'Scott James Remnant <scott@ubuntu.com> and AUTHORS'


import os
import sys
import gettext
from optparse import OptionParser
import subprocess
import RestrictedManager.core
from RestrictedManager.core import *

import gettext

class RepositoryError(Exception):
    ''' an exception raised when the needed restricted package
        is not in a repository.'''
    pass

def _(str):
    ''' translation function with modifications needed for KDE '''
    return unicode(gettext.gettext(str), 'UTF-8')

class RestrictedManagerCommon(object):
    ''' The main common class, currently inherited by all of the
        frontends. '''
    def __init__(self,args, opts):
        ''' __init__() is common to all frontends. It launches
        the frontend-specific methods/classes using the
        launch_*() methods. '''

        gettext.textdomain("restricted-manager")
                
        # these classes are usually implemented in the children of RestrictedManagerCommon
        RestrictedManager.core.DefaultHandler.install_package = self.install_package
        RestrictedManager.core.DefaultHandler.remove_package = self.remove_package
        RestrictedManager.core.FWHandler.FWSelectDialog = self.FWSelectDialog


        # we have to check if opts&args aren't empty (like in KDE-KCM)
        if args != None and opts != None: 
            if opts.update:
                specials = get_specials()
                load_restricted_list(specials, force=True)
            elif opts.list:
                handlers = get_handlers()
                for handler in handlers.values():
                    print handler.name
                                    
            elif opts.check:
                handlers = get_handlers()
                used = load_cache_list("used")
                seen = load_cache_list("seen")
                new_handlers = [] # those which did not exist before
                new_used = [] # those which were not loaded before
        
                for handler in handlers.values():
                    if handler.name not in seen:
                        new_handlers.append(handler.name)
                    if handler.is_loaded() and handler.name not in used:
                        new_used.append(handler.name)

                if new_used:
                    used += new_used
                    save_cache_list("used", used)
                if new_handlers:
                    seen += new_handlers
                    save_cache_list("seen", seen)

                # throw out new handlers which are already enabled, no need to
                # bother people about them; also throw out handlers which need
                # a driver package which is not available for installation
                for h in new_handlers + []: # create a copy for iteration
                    if handlers[h].is_enabled():
                        new_handlers.remove(h)
                    if hasattr(handlers[h], "driver_package") and \
                        not DefaultHandler.package_available(handlers[h].driver_package):
                        print "Warning: handler", h, "needs unavailable driver package", handlers[h].driver_package
                        new_handlers.remove(h)

                if new_handlers:
                    self.launch_notification(_("Restricted drivers available"),
                        _("In order to use your hardware more efficiently, you"
                        " can enable drivers which are not free software."),
                        _("Click on this icon to open the Restricted Drivers control panel."))
                elif new_used:
                    self.launch_notification(_("New restricted drivers in use"),
                        # %(os)s stands for the OS name. Prefix it or suffix it,
                        # but do not replace it.
                        _("In order for this computer to function properly, %(os)s is "
                        "using driver software that cannot be supported by %(os)s.") % {"os": get_os_name()},
                        _("Click on this icon to open the Restricted Drivers control panel."))
                        
            elif opts.check_composite:
                handlers = get_handlers()

                # check for nvidia
                for d in ("nvidia_new", "nvidia", "nvidia_legacy"):
                    if handlers.has_key(d):
                        if handlers[d].is_enabled():
                            print d, "restricted driver is already enabled" # debug
                            self.exit(1)
                        else:
                            self.runme_as_root(['--enable', d])
                            sys.exit(0)

                print "No nvidia hardware available" # debug
                sys.exit(1)
        
            elif os.getuid() != 0:
                # instead of complaining about root, launch yourself like root
                self.runme_as_root(sys.argv[1:])
            elif opts.enable:
                handlers = get_handlers()
                try:
                    handler = handlers[opts.enable]
                except KeyError:
                    self.show_dialog('error', _('Unknown module: %s') % opts.enable)
                    self.exit(1)
        
                if handler.is_enabled():
                    self.show_dialog('error', _('%s is already enabled') % opts.enable)
                    self.exit(1)
        
                ch = handler.can_change()
                if ch:
                    self.show_dialog(type='error',message = ch)
                    self.exit(1)
        
                #if ConfirmDialog(xml, handler, False).run() != gtk.RESPONSE_OK:
                if not self.launch_confirm(handler, False):
                    self.exit()
        
                handler.enable()
            elif opts.disable:
                handlers = get_handlers()
                try:
                    handler = handlers[opts.disable]
                except KeyError:
                    self.show_dialog('error', _('Unknown module: %s') % opts.disable)
                    self.exit(1)
        
                if not handler.is_enabled():
                    self.show_dialog('error', _('%s is already disabled') % opts.disable)
                    self.exit(1)
        
                ch = handler.can_change()
                if ch:
                    self.show_dialog(type = 'error',message = ch)
                    self.exit(1)
        
                if not self.launch_confirm(handler, True):
                    self.exit(1)
        
                handler.disable()
            else:
                # check whether we have restricted modules installed
                # FIXME: to be honest, this only works if you have a standard kernel (ie. from the repos)
                #        we should make sure this is true, otherwise warn the user
                lrm = 'linux-restricted-modules-' + os.uname()[2]
                if not DefaultHandler.package_installed(lrm):
                    self.show_dialog(type='error',message = _(
                                'You need to install the package\n\n  %s\n\nfor this program to work.') % lrm)
                    self.exit(0)
        
                self.handlers = get_handlers()
        
                if not self.handlers:
                    self.show_dialog(type='info', message=_(
                                'Your hardware does not need any restricted drivers.'))
                    self.exit(0)
                self.launch_manager()
        else:
            # triggered only by non-standalone instances of RestrictedManagerKDE
            lrm = 'linux-restricted-modules-' + os.uname()[2]
            if not DefaultHandler.package_installed(lrm):
                self.show_dialog(type='error',message = _(
                            'You need to install the package\n\n  %s\n\nfor this program to work.') % lrm)
                self.exit(0)
    
            self.handlers = get_handlers()
    
            if not self.handlers:
                self.show_dialog(type='info', message=_(
                            'Your hardware does not need any restricted drivers.'))
                self.exit(0)
            self.launch_manager()

    def runme_as_root(self, extra_argv = []):
        '''Execute myself through a sudo frontend.
           Not implemented in the Common core for extra stability.'''
        raise NotImplementedError
    
    def exit(self, return_value=0):
        ''' exit() provides an overridable exit method for the frontends that need it.
            The KDE frontend uses it when in non-standalone mode.'''
        sys.exit(return_value)

    def title_label(self, in_use=False):
        '''Returns an appropriate window title into the given label, depending on
        whether some restricted drivers are in use (in_use=True) or not
        (in_use=False).'''
    
        if in_use:
            t = _('Proprietary drivers are being used to make this computer work properly.')
        else:
            t = _('No proprietary drivers are in use on this system')
    
        return (t,
            # %(os)s stands for the OS name. Prefix it or suffix it,
            #  but do not replace it.
            _('Proprietary drivers do not have public source code that %(os)s '
            'developers are free to modify. They represent a risk to you '
            'because they are only available on the types of computer chosen by '
            'the manufacturer, and security updates to them depend solely on the '
            'responsiveness of the manufacturer. %(os)s cannot fix or improve '
            'these drivers.') % {'os': get_os_name()}
        )
 
    def install_package(self, handlerInstance, package):
        ''' An overriddable method for installing the package) 
            Also hosts the package installation common code.
            The handlerInstance is passed from the driver handler
            backend.'''
        
        # Before doing anything, check if the repository needed for
        # package is enabled
        apt_cache = subprocess.Popen(("/usr/bin/apt-cache", "show", package),
                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        (stdout, stderr) = apt_cache.communicate()
        if apt_cache.returncode != 0 or not stdout: 
            self.show_dialog(type = 'error', message = _(
                    "The software source for the package\n\n   %s\n\n is not enabled.")
                     % package)

            raise RepositoryError
 
    def remove_package(self, handlerInstance, package):
        ''' An overriddable method for removing the package) '''
        pass
 
    def launch_notification(self, title, text, action_text):
        ''' An overriddable method for launching the notification '''
        pass
 
    def launch_manager(self, iconify=False):
        ''' An overriddable method for launching the manager.
            The iconify= parameter is used in KDE frontend
            to hide the window in system tray. '''
        pass
 
    def launch_confirm(self, handler, is_enabled=False, parent = None):
        ''' An overriddable method for launching the confirm dialog. '''
        pass
    
    def show_dialog(self, type, message):
        ''' An overloadable method for displaying a message '''
        if type == 'error':
            print>>sys.stderr, 'ERROR:' + message
        elif type == 'info':
            print message
    

    class ManagerWindow(object):
        ''' the common manager window class, contains just translations now '''
        def __init__(self):
            self.caption_text = _('Restricted Drivers')
            self.component_text = _('Component')
            self.enabled_text = _('Enabled')
            self.status_text = _('Status')
            self.driver_text = _('Driver')
            self.firmware_text = _('Firmware')
            self.restart_text = _('Needs computer restart')
            self.in_use_text = _('In use')
            self.not_in_use_text = _('Not in use')


    class Notification(object):
        ''' the Notification common class contains translations for all of the frontends'''
        def __init__(self, parent, title, text, action_text):
            self.parent = parent
            self.title = title
            self.text = text
            self.action_text = action_text

    class ConfirmDialog(object):
        def __init__(self, handler, is_enabled):
            ''' handles setting up frob & button label '''
            self.handler = handler
            self.is_enabled = is_enabled
            self.frob = ''
            self.button_label = ''
            
            if self.is_enabled:
                if self.handler._type == 'Driver':
                    self.frob = _('Disable the Driver?')
                    self.button_label = _('_Disable Driver')
                else:
                    assert self.handler._type == 'Firmware'
                    self.frob = _('Disable the Firmware?')
                    self.button_label = _('_Disable Firmware')
            else:
                if self.handler._type == 'Driver':
                    self.frob = _('Enable the Driver?')
                    self.button_label = _('_Enable Driver')
                else:
                    assert self.handler._type == 'Firmware'
                    self.frob = _('Enable the Firmware?')
                    self.button_label = _('_Enable Firmware')


    class FWSelectDialog(object):
        ''' A common parent dialog for all the frontends '''
        class FWProgressDialog(object):
            ''' a common parent for all the frontend firmware progress dialogs '''
            def __init__(self):
                ''' init just inserts translations now '''
                self.dlfw_text = _('Downloading Firmware')


        def __init__(self, FWCheckCommand = None):
            ''' setting up translations and other common aspects '''
            self.browse_text = _('Browse')
            self.loc_label_text = _('Location')
            self.label_title = _('Specify firmware location')
            self.label =  _('Please specify where the firmware can be found.\n\n'
                            'This can either be a file on your system (CD-ROM, Windows partition, etc) or a URL.')
            self.radio_local_text = _('Use a local file')
            self.radio_remote_text = _('Download from Internet')
            self.downloading_text = _('Downloading')
            self.dl_firmware_text = _('Downloading Firmware')
            self.error_fw_invalid_text = _('The firmware location or the firmware file itself you chose is not valid\n\n'
                                      'Please make sure you select a firmware file in a proper location') 
            self.check_command = FWCheckCommand

        def check_firmware(self, location):
            ''' checks if the (local) location is valid and the file is a firmware 
                file fetching is up to the frontend '''
            fw_location_is_wrong = True
            
            if os.path.exists(location):
                fw_location_is_wrong = False
                
            if not fw_location_is_wrong:
                # So the url is valid but are we sure it is a firmware?
                # Check wich check_command if given and update fw_location_is_wrong
                # according to the check_program exit status
                if self.check_command is not None:
                   self.check_command.append(location)
                   proc = subprocess.Popen(self.check_command)
                   while proc.poll() == None: time.sleep(0.1)
                   if proc.returncode != 0:
                      fw_location_is_wrong = True
                      self.check_command.remove(location)

            return fw_location_is_wrong

def get_opt_args():
    ''' parses the application arguments '''
    parser = OptionParser()
    parser.add_option ('-u', '--update', action='store_true',
                       dest='update', default=False,
                       help=_('Update restricted modules list'))
    parser.add_option ('-c', '--check', action='store_true',
                       dest='check', default=False,
                       help=_('Check for newly used restricted modules'))
    parser.add_option ('-l', '--list', action='store_true',
                       dest='list', default=False,
                       help=_('List available restricted modules'))
    parser.add_option ('-C', '--check-composite', action='store_true',
                       dest='check_composite', default=False,
                       help=_('Check if there is a graphics driver available that supports composite and offer to enable it'))
    parser.add_option ('-e', '--enable', type='string',
                       dest='enable', default='', metavar='MODULE',
                       help=_('Enable the named module'))
    parser.add_option ('-d', '--disable', type='string',
                       dest='disable', default='', metavar='MODULE',
                       help=_('Disable the named module'))
    return parser.parse_args()
