# -*- coding: UTF-8 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import os
import sys
import subprocess
import gettext
import textwrap
from optparse import OptionParser

import pygtk
pygtk.require('2.0')

import gobject
import gtk
import gtk.glade
import pynotify

import RestrictedManager.core
from RestrictedManager.core import *

from RestrictedManager.RestrictedManagerCommon import RestrictedManagerCommon, get_opt_args, RepositoryError, _, __file__


class RestrictedManagerGtk(RestrictedManagerCommon):
    ''' Restricted Manager  class for Gtk/GNOME  '''
    def __init__(self):
        (opts,args) = get_opt_args()
        gtk.glade.textdomain("restricted-manager")
        try:
            self.xml = gtk.glade.XML('/usr/share/restricted-manager/manager.glade')
        except RuntimeError:
            glade_filename = os.path.join(os.path.dirname(RestrictedManager.RestrictedManagerCommon.__file__),
                                        'manager.glade')
            self.xml = gtk.glade.XML(glade_filename)


        self.set_title_label(self.xml.get_widget('label_heading'), False)

        RestrictedManagerCommon.__init__(self,args,opts)
        self.mngr = None

    def gtk_idle(self):
        while gtk.events_pending():
            gtk.main_iteration(False)

    # moved from core
    def install_package(self, handlerInstance, package):
        '''Install given package through synaptic and return synaptic's exit
           status.
           Might throw OSError if synaptic is not available.
           The handlerInstance is passed from the driver handler
           backend.'''

        try:
            RestrictedManagerCommon.install_package(self, handlerInstance, package)
        except RepositoryError:
            return

        argv = ['/usr/sbin/synaptic', '--set-selections', '--non-interactive',
            '--hide-main-window']
        if package_install_xid:
            argv += ['--parent-window-id', str(package_install_xid)]
    
        os.environ['DEBIAN_PRIORITY'] = 'critical'
        synaptic = subprocess.Popen(argv, stdin=subprocess.PIPE)
        if self.gtk_idle():
            synaptic.stdin.write(package + ' install')
            synaptic.stdin.close()
            while synaptic.poll() is None:
                time.sleep(0.1)
                self.gtk_idle()
        else:
            synaptic.communicate(package + ' install')
    
        if handlerInstance.package_installed(package):
            update_installed_packages([package], [])
        return synaptic.returncode


    def remove_package(self, handlerInstance, package):
        '''Remove given package through synaptic and return synaptic's exit
           status.
           Might throw OSError if synaptic is not available.
           The handlerInstance is passed from the driver handler
           backend.'''
    
        argv = ['/usr/sbin/synaptic', '--set-selections', '--non-interactive',
            '--hide-main-window']
        if package_install_xid:
            argv += ['--parent-window-id', str(package_install_xid)]
    
        synaptic = subprocess.Popen(argv, stdin=subprocess.PIPE)
        if self.gtk_idle():
            synaptic.stdin.write(package + ' deinstall')
            synaptic.stdin.close()
            while synaptic.poll() is None:
                time.sleep(0.1)
                self.gtk_idle()
        else:
            synaptic.communicate(package + ' deinstall')
    
        if not handlerInstance.package_installed(package):
            update_installed_packages([], [package])
    
        return synaptic.returncode
    
    def set_title_label(self, label, in_use=False):
        (bold, normal) = RestrictedManagerCommon.title_label(self, in_use)
        if label != None:
            label.set_label('<span weight="bold">%s</span>\n\n%s' % (bold, normal))

    def show_dialog(self,type,message):
        if type == 'error':
            gtype = gtk.MESSAGE_ERROR
        elif type == 'info':
            gtype = gtk.MESSAGE_INFO
        md = gtk.MessageDialog(type=gtype, buttons=gtk.BUTTONS_CLOSE, message_format=message)
        md.run()
        md.hide()
    
    def launch_notification(self, title, text, action_text):
        ''' launching GNOME-based notification '''
        # in this case, self is a parent
        self.Notification(self, title, text, action_text)
        gtk.main()

    def launch_confirm(self, handler, is_enabled, parent=None):
        cdlg = self.ConfirmDialog(self.xml, handler,is_enabled, parent)
        if cdlg.run() == gtk.RESPONSE_OK:
            return True
        else:
            return False
        
    def launch_manager(self,iconify=False):
        RestrictedManagerCommon.launch_manager(self)
        
        self.ManagerWindow(self.xml, self.handlers, self)
        RestrictedManager.core.package_install_xid = \
            self.xml.get_widget('dialog_manager').window.xid

        gtk.main()

    def runme_as_root(self, extra_argv = []):
        argv = ["gksu", "-D", _("Restricted Drivers manager"),
                "--", sys.argv[0]] + extra_argv
        os.execvp("gksu", argv)

 
    class ManagerWindow(RestrictedManagerCommon.ManagerWindow):
        def __init__(self, xml, handlers, parent):
            RestrictedManagerCommon.ManagerWindow.__init__(self)
            self.xml = xml
            self.parent = parent
            gtk.window_set_default_icon_name('restricted-manager')
            self.dialog = self.xml.get_widget('dialog_manager')
            self.dialog.connect('delete_event', self.on_dialog_delete_event)
    
            self.treeview = self.xml.get_widget('treeview_drivers')
            self.treeview.connect('cursor-changed', self.on_drivers_cursor_changed)
    
            text_renderer = gtk.CellRendererText()
            col1 = gtk.TreeViewColumn(self.component_text)
            col1.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
            col1.set_expand(True)
            col1.pack_start(text_renderer, True)
            col1.set_attributes(text_renderer, text=3)
            self.treeview.append_column(col1)
    
            toggle_renderer = gtk.CellRendererToggle()
            toggle_renderer.set_property('activatable', True)
            toggle_renderer.connect('toggled', self.on_renderer_toggled)
            col2 = gtk.TreeViewColumn(self.enabled_text)
            col2.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
            col2.set_expand(False)
            col2.pack_start(toggle_renderer, True)
            col2.set_attributes(toggle_renderer, active=4, visible=7)
            self.treeview.append_column(col2)
    
            pixbuf_renderer = gtk.CellRendererPixbuf()
            text_renderer = gtk.CellRendererText()
            col3 = gtk.TreeViewColumn(self.status_text)
            col3.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
            col3.set_expand(False)
            col3.pack_start(pixbuf_renderer, False)
            col3.pack_end(text_renderer, True)
            col3.set_attributes(pixbuf_renderer, pixbuf=5)
            col3.set_attributes(text_renderer, text=6)
            self.treeview.append_column(col3)
    
            self.button_close = self.xml.get_widget('button_close')
            self.button_close.connect('clicked', self.on_button_close_clicked)
            self.xml.get_widget('button_help').connect('clicked', self.on_button_help_clicked)
    
            self.handlers = handlers
            self.reset_model()
    
            # to remove the selection outline around the first item of the tree
            self.treeview.grab_focus()
    
            self.dialog.show()
    
        def reset_model(self):
            self.model = gtk.TreeStore(gobject.TYPE_PYOBJECT,
                                    gobject.TYPE_PYOBJECT,
                                    gobject.TYPE_PYOBJECT,
                                    str, bool, gtk.gdk.Pixbuf, str,
                                    bool # toggle visible or not
                                    )
    
            theme = gtk.icon_theme_get_default()
    
            any_enabled = False
    
            parents = {}
            handlers = sorted(self.handlers.values(), key=lambda x: (x._type, x.description()))
            for handler in handlers:
                is_enabled = handler.is_enabled()
                is_loaded = handler.is_loaded()
    
                if handler._type == 'Driver':
                    type = self.driver_text
                else:
                    assert handler._type == 'Firmware'
                    type = self.firmware_text
    
                if type not in parents.keys():
                    parents[type] = self.model.append(None, [None, None, None, type, None, None, None, False])
    
                if (is_enabled != is_loaded) and handler.is_changed():
                    status, icon = self.restart_text, 'reboot-notifier'
                elif is_loaded:
                    status, icon = self.in_use_text, gtk.STOCK_YES
                    any_enabled = True
                else:
                    status, icon = self.not_in_use_text, gtk.STOCK_NO
                try:
                    pixbuf = theme.load_icon(icon, 16, gtk.ICON_LOOKUP_USE_BUILTIN)
                except:
                    # do not fail if icon is not found, just create a 1px pixbuf
                    pixbuf = gtk.gdk.Pixbuf(gtk.gdk.COLORSPACE_RGB, True, 8, 1, 1)
                self.model.append(parents[type], [ handler, is_enabled, is_loaded,
                                    handler.description(), is_enabled, pixbuf,
                                    status, True ])
    
            self.treeview.set_model(self.model)
            self.treeview.expand_all()
    
            self.parent.set_title_label(self.xml.get_widget('label_heading'), any_enabled)
    
        def on_dialog_delete_event(self, *args):
            gtk.main_quit()
    
        def on_renderer_toggled(self, widget, path, *args):
            self.action(path)

        def on_drivers_cursor_changed(self, widget):
            handler = self.model[widget.get_cursor()[0]][0]
            if isinstance(handler, DefaultHandler):
                tip = ''
                for par in handler.rationale.split('\n'):
                    if tip:
                        tip += '\n'
                    tip += '\n'.join(textwrap.wrap(par, 60))
                self.treeview.set_tooltip_text(tip)
                self.treeview.set_property('has-tooltip', True)
            else:
                self.treeview.set_property('has-tooltip', False)
    
        def action(self, path):
            handler = self.model[path][0]
            is_enabled = self.model[path][1]
            is_loaded = self.model[path][2]
    
            ch = handler.can_change()
    
            if not ch:
                res = self.parent.launch_confirm(handler, is_enabled, self.dialog)
                if  res == False:
                    return
    
                self.treeview.set_sensitive(False)
                if is_enabled:
                    handler.disable()
    
                    if handler.is_loaded():
                        notify_reboot_required()
                else:
                    handler.enable()
                self.treeview.set_sensitive(True)
            else:
                md = gtk.MessageDialog(self.dialog, type=gtk.MESSAGE_ERROR,
                            buttons=gtk.BUTTONS_CLOSE, message_format=ch)
                md.run()
                md.hide()
                return
    
            self.reset_model()
            self.treeview.grab_focus()
    
        def on_button_close_clicked(self, *args):
            gtk.main_quit()

        def on_button_help_clicked(self, *args):
            gobject.spawn_async(["yelp", "ghelp:hardware#restricted-manager"],
                flags=gobject.SPAWN_SEARCH_PATH)
    
    class ConfirmDialog(RestrictedManagerCommon.ConfirmDialog):
        def __init__(self, xml, handler, is_enabled, parent=None):
            RestrictedManagerCommon.ConfirmDialog.__init__(self, handler, is_enabled)
            self.xml = xml  
            self.dialog = self.xml.get_widget('dialog_confirm')
            self.dialog.set_title('')
            if parent is not None:
                self.dialog.set_transient_for(parent)
            self.xml.get_widget('button_cancel').grab_focus()
            self.xml.get_widget('label_ok').set_label(self.button_label)
 
            label = self.xml.get_widget('label_text')
            label.set_label('<span weight="bold" size="larger">'
                            '%s'
                            '</span>\n\n'
                            '<span weight="bold">%s</span>\n\n'
                            '%s' % (self.frob, self.handler.description(), self.handler.rationale))
    
        def run(self):
            ret = self.dialog.run()
            self.dialog.hide()
            return ret
    
    class Notification(RestrictedManagerCommon.Notification):
        def __init__(self, parent, title, text, action_text):
            RestrictedManagerCommon.Notification.__init__(self, parent, title,
                text, action_text)
            self.trayicon = gtk.status_icon_new_from_icon_name('restricted-manager')
            self.trayicon.connect('activate', self.open_manager)
            self.trayicon.set_tooltip(self.title)
    
            gobject.timeout_add (500, self.show_notify)
    
        def show_notify(self):
            pynotify.init('restricted-manager')
    
            notify = pynotify.Notification(self.title,
                '<span weight="bold">%s</span>\n\n%s' % (self.text,
                self.action_text), 'restricted-manager')
            notify.set_urgency(pynotify.URGENCY_NORMAL)
            notify.attach_to_status_icon(self.trayicon)
            notify.set_timeout(10000)
            notify.show()
    
        def open_manager(self, *args):
            self.parent.runme_as_root()
    
    class FWSelectDialog(RestrictedManagerCommon.FWSelectDialog):
        '''The purpose of this dialog is to show the user two options:
        - specify a file on current file system
        - choose a url where the firmware can be downloaded.'''

        class FWProgressDialog(RestrictedManagerCommon.FWSelectDialog.FWProgressDialog):
            '''Dialog wich is showed in case user selects an url,
            so that he knows how the download is going.'''
            def __init__(self, xml, parent = None):
                RestrictedManagerCommon.FWSelectDialog.FWProgressDialog.__init__(self)
                self.user_clicked_cancel = False

                self.dialog_progress = xml.get_widget('dialog_progress')

                if parent is not None:
                    self.dialog_progress.set_transient_for(parent)

                label_title = xml.get_widget('prog_label_title')
                label_title.set_label('<span weight="bold" size="larger">%s</span>' % (self.dlfw_text))
                self.label_text = xml.get_widget('prog_label_text')

                self.pbar = xml.get_widget('progressbar')
                self.pbar.set_fraction(0)

                self.button_cancel = xml.get_widget('button_cancel')
                self.button_cancel.connect('clicked', self.cancel_clicked, None)

                self.dialog_progress.show()

            def cancel_clicked(self, widget, data=None):
               self.dialog_progress.hide()
               self.user_clicked_cancel = True

            def dl_progress(self, blocks, block_size, total_size):
               '''callback for updating the progressbar'''
               if total_size > 0:
                   percentage = blocks / float(total_size / block_size + 1)
                   self.pbar.set_fraction(percentage)
               else:
                   self.pbar.pulse()

               while gtk.events_pending():
                   gtk.main_iteration()

               return self.user_clicked_cancel



        def __init__(self, FSText = None, UrlText = None, FWCheckCommand = None):
            RestrictedManagerCommon.FWSelectDialog.__init__(self, FWCheckCommand)
            try:
                xml = gtk.glade.XML('/usr/share/restricted-manager/fwhandler.glade')
            except RuntimeError:
                glade_filename = os.path.join(os.path.dirname(RestrictedManager.__file__), 'fwhandler.glade')
                xml = gtk.glade.XML(glade_filename)
    
            self.xml = xml
            self.dialog = xml.get_widget('dialog_bcm')
            self.dialog.set_title('')
            xml.get_widget('button_cancel').grab_focus()
    
            label_text = xml.get_widget('label_text')
            # translated strings coming from RestrictedManagerCommon
            label_text.set_label('<span weight="bold" size="larger">'
                                '%s'
                                '</span>\n\n'
                                '%s' % (self.label_title, self.label))
    
            # Browse files option
            # entry box
            self.entry_browse = gtk.Entry()
            if FSText is not None:
                self.entry_browse.set_text(FSText)
            # bottone 'browse'
            button_browse = gtk.Button(self.browse_text)
            # connect 'browse' button to file chooser
            button_browse.connect("clicked", self.browse_clicked, None)
    
            # create a pack under the radio button
            # so that we can put an entry and the 'browse' button
            hbox_browse = gtk.HBox(homogeneous=False, spacing=0)
            vbox2 = xml.get_widget('vbox2')
            vbox2.pack_start(hbox_browse, False, True, 0)
            hbox_browse.show()
            hbox_browse.pack_start(self.entry_browse, True, True, 0)
            self.entry_browse.show()
            hbox_browse.pack_start(button_browse, False, True, 0)
            button_browse.show()
            # connect radio button to the entry
            self.radio_browse = xml.get_widget('radio_browse')
            self.radio_browse.set_label(self.radio_local_text)
            self.radio_browse.connect('toggled', self.entry_toggle_sensitive, self.entry_browse)

            # URL option
            # FIXME: use comboboxentry
            # http://www.pygtk.org/pygtk2tutorial/sec-ComboBoxAndComboboxEntry.html
            self.entry_url = gtk.Entry()
            self.entry_url.set_sensitive(False)
            if UrlText is not None:
                self.entry_url.set_text(UrlText)
            # pack
            vbox4 = xml.get_widget('vbox4')
            vbox4.pack_start(self.entry_url, True, True, 0)
            self.entry_url.show()
            # connect radio button to the entry
            self.radio_url = xml.get_widget('radio_url')
            self.radio_url.set_label(self.radio_remote_text)
            self.radio_url.connect('toggled', self.entry_toggle_sensitive, self.entry_url)
    
            self.check_command = FWCheckCommand
    
        # only one entry box can be active
        def entry_toggle_sensitive(self, checkbutton, entry):
            entry.set_sensitive(checkbutton.get_active())
    
        # user clicked browse, so we open a file chooser dialog
        def browse_clicked(self, widget, data=None):
            chooser = gtk.FileChooserDialog(title=None,
                                            action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                            buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK)
                                            )
            response = chooser.run()
            chooser.hide()
    
            if response == gtk.RESPONSE_OK:
                self.file_chosen = chooser.get_filename()
                self.entry_browse.set_text(self.file_chosen)
            else:
                return
    
        def run(self):
            fw_location = ''
            fw_location_is_wrong = True
            ret = gtk.RESPONSE_OK
    
            # keep looping until we have a reasonable answer or cancel is pressed
            while fw_location_is_wrong and ret == gtk.RESPONSE_OK:
                ret = self.dialog.run()
                if ret == gtk.RESPONSE_OK:
    
                    if self.radio_browse.get_active():
                        # user selected from fs
                        fw_location = self.entry_browse.get_text()
                        # checking for URL & firmware
                        fw_location_is_wrong = self.check_firmware(fw_location)
                    else:
                        # user wants to dl the firmware
                        fw_location = self.entry_url.get_text()
                        try:
                            progress_subdialog = self.FWProgressDialog(self.xml, parent = self.dialog)
    
                            progress_subdialog.label_text.set_label('<i>%s %s</i>' % (self.downloading_text, fw_location))
                            result, get, headers = fwurlretrieve(fw_location, reporthook = progress_subdialog.dl_progress)
    
                            if result == True:
                                progress_subdialog.dialog_progress.hide()
                                fw_location_new = get
                                # checking for URL & firmware
                                if self.check_firmware(fw_location_new):
                                    # it is not a fw
                                    fw_location_is_wrong = True
                                else:
                                    fw_location_is_wrong = False
                                    fw_location = fw_location_new
                        except URLError:
                            progress_subdialog.dialog_progress.hide()
                            pass

                    if fw_location_is_wrong:
                        md = gtk.MessageDialog(type=gtk.MESSAGE_ERROR,
                                            buttons=gtk.BUTTONS_CLOSE,
                                            message_format=self.error_fw_invalid_text
                                            )
                        md.run()
                        md.hide()
                else:
                    ret = gtk.RESPONSE_CANCEL
            self.dialog.hide()
    
            if ret == gtk.RESPONSE_OK: response = True
            else: response = False
    
            return response, fw_location
    
