/* ------------------------------------------------------------------------
 * $Id: BSP.hh,v 1.3 2001/08/02 14:22:08 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-16 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _BSP_hh_
#define _BSP_hh_

// -- System Includes
#include <vector>

// -- 3Dwm Includes
#include "Celsius/Plane3D.hh"
#include "Celsius/Vector3D.hh"
#include "Celsius/Matrix3D.hh"
#include "Solid/Face.hh"
#include "Solid/BSPTreeBuilder.hh"

// -- Class Declarations

/**
 * General BSP node class. Abstract base class for the entire BSP node
 * class hierarchy. 
 **/
class BSP {
public:
    
    /**
     * Constructor.
     **/
    BSP() { }
    
    /**
     * Destructor.
     **/ 
    virtual ~BSP() { }

    /**
     * Clone the current BSP tree and return the resulting pointer.
     * This will perform a deep copy of the BSP tree, including all
     * plane and face data.
     *
     * @return pointer to a clone of the current BSP tree.
     **/
    virtual BSP *clone() const = 0;

    /**
     * Return the number of faces in this BSP node (including all
     * subtrees of the node).
     *
     * @return number of faces contained in the BSP node.
     **/
    virtual int getFaceCount() const = 0; 

    /**
     * Complement the current BSP tree. This will flip all boundary
     * partitioning planes as well as all triangle faces and leaves
     * will be inverted (so that inner cells become outer, and vice
     * versa).
     **/ 
    virtual void complement() = 0;

    /**
     * Perform an incremental CSG operation by combine the current BSP
     * tree with a set of triangle faces. The geometry data (vertices,
     * normals and texture coordinates) referenced in the list of
     * triangles should have been merged with the current BSP tree's
     * data prior to calling this method. This call will be propagated
     * through the BSP tree, splitting, pruning and adding faces as
     * necessary.
     *
     * @param t BSP container that holds geometry data for the tree.
     * @param faces list of triangles faces to combine with the tree.
     * @param op set operation to perform (union or intersection).
     **/
    virtual BSP *combine(BSPTree &t, const std::vector<TriangleFace> &faces,
			 BSPTreeBuilder::SetOperation op) = 0;
    
    /**
     * Retrieve the faces of the BSP tree (including all subtrees of
     * the node). This is how you convert from a BSP tree to a flat
     * boundary representation of the polyhedron that the BSP bounds.
     *
     * @return list of triangle faces contained in the BSP tree.
     **/ 
    virtual void buildFaceList(std::vector<TriangleFace> &faces) const = 0;

    /**
     * Decide whether a given point is inside the polyhedron
     * represented by the BSP tree or not. This will result in a
     * recursive traversal of the tree until a homogenous leaf has
     * been reached.
     *
     * @param p point to test for containment.
     * @return true if the point is inside the tree, false if not.
     **/
    virtual bool isContained(const Vector3D &p) const = 0;

    /**
     * Partition a list of faces by recursively partitioning the faces
     * against the partitioning planes of each BSP node and sorting
     * them in coincident, inside and outside lists.
     * 
     * @param t BSP tree container holding all tree-wide shared data.
     * @param faces list of triangle faces to split.
     * @param on_list list of faces coincident with the partitioning plane.
     * @param in_list list of inside faces.
     * @param out_list list of outside faces.
     **/
    virtual void partitionFaceList(BSPTree &t,
				   const std::vector<TriangleFace> &faces,
				   std::vector<TriangleFace> &on_list,
				   std::vector<TriangleFace> &in_list,
				   std::vector<TriangleFace> &out_list) const = 0; 
    
    /**
     * Transform the BSP tree with a given transformation matrix.
     * This will transform the partitioning planes in the BSP nodes.
     *
     * @param m transformation matrix.
     **/
    virtual void transform(const Matrix3D &inv_trafo) = 0;
    
private:
    
    // Don't use copy constructor, use the clone() method!
    BSP(const BSP &bsp);
};

/**
 * Internal BSP node class. Internal BSP nodes all have a binary
 * partitioning plane associated with them in addition to a list of
 * triangle faces coincident with the plane. Moreover, BSP nodes also
 * have two BSP subtrees, one representing all faces in front of the
 * partitioning plane, and one representing all faces behind it.
 **/
class BSPNode : public BSP {
public:

    /**
     * Constructor.
     *
     * @param bp binary partitioning plane to use.
     **/
    BSPNode(const Plane3D &bp);

    /**
     * Destructor.
     **/
    virtual ~BSPNode();

    /**
     * Retrieve a mutable list of faces for this node. This is NOT a
     * recursive call, and will only return the local face list.
     *
     * @return mutable list of triangle faces.
     **/ 
    std::vector<TriangleFace> &getFaces() {
	return _faces; 
    }
    
    /**
     * Retrieve a const list of faces for this node. This is NOT a
     * recursive call, and will only return the local face list.
     *
     * @return const list of triangle faces.
     **/ 
    const std::vector<TriangleFace> &getFaces() const {
	return _faces;
    }    

    /**
     * Set the front BSP subtree of the node. This subtree contains
     * all the faces of the geometry that are located in front of the
     * binary partitioning plane. 
     *
     * @param front pointer to the front BSP subtree.
     **/
    void setFront(BSP *front);

    /**
     * Set the back BSP subtree of the node. This subtree contains all
     * the faces of the geometry that are located behind the binary
     * partitioning plane.
     *
     * @param back pointer to the back BSP subtree.
     **/
    void setBack(BSP *back);

    /**
     * Clone the current BSP tree and return the resulting pointer.
     * This will perform a deep copy of the BSP tree, including all
     * plane and face data.
     *
     * @return pointer to a clone of the current BSP tree.
     **/
    virtual BSP *clone() const;

    /**
     * Retrieve the front subtree (if any).
     *
     * @return pointer to the front BSP subtree.
     **/
    virtual BSP *getFront() const {
	return _front; 
    }
    
    /**
     * Retrieve the back subtree (if any).
     *
     * @return pointer to the back BSP subtree.
     **/
    virtual BSP *getBack() const {
	return _back; 
    }

    /**
     * Return the number of faces in this BSP node (including all
     * subtrees of the node).
     *
     * @return number of faces contained in the BSP node.
     **/
    virtual int getFaceCount() const; 

    /**
     * Complement the current BSP tree. This will flip all boundary
     * partitioning planes as well as all triangle faces and leaves
     * will be inverted (so that inner cells become outer, and vice
     * versa).
     **/ 
    virtual void complement();

    /**
     * Perform an incremental CSG operation by combine the current BSP
     * tree with a set of triangle faces. The geometry data (vertices,
     * normals and texture coordinates) referenced in the list of
     * triangles should have been merged with the current BSP tree's
     * data prior to calling this method. This call will be propagated
     * through the BSP tree, splitting, pruning and adding faces as
     * necessary.
     *
     * @param t BSP container that holds geometry data for the tree.
     * @param faces list of triangles faces to combine with the tree.
     * @param op set operation to perform (union or intersection).
     **/
    virtual BSP *combine(BSPTree &t, const std::vector<TriangleFace> &faces, 
			 BSPTreeBuilder::SetOperation op);
    
    /**
     * Retrieve the faces of the BSP tree (including all subtrees of
     * the node). This is how you convert from a BSP tree to a flat
     * boundary representation of the polyhedron that the BSP bounds.
     *
     * @return list of triangle faces contained in the BSP tree.
     **/ 
    virtual void buildFaceList(std::vector<TriangleFace> &faces) const;

    /**
     * Decide whether a given point is inside the polyhedron
     * represented by the BSP tree or not. This will result in a
     * recursive traversal of the tree until a homogenous leaf has
     * been reached.
     *
     * @param p point to test for containment.
     * @return true if the point is inside the tree, false if not.
     **/
    virtual bool isContained(const Vector3D &p) const;

    /**
     * Partition a list of faces by recursively partitioning the faces
     * against the partitioning planes of each BSP node and sorting
     * them in coincident, inside and outside lists.
     * 
     * @param t BSP tree container holding all tree-wide shared data.
     * @param faces list of triangle faces to split.
     * @param on_list list of faces coincident with the partitioning plane.
     * @param in_list list of inside faces.
     * @param out_list list of outside faces.
     **/
    virtual void partitionFaceList(BSPTree &t,
				   const std::vector<TriangleFace> &faces,
				   std::vector<TriangleFace> &on_list,
				   std::vector<TriangleFace> &in_list,
				   std::vector<TriangleFace> &out_list) const;

    /**
     * Transform the BSP tree with a given transformation matrix.
     * This will transform the partitioning planes in the BSP nodes.
     *
     * @param m transformation matrix.
     **/
    virtual void transform(const Matrix3D &inv_trafo);
    
private:
    
    // Don't use copy constructor, use the clone() method!    
    BSPNode(const BSPNode &n);

    /**
     * Prune away unnecessary faces (and subfaces) when performing a
     * boolean set operation. This will partition the faces of this
     * BSP node using the passed BSP tree and discard all faces that
     * are unnecessary under the given set operation.
     *
     * @param t BSP tree container holding all tree-wide shared data.
     * @param bsp pointer to BSP tree root to use for partitioning.
     * @param op boolean set operation to perform.
     **/
    void pruneFaces(BSPTree &t, BSP *bsp, BSPTreeBuilder::SetOperation op);

    Plane3D _bp;
    Vector3D _point;
    BSP *_front, *_back;
    std::vector<TriangleFace> _faces;
};

/**
 * BSP leaf class. Leaves in a BSP represent a single, homogenous cell
 * that is either inside or outside of the polyhedron that the entire
 * BSP tree represents. 
 **/
class BSPLeaf : public BSP {
public:

    /**
     * Constructor.
     *
     * @param inside is this leaf inside? 
     **/ 
    BSPLeaf(bool inside) : _inside(inside) { }

    /**
     * Destructor.
     **/
    virtual ~BSPLeaf() { }

    /** 
     * Is this BSP leaf inside the polyhedron represented by the BSP
     * tree? If not, it is naturally outside the polyhedron.
     *
     * @return true if the leaf is inside the polyhedron.
     **/ 
    bool isInside() const {
	return _inside; 
    }

    /**
     * Clone the current BSP tree and return the resulting pointer.
     * This will perform a deep copy of the BSP tree, including all
     * plane and face data.
     *
     * @return pointer to a clone of the current BSP tree.
     **/
    virtual BSP *clone() const {
	return new BSPLeaf(_inside); 
    }

    /**
     * Return the number of faces in this BSP node (including all
     * subtrees of the node).
     *
     * @return number of faces contained in the BSP node.
     **/
    virtual int getFaceCount() const {
	return 0; 
    }
    
    /**
     * Complement the current BSP tree. This will flip all boundary
     * partitioning planes as well as all triangle faces and leaves
     * will be inverted (so that inner cells become outer, and vice
     * versa).
     **/ 
    virtual void complement() {
	_inside = !_inside;
    }

    /**
     * Perform an incremental CSG operation by combine the current BSP
     * tree with a set of triangle faces. The geometry data (vertices,
     * normals and texture coordinates) referenced in the list of
     * triangles should have been merged with the current BSP tree's
     * data prior to calling this method. This call will be propagated
     * through the BSP tree, splitting, pruning and adding faces as
     * necessary.
     *
     * @param t BSP container that holds geometry data for the tree.
     * @param faces list of triangles faces to combine with the tree.
     * @param op set operation to perform (union or intersection).
     **/
    virtual BSP *combine(BSPTree &t, const std::vector<TriangleFace> &faces, 
			 BSPTreeBuilder::SetOperation op);
    
    /**
     * Retrieve the faces of the BSP tree (including all subtrees of
     * the node). This is how you convert from a BSP tree to a flat
     * boundary representation of the polyhedron that the BSP bounds.
     *
     * @return list of triangle faces contained in the BSP tree.
     **/ 
    virtual void buildFaceList(std::vector<TriangleFace> &faces) const { }

    /**
     * Decide whether a given point is inside the polyhedron
     * represented by the BSP tree or not. This will result in a
     * recursive traversal of the tree until a homogenous leaf has
     * been reached.
     *
     * @param p point to test for containment.
     * @return true if the point is inside the tree, false if not.
     **/
    virtual bool isContained(const Vector3D &p) const {
	return _inside;
    }
    
    /**
     * Partition a list of faces by recursively partitioning the faces
     * against the partitioning planes of each BSP node and sorting
     * them in coincident, inside and outside lists.
     * 
     * @param t BSP tree container holding all tree-wide shared data.
     * @param faces list of triangle faces to split.
     * @param on_list list of faces coincident with the partitioning plane.
     * @param in_list list of inside faces.
     * @param out_list list of outside faces.
     **/
    virtual void partitionFaceList(BSPTree &t,
				   const std::vector<TriangleFace> &faces,
				   std::vector<TriangleFace> &on_list,
				   std::vector<TriangleFace> &in_list,
				   std::vector<TriangleFace> &out_list) const;

    /**
     * Transform the BSP tree with a given transformation matrix.
     * This will transform the partitioning planes in the BSP nodes.
     *
     * @param m transformation matrix.
     **/
    virtual void transform(const Matrix3D &inv_trafo) { }

private:

    // Don't use copy constructor, use the clone() method!    
    BSPLeaf(const BSPLeaf &n);

    bool _inside;
};

#endif /* BSP.hh */
