/* ------------------------------------------------------------------------
 * $Id: BSPTreeBuilder.hh,v 1.1 2001/07/27 14:50:15 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-17 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _BSPTreeBuilder_hh_
#define _BSPTreeBuilder_hh_

// -- System Includes
#include <vector>

// -- 3Dwm Includes
#include "Celsius/Plane3D.hh"
#include "Celsius/Vector2D.hh"
#include "Celsius/Vector3D.hh"
#include "Solid/Face.hh"

// -- Forward Declarations
class BSP;
class BSPTree;

// -- Class Declarations

/**
 * Binary space partitioning tree building class. Contains all the
 * various methods for constructing BSP trees out of boundary
 * representations of polyhedra. Also contains methods to carry out
 * boolean set operators on BSP trees.
 **/
class BSPTreeBuilder {
public:
    
    /**
     * Entity classification type enumeration. Used when classifying
     * points and polygons against a partitioning plane.
     **/
    typedef enum { 
	Coincident, InFrontOf, InBackOf
    } Classification;
    
    /**
     * Binary set operation type enumeration. The library supports
     * union and intersection set operations, but subtraction can be
     * performed by complementing the second set and then performing
     * an intersection between the first and the resulting set.
     **/
    typedef enum {
	Union, Intersection 
    } SetOperation;
    
    /**
     * Build a binary space partitioning (BSP) tree given a set of
     * face indices and the corresponding vertices. Additional
     * vertices may be generated during the tree building process, so
     * a deep copy is taken of the vertices and stored internally in
     * the tree builder. 
     *
     * @param faces list of triangle faces.
     * @param vertices list of vertices.
     * @param normals list of normals.
     * @param texCoords list of texture coordinates.
     * @return resulting BSP tree container object.
     **/
    static BSPTree *build(const std::vector<TriangleFace> &faces,
			  const std::vector<Vector3D> &vertices,
			  const std::vector<Vector3D> &normals,
			  const std::vector<Vector2D> &texCoords);
    
    /**
     * Combine two BSP trees using a boolean set operation. This can
     * either be a union (the two trees are merged) or an intersection
     * (the intersecting parts of the two trees, if any, are
     * retained). Subtraction can be performed by first complementing
     * the second tree, and then performing an intersection between
     * the first and the resulting trees. The existing trees are not
     * modified. 
     *
     * @param t1 first BSP tree operand.
     * @param t2 second BSP tree operand.
     * @param op set operation to perform when combining.
     * @return pointer to a new tree representing the combination.
     **/
    static BSPTree *combine(BSPTree *t1, BSPTree *t2, SetOperation op);

    /**
     * Complement a BSP tree. This means that all cells (BSP leaves)
     * and triangle faces will be flipped. The existing tree is not
     * modified.
     * 
     * @param tree source BSP tree.
     * @return pointer to a new tree representing the complement.
     **/
    static BSPTree *complement(BSPTree *tree);
    
    static void test();
    
    /**
     * Classify a point against a binary partitioning plane, thus
     * deciding whether the point is in front of, behind or coincident
     * with the partitioning. Used by the triangle classification
     * method. This method should be numerically robust, since it
     * assigns a specific thickness to the binary partitioning plane
     * (this thickness value can be changed).
     *
     * @param bp partitioning plane to use for classification.
     * @param p vertex to classify.
     * @return point classification flag.
     **/
    static Classification classifyPoint(const Plane3D &bp, const Vector3D &p);
    
    /**
     * Partition a triangle against a partitioning plane. Splitting a
     * triangle will result in one, two or possibly three resulting
     * triangles and a number of vertices. The passed face lists will
     * all be updated accordingly.
     * 
     * @param bp binary partitioning plane.
     * @param f triangle face to split.
     * @param on_list list of faces coincident with the partitioning plane.
     * @param front_list list of front faces.
     * @param back_list list of back faces.
     **/
    static void partitionTriangleFace(BSPTree &t, const Plane3D &bp,
				  const TriangleFace &f,
				  std::vector<TriangleFace> &on_list,
				  std::vector<TriangleFace> &front_list,
				  std::vector<TriangleFace> &back_list);

    /**
     * Partition a number of faces into coincident, front and back
     * face lists given a binary partitioning plane. TriangleFaces may be
     * split in the process.
     *
     * @param t BSP tree container holding all tree-wide shared data.
     * @param bp binary partitioning plane.
     * @param faces list of triangle faces to split.
     * @param on_list list of faces coincident with the partitioning plane.
     * @param front_list list of front faces.
     * @param back_list list of back faces.
     **/
    static void partition(BSPTree &t, const Plane3D &bp,
			  const std::vector<TriangleFace> &faces,
			  std::vector<TriangleFace> &on_list,
			  std::vector<TriangleFace> &front_list, 
			  std::vector<TriangleFace> &back_list);
    
    /**
     * Build a binary space partitioning tree from a set of faces and
     * vertices. This method is designed to call itself recursively on
     * each subproblem of the algorithm.
     *
     * @param t BSP tree container holding all tree-wide shared data.
     * @param faces list of triangle faces in this subproblem.
     * @param front is this in the front halfspace? 
     * @return pointer to a BSP tree representing the halfspace.
     **/
    static BSP *buildTree(BSPTree &t, const std::vector<TriangleFace> &faces,
			  bool front);
    
private:    
    static float _thickness;
};

#endif /* BSPTreeBuilder.hh */
