/* ------------------------------------------------------------------------
 * $Id: FunctionObject.hh,v 1.2 2001/06/20 12:32:11 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-17 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _FunctionObject_hh_
#define _FunctionObject_hh_

// -- System Includes
#include <vector>
#include <string>

// -- Zorn Includes
#include "Value.hh"

namespace Zorn {

    // -- Forward Declarations
    class Store;

    // -- Class Declarations

    /**
     * Value list type definition.
     **/
    typedef std::vector<Value> ValueList;

    /**
     * Native function object class. This class is used for mapping
     * Zorn functional calls to C++. To avoid lots of light-weight
     * objects, this class has been designed to be able to handle more
     * than one Zorn function. Since the same function object can map
     * to several Zorn functions, a simple reference counting scheme
     * has been implemented so that the function set may know when it
     * is legal to delete a specific function object.
     **/
    class FunctionObject {
    public:

	/**
	 * Constructor.
	 **/
	FunctionObject() : _refCount(0) { }

	/**
	 * Destructor.
	 **/
	virtual ~FunctionObject() { }
    
	/**
	 * Execute a Zorn function with a given name in the function
	 * object, passing it a vector if values and the memory store.
	 *
	 * @param id name of Zorn function to execute.
	 * @param vs value list to pass to the function.
	 * @param store memory store to use for memory access.
	 * @return the return value of the Zorn function.
	 **/
	virtual Value execute(const std::string &id,
			      const ValueList &vs, Store *store) = 0;
    
	/**
	 * Retrieve the parameter count of a Zorn function with a
	 * given name in the function object.
	 *
	 * @param id name of Zorn function to query.
	 * @return number of parameters to pass to the function.
	 **/
	virtual int getParamCount(const std::string &id) = 0;

	/**
	 * Retrieve the names of all the Zorn functions implemented by
	 * this function object.
	 *
	 * @return a vector of Zorn function names.
	 **/
	virtual std::vector<std::string> getFunctionNames() const = 0;

	/**
	 * Increment the reference count of this particlar function
	 * object instance.
	 **/
	void increment() { _refCount++; }

	/**
	 * Decrement the reference count of this particular function
	 * object instance.
	 **/
	void decrement() { _refCount--; }
    
	/**
	 * Retrieve the reference count of this function object
	 * instance.
	 *
	 * @return the reference count.
	 **/
	int getRefCount() const { return _refCount; }

    private:
    
	int _refCount;
    };

};

#endif /* FunctionObject.hh */
