# Part of the A-A-P recipe executive 

# Copyright (c) 2002-2003 Lars Ivar Igesund and stichting NLnet Labs
# Permission to copy and use this file is specified in the file COPYING.
# If this file is missing you can find it here: http://www.a-a-p.org/COPYING

#
# This module sets up variables and actions for using the DMD compiler tools.
#

from RecPython import *
import Global
from Action import action_add
from Dictlist import str2dictlist
from RecPos import RecPos


def exists():
    """
    Return TRUE when the DMD toolchain can be found.
    """
    if program_path("dmd"):
        if os.name == "nt":
            if program_path("link"):
                return program_path("lib")
            else:
                return FALSE
        else:
            return program_path("gcc")
    else:
        return FALSE


def define_actions():
    """
    Define the actions that DMD can accomplish.
    """
    rd = Global.globals
    # Compile one sourcefile at the time
    rpstack = [ RecPos("compile_dmd d action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "compile_dmd object,dllobject,libobject d"),
            "opt =\n"
            "@if _no.OPTIMIZE and int(_no.OPTIMIZE) > 0:\n"
            "  opt = -O\n"
            "dbg =\n"
            "DEBUG ?=\n"
            "@if _no.DEBUG == 'yes':\n"
            "  dbg = -g\n"
            ":sys $DMD $?DFLAGS $?DVERSION $opt $dbg $?DDEBUG $?DIMPORT -of$target -c $source")
    # Build a program from object files
    rpstack = [ RecPos("build_dmd action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "build_dmd object"),
            "cmd = \n"
            "@if os.name == \"nt\":\n"
            "   :progsearch LINK link\n"
            "   @exec \"import tools.dmd\"\n"
            "   DLINKFLAGS = `tools.dmd.find_phobos()` $?DLINKFLAGS \n"
            "   :sys $LINK /noi $source,$target,,$DLINKFLAGS\n"
            "@else:\n"
            "   :sys $DMD $?DLINKFLAGS -of$target $source")
    # Build a static lib from object files
    rpstack = [ RecPos("buildlib_dmd action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "buildlib_dmd libobject"),
            "@if os.name == \"nt\":\n"
            "   :progsearch LIB lib\n"
            "   :sys $LIB $?DLINKFLAGS -p512 -c $target $source\n"
            "@else:\n"
            "   :sys $DMD $?DLINKFLAGS -of$target $source")
    # Build a dll from object files  
    rpstack = [ RecPos("builddll_dmd action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "builddll_dmd dllobject"),
            "@exec \"import tools.dmd\"\n"
            "@if not tools.dmd.find_dll_main_object(source):\n"
            "   @f = file(\"aap_dllmain.d\", 'w')\n"
            "   @f.write(tools.dmd.dll_main_source())\n"
            "   @f.close()\n"
            "   :do compile {target = aap_dllmain.obj} aap_dllmain.d\n"
            "   source += aap_dllmain.obj\n"
            ":sys $DMD $?DLINKFLAGS -of$target $source\n"
            ":del {f} {q} aap_dllmain.*")
    # Build a program directly from source
    rpstack = [ RecPos("buildonestep_dmd d action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "buildonestep_dmd d"),
            "opt =\n"
            "@if _no.OPTIMIZE and int(_no.OPTIMIZE) > 0:\n"
            "  opt = -O\n"
            "dbg =\n"
            "DEBUG ?=\n"
            "@if _no.DEBUG == 'yes':\n"
            "  dbg = -g\n"
            ":sys $DMD $?DFLAGS $?DVERSION $opt $dbg $?DDEBUG $?DIMPORT"
            "       $?DLINKFLAGS -of$target $source\n"
            ":del {f} {q} *.obj")
    # Build a static lib directly from source
    rpstack = [ RecPos("buildlibonestep_dmd d action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "buildlibonestep_dmd d"),
            "opt =\n"
            "@if _no.OPTIMIZE and int(_no.OPTIMIZE) > 0:\n"
            "  opt = -O\n"
            "dbg =\n"
            "DEBUG ?=\n"
            "@if _no.DEBUG == 'yes':\n"
            "  dbg = -g\n"
            ":sys $DMD -c $?DFLAGS $?DVERSION $opt $dbg $?DDEBUG $?DIMPORT"
            "       -op $source\n"
            ":progsearch LIB lib\n"
            "BDIR = \n"
            "objects = `src2obj(source)`\n"
            ":sys $LIB $?DLINKFLAGS -p512 -c $target $objects\n"
            ":del {f} {q} *.obj")
    # Build a dll directly from source
    rpstack = [ RecPos("builddllonestep_dmd d action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "builddllonestep_dmd d"),
            "opt =\n"
            "@if _no.OPTIMIZE and int(_no.OPTIMIZE) > 0:\n"
            "  opt = -O\n"
            "dbg =\n"
            "DEBUG ?=\n"
            "@if _no.DEBUG == 'yes':\n"
            "  dbg = -g\n"
            "@exec \"import tools.dmd\"\n"
            "@if not tools.dmd.find_dll_main_source(source):\n"
            "   @f = file(\"aap_dllmain.d\", 'w')\n"
            "   @f.write(tools.dmd.dll_main_source())\n"
            "   @f.close()\n"
            "   source += aap_dllmain.d\n"
            ":sys $DMD $?DFLAGS $?DVERSION $opt $dbg $?DDEBUG $?DIMPORT"
            "       $?DLINKFLAGS -of$target $source\n"
            ":del {f} {q} aap_dllmain.d\n"
            ":del {q} *.obj")
        
    if not rd["_top"].get("DMD"):
        rd["_top"]["DMD"] = "dmd"


def use_actions(scope):
    """
    Setup variables so that the default actions use the DMD actions.
    """
    scope["D_COMPILE_ACTION"] = "compile_dmd"
    scope["D_BUILD_ACTION"] = "build_dmd"
    scope["D_BUILDLIB_ACTION"] = "buildlib_dmd"
    scope["D_BUILDDLL_ACTION"] = "builddll_dmd"
    scope["D_BUILDONESTEP_ACTION"] = "buildonestep_dmd"
    scope["D_BUILDDLLONESTEP_ACTION"] = "builddllonestep_dmd"
    scope["D_BUILDLIBONESTEP_ACTION"] = "buildlibonestep_dmd"

def find_phobos():
    dmd_path = program_path("dmd")
    import re
    phobos_path = re.sub('bin', 'lib', dmd_path)
    return re.sub('dmd.EXE', '', phobos_path)


def find_dll_main_source(sourcestr):
    import re
    m = re.compile(r"BOOL\s+DllMain\s*\(\s*HINSTANCE")
    for si in var2list(sourcestr):
        f = file2string(si)
        if m.search(f):
            return 1

    return None 

def find_dll_main_object(sourcestr):
    for si in var2list(sourcestr):
        f = file2string(si)
        from string import find
        if find(f, "_DllMain@12") > -1:
            return 1

    return None 

def dll_main_source():
    source = """
        import windows;

        HINSTANCE g_hInst;

        extern (C)
        {
            void gc_init();
            void gc_term();
            void _minit();
            void _moduleCtor();
        }

        extern (Windows)
        BOOL DllMain(HINSTANCE hInstance,
                     ULONG ulReason,
                     LPVOID pvReserved)
        {   
            switch (ulReason)
            {
                case DLL_PROCESS_ATTACH:
                gc_init();
                _minit();
                _moduleCtor();
                break;

                case DLL_PROCESS_DETACH:
                gc_term();
                break;

                case DLL_THREAD_ATTACH:
                case DLL_THREAD_DETACH:
                return false;
            }
            g_hInst = hInstance;
            return true;
        }
        """
    return source

# vim: set sw=4 et sts=4 tw=79 fo+=l:
