
//=============================================================================
/**
 *  @file    Cubit_Client.h
 *
 *  $Id: Cubit_Client.h 93650 2011-03-28 08:44:53Z johnnyw $
 *
 *    This class implements a simple "cube" CORBA client for the Cubit
 *    example using stubs generated by the TAO ORB IDL compiler.
 *
 *
 *  @author Aniruddha Gokhale
 *  @author Sumedh Mungee
 *  @author and Sergio Flores-Gaitan
 */
//=============================================================================


#ifndef _CUBIT_CLIENT_H
#define _CUBIT_CLIENT_H

#include "ace/Get_Opt.h"

#if !defined (ACE_LACKS_PRAGMA_ONCE)
# pragma once
#endif /* ACE_LACKS_PRAGMA_ONCE */

#include "ace/Profile_Timer.h"
#include "tao/ORB_Constants.h"
#include "cubitC.h"

/**
 * @class Cubit_Client
 *
 * @brief Defines a class that encapsulates behaviour of the Cubit client
 * example.  Provides a better understanding of the logic in an
 * object oriented way.
 *
 * This class declares an interface to run the example client for
 * Cubit CORBA server.  All the complexity for initializing the
 * server is hidden in the class.  Just the run() interface is needed.
 */
class Cubit_Client
{
public:
  // = Constructor and destructor.
  Cubit_Client (int shutdown = 0);

  ~Cubit_Client (void);

  /// Execute client example code.
  int run ();

  /// Initialize the client communication endpoint with server.
  /// <collocation_test_ior> is used to pass in the ior file name.
  int init (int argc, ACE_TCHAR **argv);

private:
  /// Simple function that returns the subtraction of 117 from the
  /// parameter.
  int func (u_int i);

  /// Function to read the cubit factory ior from a file.
  int read_ior (const ACE_TCHAR *filename);

  /// Parses the arguments passed on the command line.
  int parse_args (void);

  /// Obtain the bit mask representation of the specified test
  u_int opt_to_mask (const ACE_TCHAR *opt);

  /// Enable the specified test.
  void enable_test (u_int mask);

  /// Disable the specified test.
  void disable_test (u_int mask);

  /// Check if the specified test has been enabled.
  int check_enabled (u_int mask) const;

  /// Prints the time stats.
  void print_stats (const char *call_name,
                    ACE_Profile_Timer::ACE_Elapsed_Time &elapsed_time);

  /// Exercise the union.  Cube a union.
  void cube_union (void);

  /// calculate the cube from a struct.
  void cube_struct (int i);

  /// calculate the cube from a long.
  void cube_long (int i);

  /// Oneway operation test.
  void cube_oneway (int i);

  /// Twoway operation test.
  void cube_void (int i);

  /// Cube an octet.
  void cube_octet (int i);

  /// Cube a short.
  void cube_short (int i);

  /// Cube a sequence <l> is the length of the sequence.
  void cube_long_sequence (int i, int l);

  /// Cube a sequence of octets, <l> is the length of the sequence.
  void cube_octet_sequence (int i, int l);

  /// Cube a sequence of manys, <l> is the length of the sequence.
  void cube_many_sequence (int i, int l);

  /// Cube a sequence <l> is the length of the sequence.
  void cube_rti_data (int i, int numUpdates, int numAttrs);

  /// calculate the cube from a struct in an any.
  void cube_any_struct (int i);

  /// calculate the cube from a long in an any.
  void cube_any (int i);

  /// Invoke the method with <do_shutdown> != 0 to shutdown the server.
  int shutdown_server (int do_shutdown);

  /// # of arguments on the command line.
  int argc_;

  /// arguments from command line.
  ACE_TCHAR **argv_;

  /// Key of factory obj ref.
  ACE_TCHAR *cubit_factory_key_;

  /// Number of times to do the cube operations.
  u_int loop_count_;

  /// Flag to tell server to shutdown.
  int shutdown_;

  /// If set the test goes into quiet mode
  int quiet_;

  /// Remember our orb, must show up before the other objects to
  /// shutdown in the proper order.
  CORBA::ORB_var orb_;

  /// factory pointer for cubit.
  Cubit_Factory_var factory_;

  /// Cubit obj ref.
  Cubit_var cubit_;

  /// # of calls made to functions.
  u_int call_count_;

  /// # of errors incurred in the lifetime of the application.
  u_int error_count_;

  /// File from which to obtain the IOR.
  FILE *cubit_factory_ior_file_;

  /// File handle to read the IOR.
  ACE_HANDLE f_handle_;

  /// The number of bytes to send in the octet sequence test.
  u_int bytes_in_octet_sequence_;

  /// Bitset that determines which tests get run.
  u_int test_enable_bitset_;

  /// Bitset that determines which tests do not get run.
  u_int test_disable_bitset_;
};

#endif /* _CUBIT_CLIENT_H */
