/*
    ACfax - Fax reception with X11-interface for amateur radio
    Copyright (C) 1995-1998 Andreas Czechanowski, DL4SDC

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    andreas.czechanowski@ins.uni-stuttgart.de
*/
    
/*
 * acfax.c - the main program
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#ifdef USE_XAW3D
#include <X11/Xaw3d/ThreeD.h>
#include <X11/Xaw3d/Form.h>
#include <X11/Xaw3d/MenuButton.h>
#include <X11/Xaw3d/Toggle.h>
#include <X11/Xaw3d/Viewport.h>
#include <X11/Xaw3d/Box.h>
#include <X11/Xaw3d/SimpleMenu.h>
#include <X11/Xaw3d/SmeBSB.h>
#include <X11/Xaw3d/AsciiText.h>
#else
#include <X11/Xaw/Form.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/AsciiText.h>
#endif
#include "Canvas.h"
#include "global.h"
#include "mod_demod.h"
#include "sblaster.h"
#include "x_image.h"
#include "fax_funcs.h"
#include "widgets.h"

static char *fallbacks[] = {
  "*background: grey80",
  "*foreground: black",
  "*menuform*filebut.label: file-ops",
  "*menuform*filebut*file_mw.label: Image-file ops :",
  "*menuform*filebut*autosave.label: save (auto filename)",
  "*menuform*filebut*saveas.label: save as...",
  "*menuform*filebut*close.label: close savefile",
  "*menuform*filebut*quit.label: quit ACfax",
  "*optbut.label: Options...",
  "*modebut.label: Modes...",
/*  "*msg_shell*msg_ok.label: -OK-", */
  "*canvas.width: 1280",
  "*canvas.height: 1024",
  "*modemform*modemdesc.background: beige",
  "*modemform*modemdesc.label: modem settings",
  "*modemform*modinfo.background: beige",
  "*modemform*modinfo.label: modem\\ntype",
  "*modemform*filinfo.background: beige",
  "*modemform*filinfo.label: filter\\nselect",
  "*modemform*mflt1.label: narrow filter",
  "*modemform*mflt2.label: middle filter",
  "*modemform*mflt3.label: wide filter",
  "*modemform*fmdevinfo.background: beige",
  "*modemform*fmdevinfo.label: FM deviation (+/- Hz)",
  "*modemform*siginfo.background: beige",
  "*modemform*siginfo.label: Signal",
  "*faxctrls*runinfo.background: beige",
  "*faxctrls*runinfo.label: operation\\ncontrol",
  "*faxctrls*dirinfo.background: beige",
  "*faxctrls*dirinfo.label: writing\\ndirection",
  "*faxctrls*polinfo.background: beige",
  "*faxctrls*polinfo.label: image\\npolarity",
  "*faxctrls*phsinfo.background: beige",
  "*faxctrls*phsinfo.label: sync\\npolarity",
  "*faxctrls*adjinfo.background: beige",
  "*faxctrls*adjinfo.label: image\\nadjust",
  "*faxparams*numinfo.background: beige",
  "*faxparams*numinfo.label: numeric FAX-parameters",
  "*faxparams*statxt.label: APT start:",
  "*faxparams*stotxt.label: stop:",
  "*faxparams*lpmtxt.label: LPM:",
  "*faxparams*ioctxt.label: IOC:",
  NULL
};

struct prginfo {
  String callsign;
  String sizedesc;
} prg_info;

Display	*dpy;		/* the X-Display to use */
Screen	*scr;		/* screen of the display */
Pixmap	cpxmap;		/* The "backing store" for the canvas-widget */
GC	cgc;		/* GC for XPutImage from horimage/verimage to cpxmap */
/* XEvent	event;	     for XtAppNextEvent / XtDispatchEvent */
/* unsigned canwid;     width of canvas-widget */
/* unsigned canhei;     height of canvas-widget */
/* Colormap cmap, rootcmap;  colormap we use, and default-colormap */
/* unsigned coltab[256];     mapping table linear grayscale->pixel-value */
/* char	cused[256];	table indicating used pixel-values */
int	mode_lock = 0;  /* locks the operation control callback */
/* int	xpos, ypos;	position to fill with data */
/* int	ixoc;		number of pixels of one scan-line / PI */
/* int	right2left;	boolean indicating reverse line writing direction */
/* int	bot2top;	boolean indicating reverse line stacking direction */
/* int	mod_mode;	can be MOD_FM or MOD_AM */
/* int	fax_state;	one of FAX_APT, FAX_PHAS or FAX_RX */
/* int	max_val;	maximum level for demodulated output */
/* int	aptstart;	possible APT start values */
/* int	aptstop;	possible APT stop values */
int	save_width;	/* width of scan-line in pixels for saving */
struct fileshell_args	save_fsa; /* structure for file-save */
struct fileshell_args	load_fsa; /* structure for file-load */

XtAppContext	mainapp;
Widget	toplevel;
/*
Widget	last;		/ just for convenience.... /
Widget	form,		/ where to put it all in /
	vport,		/ a viewport to look thru onto the canvas /
	canvas;		/ widget holding picture using XImages /
*/
/*
Widget	menuform,
	filebut,	/ buttons in the top-line : "file" /
	modebut,	/ mode-select button /
	optbut;		/ option-select button /
Widget  file_mw,	/ popup-menu for "file" /
	file_saveas,	/ menu-item "save as" /
	file_close,	/ menu-item "close savefile" /
	file_autosave;	/ menu-item "save (auto filename) /
Widget  msg_shell,	/ popup-shell for aux. messages /
	msg_form,	/ form widget holding... /
	msg_text,	/ displayed text /
	msg_ok;		/ additional button (Ok/cancel/confirm/...) /
Widget  modemform,
	faxformA,
	faxformB;
Widget	modeminfo,
	mmode,
	fmdevinfo,
	fmdevval,
	fmdevtxt;
Widget	runinfo,
	rmstop,
	rmsyn,
	rmrun;
Widget	dirinfo,
	dirhv,
	dirlr,
	dirtb,
	polinfo,
	norinv;
Widget	phsinfo,
	posneg,
	statxt,
	staval,
	stotxt,
	stoval;
Widget	numinfo,
	lpmtxt,
	lpmval,
	ioctxt,
	iocval;
Widget	adjinfo,
	lshift,
	azimut;

*/

/*
Pixmap	px_horiz,
	px_vert,
	px_right,
	px_left,
	px_top,
	px_bottom,
	px_normal,
	px_invers,
	px_nphs,
	px_iphs,
	px_azimut,
	px_shift,
	px_stop,
	px_syn,
	px_run,
	px_mam,
	px_mfm,
	px_narrow,
	px_middle,
	px_wide;
*/

#define elofs(element) XtOffsetOf(struct prginfo, element)

static XtResource resources[] = {
  { "callsign", "Callsign", XtRString, sizeof(String),
    elofs(callsign), XtRImmediate, (XtPointer)"nocall" },
  { "size", "Size", XtRString, sizeof(String),
    elofs(sizedesc), XtRImmediate, (XtPointer)"double" },
};

static XrmOptionDescRec options[] = {
  { "-callsign", ".callsign", XrmoptionSepArg, (XtPointer)NULL },
  { "-size",     ".size",     XrmoptionSepArg, (XtPointer)NULL },
};

static  void usage(char *);
/* void	create_widgets(void); */
/* void	create_pixmaps(Widget); */
void	update_pxm(XImage *, int, int, int, int, unsigned, unsigned);
void	num_input(Widget, XEvent *, String *, Cardinal *);
void	show_values(void);
void	op_changed(int);
void	modmode_cb(Widget, XtPointer, XtPointer);
void	filtsel_cb(Widget, XtPointer, XtPointer);
void	stastop_cb(Widget, XtPointer, XtPointer);
void	dirhv_cb(Widget, XtPointer, XtPointer);
void	dirlr_cb(Widget, XtPointer, XtPointer);
void	dirtb_cb(Widget, XtPointer, XtPointer);
void	norinv_cb(Widget, XtPointer, XtPointer);
void	posneg_cb(Widget, XtPointer, XtPointer);
void	lshift_cb(Widget, XtPointer, XtPointer);
void	lshift_trigger(Widget, XtPointer, XtPointer);
void	azimut_cb(Widget, XtPointer, XtPointer);
void	azimut_trigger(Widget, XtPointer, XtPointer);
void	namesave_cb(Widget, XtPointer, XtPointer);
void	savefile_act(char *);
void	autosave_cb(Widget, XtPointer, XtPointer);
void	closesave_cb(Widget, XtPointer, XtPointer);
void	quit_cb(Widget, XtPointer, XtPointer);

XtTranslations txt_trans;
XtActionsRec acts[] = {
	{ "num-in", (XtActionProc)num_input },
  };

#ifndef HAVE_LTOA
/* simple, fast integer to string conversion */
char *ltoa(int n)
{
  static char res[16];
  int i, f[16];
  int d, p;

  d = 0;	/* 0 digits */
  p = 0;	/* 0 chars in string */
  /* allow to handle negative numbers */
  if (n < 0) {	
    res[p++] = '-';
  }

  /* calculate all the fractional parts */
  do {
    i = n / 10;
    f[d++] = n % 10;
    n = i;
  } while (i != 0);

  /* concatenate to string in reverse order */
  while (d) {
    res[p++] = f[--d] + '0';
  }

  /* terminate string */
  res[p] = '\0';

  return res;
}
#endif

static void usage(char *prgname)
{
  fprintf(stderr, "Usage : %s [Options]\n"
    "  Known options are :\n"
    "  -callsign callsign\n"
    "  -width {normal|double}\n", prgname);
}

/*
void create_pixmaps(Widget wid)
{
#include "bitmaps/horiz.bit"
#include "bitmaps/vert.bit"
#include "bitmaps/left.bit"
#include "bitmaps/right.bit"
#include "bitmaps/top.bit"
#include "bitmaps/bottom.bit"
#include "bitmaps/nphs.bit"
#include "bitmaps/iphs.bit"
#include "bitmaps/normal.bit"
#include "bitmaps/invers.bit"
#include "bitmaps/azimut.bit"
#include "bitmaps/shift.bit"
#include "bitmaps/stop.bit"
#include "bitmaps/syn.bit"
#include "bitmaps/run.bit"
#include "bitmaps/ammod.bit"
#include "bitmaps/fmmod.bit"
#include "bitmaps/narrow.bit"
#include "bitmaps/middle.bit"
#include "bitmaps/wide.bit"
  Window win;

  win = XtWindow(wid);
  px_horiz = XCreateBitmapFromData(dpy, win, horiz_bits, horiz_width, horiz_height);
  px_vert = XCreateBitmapFromData(dpy, win, vert_bits, vert_width, vert_height);
  px_right = XCreateBitmapFromData(dpy, win, right_bits, right_width, right_height);
  px_left = XCreateBitmapFromData(dpy, win, left_bits, left_width, left_height);
  px_top = XCreateBitmapFromData(dpy, win, top_bits, top_width, top_height);
  px_bottom = XCreateBitmapFromData(dpy, win, bottom_bits, bottom_width, bottom_height);
  px_nphs = XCreateBitmapFromData(dpy, win, nphs_bits, nphs_width, nphs_height);
  px_iphs = XCreateBitmapFromData(dpy, win, iphs_bits, iphs_width, iphs_height);
  px_normal = XCreateBitmapFromData(dpy, win, normal_bits, normal_width, normal_height);
  px_invers = XCreateBitmapFromData(dpy, win, invers_bits, invers_width, invers_height);
  px_azimut = XCreateBitmapFromData(dpy, win, azimut_bits, azimut_width, azimut_height);
  px_shift = XCreateBitmapFromData(dpy, win, shift_bits, shift_width, shift_height);
  px_stop = XCreateBitmapFromData(dpy, win, stop_bits, stop_width, stop_height);
  px_syn = XCreateBitmapFromData(dpy, win, syn_bits, syn_width, syn_height);
  px_run = XCreateBitmapFromData(dpy, win, run_bits, run_width, run_height);
  px_mam = XCreateBitmapFromData(dpy, win, ammod_bits, ammod_width, ammod_height);
  px_mfm = XCreateBitmapFromData(dpy, win, fmmod_bits, fmmod_width, fmmod_height);
  px_narrow = XCreateBitmapFromData(dpy, win, narrow_bits, narrow_width, narrow_height);
  px_middle = XCreateBitmapFromData(dpy, win, middle_bits, middle_width, middle_height);
  px_wide = XCreateBitmapFromData(dpy, win, wide_bits, wide_width, wide_height);

  XtVaSetValues(dirhv, XtNbitmap, px_horiz, NULL);
  XtVaSetValues(dirlr, XtNbitmap, px_right, NULL);
  XtVaSetValues(dirtb, XtNbitmap, px_bottom, NULL);
  XtVaSetValues(norinv, XtNbitmap, px_normal, NULL);
  XtVaSetValues(posneg, XtNbitmap, px_nphs, NULL);
  XtVaSetValues(lshift, XtNbitmap, px_shift, NULL);
  XtVaSetValues(azimut, XtNbitmap, px_azimut, NULL);
  XtVaSetValues(rmstop, XtNbitmap, px_stop, NULL);
  XtVaSetValues(rmsyn, XtNbitmap, px_syn, NULL);
  XtVaSetValues(rmrun, XtNbitmap, px_run, NULL);
  XtVaSetValues(mmode, XtNbitmap, px_mfm, NULL);
  XtVaSetValues(mfilter, XtNbitmap, px_middle, NULL);
}
*/

/* copy one of the XImages into the Pixmap of the canvas, and redisplay that
   portion of the image
*/
void update_pxm(XImage *img, int xs, int ys, int xd, int yd,
	unsigned wid, unsigned hei)
{
  XPutImage(dpy, cpxmap, cgc, img, xs, ys, xd, yd, wid, hei);
  canvasUpdateArea((CanvasWidget)canvas, xd, yd, wid, hei); 
}

/* the following makes the gcc happy not to complain about unused arguments */
/* ARGSUSED */
void num_input(Widget w, XEvent *event, String *par, Cardinal *npar)
{
  int val;
  String str;
  char ch[8];
  
  XtVaGetValues(w, XtNstring, &str, NULL);
  val = atoi(str);
  if (w == num_s.staval) {
    if (val > 500) val = 500;
    if (val < 100) val = 100;
    aptstart = val;
  } else if (w == num_s.stoval) {
    if (val > 500) val = 500;
    if (val < 100) val = 100;
    aptstop = val;
  } else if (w == num_s.lpmval) {
    setup_fax(val, 0, 0, w, 0, 0, 0, 0);
    val = lpm;
  } else if (w == num_s.iocval) {
    setup_fax(0, val, 0, w, 0, 0, 0, 0);
    val = ixoc;
  } else if (w == mod_s.fmdevval) {
    setup_fax(0, 0, 0, w, 0, 0, val, 0);
    val = devi;
  }
  sprintf(ch, "%d", val);
  XtVaSetValues(w, XtNstring, ch, NULL);
}

/* show the numeric fax-parameters and the FM-deviation of the modem */
void show_values(void)
{
  XtVaSetValues(mod_s.fmdevval, XtNstring, ltoa(devi), NULL);
  XtVaSetValues(num_s.lpmval, XtNstring, ltoa(lpm), NULL);
  XtVaSetValues(num_s.iocval, XtNstring, ltoa(ixoc), NULL);
  XtVaSetValues(num_s.staval, XtNstring, ltoa(aptstart), NULL);
  XtVaSetValues(num_s.stoval, XtNstring, ltoa(aptstop), NULL);
}

void op_changed(int mode)
{
  mode_lock = 1;
  switch(mode) {
    case FAX_APT:
	XtVaSetValues(run_s.rmstop, XtNstate, True, NULL);
	break;
    case FAX_PHAS:
	XtVaSetValues(run_s.rmsyn, XtNstate, True, NULL);
	break;
    case FAX_RX:
	XtVaSetValues(run_s.rmrun, XtNstate, True, NULL);
	break;
  }
  mode_lock = 0;
}

void modmode_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  st = (Boolean)((long)call_data);
  if (st) {
    setup_fax(0, 0, 0, w, 0, 0, 0, MOD_AM);
    XtVaSetValues(mod_s.mmode, XtNbitmap, px_mam, NULL);
  } else {
    setup_fax(0, 0, 0, w, 0, 0, 0, MOD_FM);
    XtVaSetValues(mod_s.mmode, XtNbitmap, px_mfm, NULL);
  }
}

void filtsel_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  int selectn;
  Pixmap pxm;

  selectn = (int)client_data;
  switch (selectn) {
    case FIL_NARR:
	pxm = px_narrow;
	break;
    case FIL_MIDL:
	pxm = px_middle;
	break;
    case FIL_WIDE:
	pxm = px_wide;
	break;
    default:
	return;
  }
  setup_fax(0, 0, 0, w, 0, 0, 0, selectn);
  XtVaSetValues(mod_s.mfilter, XtNbitmap, pxm, NULL);
}

void stastop_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;
  unsigned rd;

  rd = (unsigned)XawToggleGetCurrent(w);
  st = (Boolean)((long)call_data);
  fprintf(stderr, "rd = %u, st = %u\n", rd, (unsigned)st);
  if (rd == 0) {	/* all widgets of the radio-group are unset ? */
    /* XtVaSetValues(rmstop, XtNstate, True, NULL);  we may call ourself here ! */
    return;
  }
  if (!st) {
    return;		/* if unset, do not continue */
  }
  if (mode_lock) return;
  switch(rd) {
    case FAX_APT:
	fax_rx_stop(0);
	break;
    case FAX_PHAS:
	fax_rx_phase(0);
	break;
    case FAX_RX:
	fax_rx_start(0);
	break;
  }
}

void syn_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  st = (Boolean)((long)call_data);
  if (!st) {
    XtVaSetValues(w, XtNstate, True, NULL); /* we may call ourself here ! */
    return;		/* if unset, do not continue */
  }
  fax_rx_phase(0);
}

void run_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  st = (Boolean)((long)call_data);
  if (!st) {
    XtVaSetValues(w, XtNstate, True, NULL); /* we may call ourself here ! */
    return;		/* if unset, do not continue */
  }
  fax_rx_start(0);
}

void dirhv_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  XtVaGetValues(w, XtNstate, &st, NULL);
  if (st == True) {
    setup_fax(0, 0, FAX_VER, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_vert, NULL);
  } else {
    setup_fax(0, 0, FAX_HOR, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_horiz, NULL);
  }
}

void dirlr_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  XtVaGetValues(w, XtNstate, &st, NULL);
  if (st == True) {
    setup_fax(0, 0, FAX_RIG2LEF, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_left, NULL);
  } else {
    setup_fax(0, 0, FAX_LEF2RIG, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_right, NULL);
  }
}

void dirtb_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  XtVaGetValues(w, XtNstate, &st, NULL);
  if (st == True) {
    setup_fax(0, 0, FAX_BOT2TOP, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_top, NULL);
  } else {
    setup_fax(0, 0, FAX_TOP2BOT, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_bottom, NULL);
  }
}

void norinv_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  XtVaGetValues(w, XtNstate, &st, NULL);
  if (st == True) {
    setup_fax(0, 0, FAX_CINV, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_invers, NULL);
  } else {
    setup_fax(0, 0, FAX_CNOR, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_normal, NULL);
  }
}

void posneg_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean st;

  XtVaGetValues(w, XtNstate, &st, NULL);
  if (st == True) {
    setup_fax(0, 0, FAX_PWHT, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_iphs, NULL);
  } else {
    setup_fax(0, 0, FAX_PBLK, w, 0, 0, 0, 0);
    XtVaSetValues(w, XtNbitmap, px_nphs, NULL);
  }
}

void text_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  String s;

  XtVaGetValues(w, XtNstring, &s, NULL);
  fprintf(stderr, "text_cb: string = %s\n", s);
}

void show_coords(Widget w, XtPointer client_data, XtPointer call_data)
{
  XPoint *xp;
  if (call_data) {
    xp = (XPoint *) call_data;
    printf("coords = %d, %d\n", xp->x, xp->y);
  }
}

void lshift_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  XtVaSetValues(adj_s.lshift, XtNsensitive, False, NULL);
  XtVaSetValues(adj_s.azimut, XtNsensitive, False, NULL);
  XtVaSetValues(info_s.text, XtNlabel, "move mointer to sync position\n"
		"and press left mouse button.", NULL);
  XtVaSetValues(info_s.butn, XtNlabel, "Cancel", NULL);
  XawFormDoLayout(info_s.form, True);
  XtAddCallback(canvas, XtNcallback, lshift_trigger, (XtPointer) 1);
  XtAddCallback(info_s.butn, XtNcallback, lshift_trigger, (XtPointer) 0);
  XtPopup(info_s.shell, XtGrabNone);
}

void lshift_trigger(Widget w, XtPointer client_data, XtPointer call_data)
{
  XPoint *xp;

  if ((unsigned)client_data == 1) {
    if (call_data) {
      xp = (XPoint *) call_data;
      shift_fax_coords(xp->x, xp->y);
    }
  }
  XtRemoveAllCallbacks(canvas, XtNcallback);
  XtVaSetValues(adj_s.lshift, XtNsensitive, True, NULL);
  XtVaSetValues(adj_s.azimut, XtNsensitive, True, NULL);
  XtPopdown(info_s.shell);
}

void azimut_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  char message[256];
  Boolean vertmode;

  /* do not allow any further keypress on this button */
  XtVaSetValues(adj_s.azimut, XtNsensitive, False, NULL);
  XtVaSetValues(adj_s.lshift, XtNsensitive, False, NULL);
  XtVaGetValues(dir_s.dirhv, XtNstate, &vertmode, NULL);
  sprintf(message, "move mointer to the first point\n"
	"on a line that should be %s\n"
	"and press left mouse button.",
	(vertmode) ? "horizontal" : "vertical");
  XtVaSetValues(info_s.text, XtNlabel, message, NULL);
  XtVaSetValues(info_s.butn, XtNlabel, "Cancel", NULL);
  XawFormDoLayout(info_s.form, True);
  XtAddCallback(canvas, XtNcallback, azimut_trigger, (XtPointer) 1);
  XtAddCallback(info_s.butn, XtNcallback, azimut_trigger, (XtPointer) 0);
  XtPopup(info_s.shell, XtGrabNone);
}

void azimut_trigger(Widget w, XtPointer client_data, XtPointer call_data)
{
  XPoint *xp;
  static int xa, ya;

  if ((unsigned)client_data == 1) {
    /* first point, get position into xa, ya */
    if (call_data) {
      xp = (XPoint *) call_data;
      xa = xp->x;
      ya = xp->y;
      XtVaSetValues(info_s.text, XtNlabel, "now move pointer to the second point\n"
		"on that line and press left mouse button.", NULL);
      XtVaSetValues(info_s.butn, XtNlabel, "Cancel", NULL);
      /* XawFormDoLayout(info_s.form, True); */
      XtRemoveAllCallbacks(canvas, XtNcallback);
      XtAddCallback(canvas, XtNcallback, azimut_trigger, (XtPointer) 2);
      return;
    }
  } else if ((unsigned)client_data == 2) {
    /* second point, remove the menu from the screen */
    XtRemoveAllCallbacks(canvas, XtNcallback);
    if (call_data) {
      xp = (XPoint *) call_data;
      correct_fax_azimut(xp->x - xa, xp->y - ya);
    }
  }
  /* remove callbacks from canvas, re-activate azimut and lshift buttons,
     pop down the shell */
  XtRemoveAllCallbacks(canvas, XtNcallback);
  XtVaSetValues(adj_s.azimut, XtNsensitive, True, NULL);
  XtVaSetValues(adj_s.lshift, XtNsensitive, True, NULL);
  XtPopdown(info_s.shell);
}

void namesave_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  query_filename(&save_fsa);
}

void savefile_act(char *name)
{
  int ret;

  ret = save_faxfile(name, save_width);
  switch(ret) {
    case SAVE_BUSY:
	simple_info("Save-file already open !\nClose this file first");
	break;
    case SAVE_NPERM:
	simple_info("Cannot open save-file ?!");
	break;
    case SAVE_OK:
	break;
  }
}

void autosave_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  int ret;

  ret = save_faxfile(NULL, save_width);
  switch(ret) {
    case SAVE_BUSY:
	simple_info("Save-file already open !\nClose this file first");
	break;
    case SAVE_NPERM:
	simple_info("Cannot open save-file ?!");
	break;
    case SAVE_OK:
	break;
  }
}

void closesave_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  close_faxsave();
}

void quit_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
    exit(0);
}

void main(int argc, char **argv)
{
  int i;
  Pixel respix[20];
  unsigned nrespix;
  Dimension wid, hei;

  toplevel = XtAppInitialize(&mainapp, "ACfax",
		options, XtNumber(options),
		&argc, argv,
		fallbacks,
		NULL, 0); /* no arguments-list, 0 arguments */

  XtGetApplicationResources(toplevel, (XtPointer)&prg_info,
		resources, XtNumber(resources),
		NULL, 0);

  dpy = XtDisplay(toplevel);
  scr = XtScreen(toplevel);
  XtAppAddActions(mainapp, acts, XtNumber(acts));
  txt_trans = XtParseTranslationTable("#override \
		<Key>Return: num-in()\n\
		<Key>Linefeed: num-in()\n\
		Ctrl<Key>M: num-in()\n\
		Ctrl<Key>J: num-in()\n");

  /* cmap = DefaultColormap(dpy, scr); */
  /* create widgets */
  XtVaSetValues(toplevel, XtNtitle, "ACfax display", NULL);
  create_widgets(toplevel);
  XtRealizeWidget(toplevel);
  XtPopup(fax_shell, XtGrabNone);
  create_pixmaps(fax_form);

  fprintf(stderr,"allocing cmap\n"); fflush(stderr);
  for (i=0; i<20; i++) respix[i] = 0;
  respix[0] = BlackPixelOfScreen(scr);
  respix[1] = WhitePixelOfScreen(scr);
  XtVaGetValues(fax_form, XtNforeground, &respix[2],
	XtNbackground, &respix[3],
#ifdef USE_XAW3D
	XtNtopShadowPixel, &respix[4],
	XtNbottomShadowPixel, &respix[5],
#endif
	NULL);
  XtVaGetValues(mod_s.modemdesc, XtNbackground, &respix[6],
	NULL);
  for (i=0; i<13; i++)
    respix[i+7] = i;
  nrespix = 20;
  alloc_cmap(toplevel, respix, nrespix);

  update_area = update_pxm;
  mode_notify = op_changed;
  /* that's a hard one, but should set up all things right.... */
  XtVaGetValues(canvas, XtNwidth, &wid, XtNheight, &hei, NULL);
  setup_fax(120, 288, (FAX_CNOR | FAX_CUNFL | FAX_CROT0 |
	FAX_LEF2RIG | FAX_TOP2BOT | FAX_HOR | FAX_PBLK | FAX_GRAY),
	toplevel, (unsigned)wid, (unsigned)hei, 500, (MOD_FM | FIL_MIDL));
  aptstart = 300;
  aptstop = 450;
  show_values();

#if 0
  for (i=0; i<10; i++)
    printf("level = %d\n", signal_level());
#endif

  receive_on();
  fprintf(stderr,"filling cmap\n"); fflush(stderr);
  /* make the colormap active for our shells */
  XtVaSetValues(toplevel, XtNcolormap, icmap, NULL);
  XtVaSetValues(fax_shell, XtNcolormap, icmap, NULL);
  XtVaSetValues(info_s.shell, XtNcolormap, icmap, NULL);
  /* get the pixmap where we have to draw in, and its GC */
  cpxmap = canvasGetPixmap ((CanvasWidget)canvas, CanvasPicture);
  cgc = canvasGetGC ((CanvasWidget)canvas, CanvasDrawGC);

  fprintf(stderr,"initializing interface for FM reception...\n"); fflush(stderr);

/*
  comp_val = max_val >> 1;
  aptncmax = smpl_sec / 100;
  ixoc = 576;
  aptstart = 300;
  aptstop = 450;
  bot2top = 0;
  right2left = 0;
  core_start = core_dta;
  smpl_line = (2000 << 16) + 34800;
*/

  save_fsa.dir_io = malloc(256);
  *save_fsa.dir_io = '\0';
  save_fsa.title = "Save file";
  save_fsa.file_act = savefile_act;
  save_width = 1024;
  /* XtAddCallback(canvas, XtNcallback, show_coords, (XtPointer) NULL); */
  XtAddCallback(dir_s.dirhv, XtNcallback, dirhv_cb, (XtPointer) NULL);
  XtAddCallback(dir_s.dirlr, XtNcallback, dirlr_cb, (XtPointer) NULL);
  XtAddCallback(dir_s.dirtb, XtNcallback, dirtb_cb, (XtPointer) NULL);
  XtAddCallback(pol_s.norinv, XtNcallback, norinv_cb, (XtPointer) NULL);
  XtAddCallback(pol_s.posneg, XtNcallback, posneg_cb, (XtPointer) NULL);
  XtAddCallback(run_s.rmstop, XtNcallback, stastop_cb, (XtPointer) NULL);
  XtAddCallback(run_s.rmsyn, XtNcallback, stastop_cb, (XtPointer) NULL);
  XtAddCallback(run_s.rmrun, XtNcallback, stastop_cb, (XtPointer) NULL);
  XtAddCallback(adj_s.lshift, XtNcallback, lshift_cb, (XtPointer) NULL);
  XtAddCallback(adj_s.azimut, XtNcallback, azimut_cb, (XtPointer) NULL);
  XtAddCallback(mod_s.mmode, XtNcallback, modmode_cb, (XtPointer) NULL);
  XtAddCallback(file_m.autosave, XtNcallback, autosave_cb, (XtPointer) NULL);
  XtAddCallback(file_m.saveas,  XtNcallback, namesave_cb, (XtPointer) NULL);
  XtAddCallback(file_m.close, XtNcallback, closesave_cb, (XtPointer) NULL);
  XtAddCallback(file_m.quit, XtNcallback, quit_cb, (XtPointer) NULL);
  XtAddCallback(mod_s.mflt1,  XtNcallback, filtsel_cb, (XtPointer)FIL_NARR);
  XtAddCallback(mod_s.mflt2,  XtNcallback, filtsel_cb, (XtPointer)FIL_MIDL);
  XtAddCallback(mod_s.mflt3,  XtNcallback, filtsel_cb, (XtPointer)FIL_WIDE);

/*
  XtAddCallback(opt_start, XtNcallback, start_callback, (XtPointer) NULL);
  XtAddCallback(opt_stop, XtNcallback, stop_callback, (XtPointer) NULL);
*/
  XtAppMainLoop(mainapp);
}
