// -*- C++ -*- (c) 2008 Petr Rockai <me@mornfall.net>
#include <adept/util.h>
#include <wibble/regexp.h>
#include <adept/debconf.h>
#include <QCoreApplication>
#include <QLabel>
#include <QTimer>
#include <QScrollArea>
#include <KComboBox>
#include <QCheckBox>
#include <QResizeEvent>
#include <KPushButton>
#include <KLineEdit>
#include <KVBox>

#include <functional>

#ifndef ADEPT_DEBCONFGUI_H
#define ADEPT_DEBCONFGUI_H

namespace adept {

class DebconfGui;

class DebconfGui_Setter : public QObject {
    Q_OBJECT
    DebconfGui *gui;
    std::string key;
public:
    DebconfGui_Setter( DebconfGui *g, std::string k )
        : gui( g ), key( k )
    {
    }

public Q_SLOTS:
    void set( const QString &s );
    void set( bool v ) {
        set( QString( v ? "true" : "false" ) );
    }
};

class MultiSelectWidget : public KVBox {
    Q_OBJECT
    QList< QCheckBox * > m_boxes;
    QMap< QCheckBox *, QString > m_texts;
    typedef QList< QCheckBox * >::iterator Iterator;

public:
    MultiSelectWidget( QWidget *p = 0 ) : KVBox( p ) {}

    QString selection() {
        QString res;
        bool first = true;
        for ( Iterator i = m_boxes.begin(); i != m_boxes.end(); ++i ) {
            if ( (*i)->isChecked() ) {
                if ( !first )
                    res += ", ";
                first = false;
                res += m_texts[ *i ];
            }
        }
        return res;
    }

    void setItems( const QStringList &items ) {
        for ( QStringList::const_iterator i = items.begin();
              i != items.end(); ++i ) {
            QCheckBox * b = new QCheckBox( this );
            b->setText( *i );
            m_texts[ b ] = *i;
            connect( b, SIGNAL( toggled( bool ) ),
                     this, SLOT( selectionChanged() ) );
            m_boxes.push_back( b );
        }
    }

    void setSelection( const QStringList &sel ) {
        for  ( Iterator j = m_boxes.begin(); j != m_boxes.end(); ++j ) {
            (*j)->setChecked( false );
            for ( QStringList::const_iterator i = sel.begin();
                  i != sel.end(); ++i ) {
                if ( m_texts[ *j ] == *i )
                    (*j)->setChecked( true );
            }
        }
    }

public Q_SLOTS:
    void selectionChanged() {
        emit changed( selection() );
    }
Q_SIGNALS:
    void changed( const QString & );
};

class DebconfGui : public KVBox, public DebconfFrontend
{
    Q_OBJECT
    typedef DebconfGui_Setter Setter;
    QScrollArea *m_scroll;
    KVBox *m_widgets;
    QLabel *m_title;
    KPushButton *m_next, *m_back;
public:
    DebconfGui( QWidget *p ) : KVBox( p ) {
        m_title = new QLabel( this );
        m_title->setAlignment( Qt::AlignCenter | Qt::AlignVCenter );
        setMargin( 5 );
        m_scroll = new QScrollArea( this );
        m_scroll->setFrameStyle( QFrame::NoFrame );
        m_scroll->viewport()->setAutoFillBackground( false );
        m_widgets = 0;
        createButtons();
        clearWidgets();
    }

public Q_SLOTS:

    std::string property( std::string p, PropertyKey k ) {
        return DebconfFrontend::property( p, k );
    }

    void setTitle( const QString &str ) {
        m_title->setText( str );
    }

    void clearWidgets() {
        delete m_widgets;
        m_widgets = new KVBox();
        m_widgets->setSizePolicy(
            QSizePolicy( QSizePolicy::MinimumExpanding,
                         QSizePolicy::MinimumExpanding ) );
        m_widgets->setAutoFillBackground( false );
        m_scroll->setWidget( m_widgets );
    }

    void createButtons() {
        KHBox *box = new KHBox( this );
        new QWidget( box ); // spacer
        m_back = new KPushButton( box );
        m_next = new KPushButton( box );
        m_next->setText( "Next >" );
        m_back->setText( "Go Back" );
        m_next->setEnabled( false );
        m_back->setEnabled( false );
        connect( m_next, SIGNAL( clicked() ),
                 this, SLOT( next() ) );
        connect( m_back, SIGNAL( clicked() ),
                 this, SLOT( back() ) );
    }

    QWidget *createInputWidget( std::string k )
    {
        Setter *setter = new Setter( this, k );
        QWidget *widget = 0;

        if ( type( k ) == Select || type( k ) == MultiSelect ) {
            createDescriptionLabel( k );
            wibble::Splitter split( ", ", 0 );
            QStringList choices;
            std::transform( split.begin( property( k, Choices ) ),
                            split.end(), std::back_inserter( choices ),
                            std::ptr_fun( u8 ) );
                            
            if ( type( k ) == Select ) {
                KComboBox *combo = new KComboBox( m_widgets );
                combo->addItems( choices );
                combo->setCurrentIndex(
                    choices.indexOf( u8( value( k ) ) ) );
                connect( combo, SIGNAL( activated( const QString & ) ),
                         setter, SLOT( set( const QString & ) ) );
                widget = combo;
            }

            if ( type( k ) == MultiSelect ) {
                MultiSelectWidget *ms = new MultiSelectWidget( m_widgets );
                QStringList select;
                std::transform( split.begin( value( k ) ),
                                split.end(), std::back_inserter( select ),
                                std::ptr_fun( u8 ) );

                ms->setItems( choices );
                ms->setSelection( select );

                connect( ms, SIGNAL( changed( const QString & ) ),
                         setter, SLOT( set( const QString & ) ) );
                widget = ms;
            }
        }

        if ( type( k ) == String || type( k ) == Password ) {
            createDescriptionLabel( k );
            KLineEdit *edit = new KLineEdit( m_widgets );
            edit->setText( u8( value( k ) ) );
            if ( type( k ) == Password )
                edit->setEchoMode( QLineEdit::Password );
            connect( edit, SIGNAL( textChanged( const QString & ) ),
                     setter, SLOT( set( const QString & ) ) );
            widget = edit;
        }

        if ( type( k ) == Boolean ) {
            QCheckBox *check = new QCheckBox( m_widgets );
            check->setText( u8( property( k, Description ) ) );
            check->setChecked( value( k ) == "true" );
            connect( check, SIGNAL( toggled( bool ) ),
                     setter, SLOT( set( bool ) ) );
            widget = check;
        }

        if ( type( k ) != Note && type( k ) != Error && !widget ) {
            QLabel *err = new QLabel( m_widgets );
            err->setText(
                "<b>Not implemented</b>: The input widget for data"
                " type '" + u8( property( k, Type ) )
                + "' is not implemented. Will use default of '"
                + u8( value( k ) ) + "'." );
            err->setWordWrap( true );
            widget = err;
        }
        connect( widget, SIGNAL( destroyed() ),
                 setter, SLOT( deleteLater() ) );
        return widget;
    }

    void createDescriptionLabel( std::string k ) {
        QLabel *l = new QLabel( m_widgets );
        l->setText( u8( property( k, Description ) ) );
        l->setWordWrap( true );
    }

    void createWidget( std::string k ) {
        std::cerr << "creating widget for " << k << std::endl;

        QLabel *l = new QLabel( m_widgets );
        QString ext = u8( property( k, ExtendedDescription ) );
        QRegExp rx( "\\\\n" );
        ext.replace( rx, "\n" );
        l->setText( formatLongDescription( ext ) + "<br>" );
        l->setWordWrap( true );
        l->setAlignment( Qt::AlignJustify );

        createInputWidget( k );

    }

    virtual void cmd_go( std::string ) {
        setTitle( i18n( "<b>" ) + u8( title ) + i18n( "</b><br>" ) );
        std::cerr << "# GO GUI" << std::endl;
        assert( m_widgets );
        for ( std::set< std::string >::iterator i = input.begin();
              i != input.end(); ++ i )
        {
            createWidget( *i );
        }
        QWidget *spacer = new QWidget( m_widgets );
        spacer->setSizePolicy( QSizePolicy( QSizePolicy::Expanding,
                                            QSizePolicy::Expanding ) );
        spacer->setMinimumSize( 1, 1 );
        input.clear();
        m_scroll->setWidget( m_widgets );
        m_scroll->setWidgetResizable( true );
        m_next->setEnabled( true );
        m_back->setEnabled( backupEnabled );
        QTimer::singleShot( 0, this, SLOT( fixupScrollArea() ) );
        emit activated();
    }

    virtual void done() {
        emit deactivated();
    }

    virtual void cleanup() {
        clearWidgets();
        m_next->setEnabled( false );
        m_back->setEnabled( false );
    }

    void next() {
        DebconfFrontend::next();
    }

    void back() {
        DebconfFrontend::back();
    }

    virtual void fixupScrollArea() {
        int w = m_scroll->width();
        int h = m_widgets->heightForWidth( w );
        if ( w > 0 && h > 0 )
            m_widgets->setMaximumSize( w, h );
        m_scroll->setVerticalScrollBarPolicy( 
            ( h < m_scroll->height() ) ?
            Qt::ScrollBarAlwaysOff : Qt::ScrollBarAsNeeded );
    }

    virtual void resizeEvent( QResizeEvent *e ) {
        QWidget::resizeEvent( e );
        fixupScrollArea();
    }
Q_SIGNALS:
    void activated();
    void deactivated();
};

inline void DebconfGui_Setter::set( const QString &s ) {
    gui->values[ key ] = std::string( s.toLocal8Bit() );
}

}

#endif
