# Values associated with memory blocks.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/c/domain"

module AdLint #:nodoc:
module C #:nodoc:

  module BuggyValueSampler
    def unique_sample
      # NOTE: A value may have multiple sample values.
      #       So, callers of this method have potential bugs!
      to_enum.first
    end
  end

  # == DESCRIPTION
  # === Value class hierarchy
  #  Value
  #    <--- SingleValue
  #           <--- ScalarValue
  #           <--- ArrayValue
  #           <--- CompositeValue
  #    <--- MultipleValue
  #           <--- VersionedValue
  #    <--- FrozenValue
  class Value
    include BuggyValueSampler

    def scalar?
      subclass_responsibility
    end

    def array?
      subclass_responsibility
    end

    def composite?
      subclass_responsibility
    end

    def undefined?
      subclass_responsibility
    end

    def ambiguous?
      subclass_responsibility
    end

    def exist?
      subclass_responsibility
    end

    def multiple?
      subclass_responsibility
    end

    def overwrite!(value)
      subclass_responsibility
    end

    def narrow_domain!(operator_symbol, operand_value)
      subclass_responsibility
    end

    def widen_domain!(operator_symbol, operand_value)
      subclass_responsibility
    end

    def invert_domain!
      subclass_responsibility
    end

    def single_value_unified_with(rhs_value)
      subclass_responsibility
    end

    def ~
      subclass_responsibility
    end

    def +@
      subclass_responsibility
    end

    def -@
      subclass_responsibility
    end

    def +(rhs_value)
      subclass_responsibility
    end

    def -(rhs_value)
      subclass_responsibility
    end

    def *(rhs_value)
      subclass_responsibility
    end

    def /(rhs_value)
      subclass_responsibility
    end

    def %(rhs_value)
      subclass_responsibility
    end

    def &(rhs_value)
      subclass_responsibility
    end

    def |(rhs_value)
      subclass_responsibility
    end

    def ^(rhs_value)
      subclass_responsibility
    end

    def <<(rhs_value)
      subclass_responsibility
    end

    def >>(rhs_value)
      subclass_responsibility
    end

    def !
      subclass_responsibility
    end

    def <(rhs_value)
      subclass_responsibility
    end

    def >(rhs_value)
      subclass_responsibility
    end

    def ==(rhs_value)
      subclass_responsibility
    end

    def !=(rhs_value)
      subclass_responsibility
    end

    def <=(rhs_value)
      subclass_responsibility
    end

    def >=(rhs_value)
      subclass_responsibility
    end

    def logical_and(rhs_value)
      # NOTE: Operator && cannot be defined as a method in Ruby.
      subclass_responsibility
    end

    def logical_or(rhs_value)
      # NOTE: Operator || cannot be defined as a method in Ruby.
      subclass_responsibility
    end

    def must_be_equal_to?(value)
      subclass_responsibility
    end

    def may_be_equal_to?(value)
      subclass_responsibility
    end

    def must_not_be_equal_to?(value)
      subclass_responsibility
    end

    def may_not_be_equal_to?(value)
      subclass_responsibility
    end

    def must_be_less_than?(value)
      subclass_responsibility
    end

    def may_be_less_than?(value)
      subclass_responsibility
    end

    def must_be_greater_than?(value)
      subclass_responsibility
    end

    def may_be_greater_than?(value)
      subclass_responsibility
    end

    def must_be_undefined?
      subclass_responsibility
    end

    def may_be_undefined?
      subclass_responsibility
    end

    def must_be_true?
      subclass_responsibility
    end

    def may_be_true?
      subclass_responsibility
    end

    def must_be_false?
      subclass_responsibility
    end

    def may_be_false?
      subclass_responsibility
    end

    def coerce_to(type)
      subclass_responsibility
    end

    def to_enum
      subclass_responsibility
    end

    def to_single_value
      subclass_responsibility
    end

    def to_defined_value
      subclass_responsibility
    end

    def eql?(rhs_value)
      subclass_responsibility
    end

    def hash
      subclass_responsibility
    end

    def dup
      subclass_responsibility
    end
  end

  class SingleValue < Value
    def multiple?
      false
    end

    def must_be_undefined?
      self.undefined?
    end

    def may_be_undefined?
      # NOTE: SingleValue has exactly one value domain.
      #       So, the value of SingleValue may be undefined when the value
      #       must be undefined.
      self.must_be_undefined?
    end

    def to_single_value
      self
    end
  end

  class ScalarValue < SingleValue
    def self.of(numeric_or_range)
      case numeric_or_range
      when Numeric
        new(ValueDomain.equal_to(numeric_or_range))
      when Range
        lower = ValueDomain.greater_than_or_equal_to(numeric_or_range.first)
        higher = ValueDomain.less_than_or_equal_to(numeric_or_range.last)
        new(lower.intersection(higher))
      else
        raise TypeError, "argument must be a Numeric or a Range."
      end
    end

    def self.not_of(numeric_or_range)
      case numeric_or_range
      when Numeric
        new(ValueDomain.not_equal_to(numeric_or_range))
      when Range
        lower = ValueDomain.less_than(numeric_or_range.first)
        higher = ValueDomain.greater_than(numeric_or_range.last)
        new(lower.union(higher))
      else
        raise TypeError, "argument must be a Numeric or a Range."
      end
    end

    def self.of_true
      not_of(0)
    end

    def self.of_false
      of(0)
    end

    def self.of_arbitrary
      new(ValueDomain.of_unlimited)
    end

    def self.of_undefined(range)
      new(ValueDomain.of_undefined(range))
    end

    def self.of_nil
      new(ValueDomain.of_nil)
    end

    def self.of_nan
      new(ValueDomain.of_nan)
    end

    def initialize(domain)
      @domain = domain
    end

    def scalar?
      true
    end

    def array?
      false
    end

    def composite?
      false
    end

    def exist?
      !@domain.empty?
    end

    def undefined?
      @domain.undefined?
    end

    def ambiguous?
      @domain.ambiguous?
    end

    def overwrite!(value)
      case single_value = value.to_single_value
      when ScalarValue
        @domain = single_value.domain
      else
        raise TypeError, "cannot overwrite scalar with non-scalar."
      end
    end

    def narrow_domain!(operator_symbol, operand_value)
      case operand_single_value = operand_value.to_single_value
      when ScalarValue
        orig_domain = @domain
        @domain = @domain.narrow(operator_symbol, operand_single_value.domain)
        !@domain.equal?(orig_domain)
      else
        raise TypeError, "cannot narrow scalar value domain with non-scalar."
      end
    end

    def widen_domain!(operator_symbol, operand_value)
      case operand_single_value = operand_value.to_single_value
      when ScalarValue
        orig_domain = @domain
        @domain = @domain.widen(operator_symbol, operand_single_value.domain)
        !@domain.equal?(orig_domain)
      else
        raise TypeError, "cannot widen scalar value domain with non-scalar."
      end
    end

    def invert_domain!
      @domain = @domain.inversion
    end

    def single_value_unified_with(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain.union(rhs_single_value.domain))
      else
        raise TypeError, "cannot unify scalar value with non-scalar."
      end
    end

    def ~
      ScalarValue.new(~@domain)
    end

    def +@
      ScalarValue.new(+@domain)
    end

    def -@
      ScalarValue.new(-@domain)
    end

    def +(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain + rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def -(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain - rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def *(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain * rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def /(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain / rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def %(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain % rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def &(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain & rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def |(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain | rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def ^(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain ^ rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def <<(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain << rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def >>(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain >> rhs_single_value.domain)
      else
        raise TypeError, "binary operation between scalar and non-scalar."
      end
    end

    def !
      ScalarValue.new(!@domain)
    end

    def <(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain < rhs_single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def >(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain > rhs_single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def ==(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain == rhs_single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def !=(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain != rhs_single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def <=(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain <= rhs_single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def >=(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain >= rhs_single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def logical_and(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain.logical_and(rhs_single_value.domain))
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def logical_or(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ScalarValue
        ScalarValue.new(@domain.logical_or(rhs_single_value.domain))
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def must_be_equal_to?(value)
      case single_value = value.to_single_value.dup
      when ScalarValue
        comp_result = (self == single_value)
        single_value.invert_domain!
        single_value.narrow_domain!(:==, self)
        comp_result.domain.intersect?(ScalarValue.of_true.domain) &&
          !comp_result.domain.contain?(ScalarValue.of_false.domain) &&
          !@domain.intersect?(single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def may_be_equal_to?(value)
      case single_value = value.to_single_value
      when ScalarValue
        (self == single_value).domain.intersect?(ScalarValue.of_true.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def must_not_be_equal_to?(value)
      case single_value = value.to_single_value
      when ScalarValue
        comp_result = (self != single_value)
        comp_result.domain.intersect?(ScalarValue.of_true.domain) &&
          !comp_result.domain.contain?(ScalarValue.of_false.domain) &&
          !@domain.intersect?(single_value.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def may_not_be_equal_to?(value)
      case single_value = value.to_single_value
      when ScalarValue
        (self != single_value).domain.intersect?(ScalarValue.of_true.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def must_be_less_than?(value)
      case single_value = value.to_single_value
      when ScalarValue
        comp_result = (self < single_value)
        comp_result.domain.intersect?(ScalarValue.of_true.domain) &&
          !comp_result.domain.contain?(ScalarValue.of_false.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def may_be_less_than?(value)
      case single_value = value.to_single_value
      when ScalarValue
        (self < single_value).domain.intersect?(ScalarValue.of_true.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def must_be_greater_than?(value)
      case single_value = value.to_single_value
      when ScalarValue
        comp_result = (self > single_value)
        comp_result.domain.intersect?(ScalarValue.of_true.domain) &&
          !comp_result.domain.contain?(ScalarValue.of_false.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def may_be_greater_than?(value)
      case single_value = value.to_single_value
      when ScalarValue
        (self > single_value).domain.intersect?(ScalarValue.of_true.domain)
      else
        raise TypeError, "comparison between scalar and non-scalar."
      end
    end

    def must_be_true?
      self.may_be_equal_to?(ScalarValue.of_true) &&
        self.must_not_be_equal_to?(ScalarValue.of_false)
    end

    def may_be_true?
      self.may_be_equal_to?(ScalarValue.of_true)
    end

    def must_be_false?
      self.must_be_equal_to?(ScalarValue.of_false)
    end

    def may_be_false?
      self.may_be_equal_to?(ScalarValue.of_false)
    end

    def coerce_to(type)
      type.coerce_scalar_value(self)
    end

    def to_enum
      @domain.each_sample
    end

    def to_defined_value
      ScalarValue.new(@domain.to_defined_domain)
    end

    def eql?(rhs_value)
      rhs_value.kind_of?(ScalarValue) && @domain.eql?(rhs_value.domain)
    end

    def hash
      @domain.hash
    end

    def dup
      ScalarValue.new(@domain)
    end

    protected
    attr_reader :domain
  end

  class ArrayValue < SingleValue
    def initialize(values)
      @values = values
    end

    attr_reader :values

    def scalar?
      false
    end

    def array?
      true
    end

    def composite?
      false
    end

    def exist?
      @values.empty? ? true : @values.all? { |value| value.exist? }
    end

    def undefined?
      @values.empty? ? false : @values.all? { |value| value.undefined? }
    end

    def ambiguous?
      @values.empty? ? false : @values.all? { |value| value.ambiguous? }
    end

    def overwrite!(value)
      case single_value = value.to_single_value
      when ArrayValue
        @values.zip(single_value.values).each do |lhs, rhs|
          rhs && lhs.overwrite!(rhs)
        end
      else
        raise TypeError, "cannot overwrite array with non-array."
      end
    end

    def narrow_domain!(operator_symbol, operand_value)
      case operand_single_value = operand_value.to_single_value
      when ArrayValue
        @values.zip(operand_single_value.values).map { |lhs, rhs|
          if rhs
            lhs.narrow_domain!(operator_symbol, rhs)
          else
            next
          end
        }.any?
      else
        raise TypeError, "cannot narrow array value domain with non-array."
      end
    end

    def widen_domain!(operator_symbol, operand_value)
      case operand_single_value = operand_value.to_single_value
      when ArrayValue
        @values.zip(operand_single_value.values).map { |lhs, rhs|
          if rhs
            lhs.widen_domain!(operator_symbol, rhs)
          else
            next
          end
        }.any?
      else
        raise TypeError, "cannot widen array value domain with non-array."
      end
    end

    def invert_domain!
      @values.each { |value| value.invert_domain! }
    end

    def single_value_unified_with(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        ArrayValue.new(@values.zip(rhs_single_value.values).map { |lhs, rhs|
          lhs.single_value_unified_with(rhs)
        })
      else
        raise TypeError, "cannot unify array value with non-array."
      end
    end

    def ~
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def +@
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def -@
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def +(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def -(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def *(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def /(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def %(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def &(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def |(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def ^(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def <<(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def >>(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      dup # NOTREACHED
    end

    def !
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      ScalarValue.of_false # NOTREACHED
    end

    def <(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs < rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def >(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs > rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def ==(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs == rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def !=(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs != rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def <=(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs <= rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def >=(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs >= rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def logical_and(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs.logical_and(rhs))
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def logical_or(rhs_value)
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this comparison operator should not be reached.
      case rhs_single_value = rhs_value.to_single_value
      when ArrayValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs.logical_or(rhs))
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def must_be_equal_to?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self == single_value).must_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def may_be_equal_to?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self == single_value).may_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def must_not_be_equal_to?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self != single_value).must_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def may_not_be_equal_to?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self != single_value).may_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def must_be_less_than?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self < single_value).must_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def may_be_less_than?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self < single_value).may_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def must_be_greater_than?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self > single_value).must_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def may_be_greater_than?(value)
      case single_value = value.to_single_value
      when ArrayValue
        (self > single_value).may_be_true?
      else
        raise TypeError, "comparison between array and non-array."
      end
    end

    def must_be_true?
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this method should not be reached.
      @values.all? { |value| value.must_be_true? }
    end

    def may_be_true?
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this method should not be reached.
      @values.all? { |value| value.may_be_true? }
    end

    def must_be_false?
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this method should not be reached.
      @values.all? { |value| value.must_be_false? }
    end

    def may_be_false?
      # NOTE: When an array variable appears in expressions, object-specifier
      #       of an array variable should be evaluated into a pointer to the
      #       array body.
      #       So, this method should not be reached.
      @values.all? { |value| value.may_be_false? }
    end

    def coerce_to(type)
      type.coerce_array_value(self)
    end

    def to_enum
      # FIXME: This method generates only one of sample values.
      @values.map { |value| value.to_enum.first }
    end

    def to_defined_value
      ArrayValue.new(@values.map { |value| value.to_defined_value })
    end

    def eql?(rhs_value)
      rhs_value.kind_of?(ArrayValue) && @values.eql?(rhs_value.values)
    end

    def hash
      @values.hash
    end

    def dup
      ArrayValue.new(@values.map { |value| value.dup })
    end
  end

  class CompositeValue < SingleValue
    def initialize(values)
      @values = values
    end

    attr_reader :values

    def scalar?
      false
    end

    def array?
      false
    end

    def composite?
      true
    end

    def exist?
      @values.empty? ? true : @values.all? { |value| value.exist? }
    end

    def undefined?
      @values.empty? ? false : @values.all? { |value| value.undefined? }
    end

    def ambiguous?
      @values.empty? ? false : @values.all? { |value| value.ambiguous? }
    end

    def overwrite!(value)
      case single_value = value.to_single_value
      when CompositeValue
        @values.zip(single_value.values).each do |lhs, rhs|
          rhs && lhs.overwrite!(rhs)
        end
      else
        raise TypeError, "cannot overwrite composite with non-composite."
      end
    end

    def narrow_domain!(operator_symbol, operand_value)
      case operand_single_value = operand_value.to_single_value
      when CompositeValue
        @values.zip(operand_single_value.values).map { |lhs, rhs|
          if rhs
            lhs.narrow_domain!(operator_symbol, rhs)
          else
            next
          end
        }.any?
      else
        raise TypeError,
          "cannot narrow composite value domain with non-composite."
      end
    end

    def widen_domain!(operator_symbol, operand_value)
      case operand_single_value = operand_value.to_single_value
      when CompositeValue
        @values.zip(operand_single_value.values).map { |lhs, rhs|
          if rhs
            lhs.widen_domain!(operator_symbol, rhs)
          else
            next
          end
        }.any?
      else
        raise TypeError,
          "cannot widen composite value domain with non-composite."
      end
    end

    def invert_domain!
      @values.each { |value| value.invert_domain! }
    end

    def single_value_unified_with(rhs_value)
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        CompositeValue.new(
          @values.zip(rhs_single_value.values).map { |lhs, rhs|
            lhs.single_value_unified_with(rhs)
          })
      else
        raise TypeError, "cannot unify composite value with non-composite."
      end
    end

    def ~
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def +@
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def -@
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def +(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def -(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def *(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def /(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def %(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def &(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def |(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def ^(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def <<(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def >>(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      dup # NOTREACHED
    end

    def !
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      ScalarValue.of_false # NOTREACHED
    end

    def <(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs < rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def >(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs > rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def ==(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs == rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def !=(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs != rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def <=(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs <= rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def >=(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs >= rhs)
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def logical_and(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs.logical_and(rhs))
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def logical_or(rhs_value)
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      case rhs_single_value = rhs_value.to_single_value
      when CompositeValue
        if @values.size == rhs_single_value.values.size
          zipped = @values.zip(rhs_single_value.values)
          zipped.reduce(ScalarValue.of_nil) do |result, (lhs, rhs)|
            result.single_value_unified_with(lhs.logical_or(rhs))
          end
        else
          ScalarValue.of_false
        end
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def must_be_equal_to?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self == single_value).must_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def may_be_equal_to?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self == single_value).may_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def must_not_be_equal_to?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self != single_value).must_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def may_not_be_equal_to?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self != single_value).may_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def must_be_less_than?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self < single_value).must_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def may_be_less_than?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self < single_value).may_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def must_be_greater_than?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self > single_value).must_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def may_be_greater_than?(value)
      case single_value = value.to_single_value
      when CompositeValue
        (self > single_value).may_be_true?
      else
        raise TypeError, "comparison between composite and non-composite."
      end
    end

    def must_be_true?
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      @values.all? { |value| value.must_be_true? }
    end

    def may_be_true?
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      @values.all? { |value| value.may_be_true? }
    end

    def must_be_false?
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      @values.all? { |value| value.must_be_false? }
    end

    def may_be_false?
      # NOTE: A composite variable cannot appear in expressions except the
      #       primary-expression(object-specifier followed by `.').
      @values.all? { |value| value.may_be_false? }
    end

    def coerce_to(type)
      type.coerce_composite_value(self)
    end

    def to_enum
      # FIXME: This method generates only one of sample values.
      @values.map { |value| value.to_enum.first }
    end

    def to_defined_value
      CompositeValue.new(@values.map { |value| value.to_defined_value })
    end

    def eql?(rhs_value)
      rhs_value.kind_of?(CompositeValue) && @values.eql?(rhs_value.values)
    end

    def hash
      @values.hash
    end

    def dup
      CompositeValue.new(@values.map { |value| value.dup })
    end
  end

  class MultipleValue < Value
    def initialize(value, ancestor)
      @base_value = value.to_single_value
      @ancestor = ancestor
      @descendants = []
    end

    attr_reader :base_value

    extend Forwardable

    def_delegator :@base_value, :scalar?
    def_delegator :@base_value, :array?
    def_delegator :@base_value, :composite?

    def undefined?
      effective_values.all? { |multi_value| multi_value.base_value.undefined? }
    end

    def ambiguous?
      effective_values.all? { |multi_value| multi_value.base_value.ambiguous? }
    end

    def exist?
      effective_values.any? { |multi_value| multi_value.base_value.exist? }
    end

    def multiple?
      true
    end

    def overwrite!(value)
      single_value = value.to_single_value
      effective_values.each do |multi_value|
        multi_value.base_value.overwrite!(single_value)
      end
    end

    def narrow_domain!(operator_symbol, operand_value)
      operand_single_value = operand_value.to_single_value
      effective_values.map { |multi_value|
        if ancestor = multi_value.ancestor
          ancestor.base_value.narrow_domain!(operator_symbol.invert,
                                             operand_single_value)
        end
        multi_value.base_value.narrow_domain!(operator_symbol,
                                              operand_single_value)
      }.any?
    end

    def widen_domain!(operator_symbol, operand_value)
      operand_single_value = operand_value.to_single_value
      effective_values.map { |multi_value|
        if ancestor = multi_value.ancestor
          ancestor.base_value.narrow_domain!(operator_symbol.invert,
                                             operand_single_value)
        end
        multi_value.base_value.widen_domain!(operator_symbol,
                                             operand_single_value)
      }.any?
    end

    def invert_domain!
      effective_values.each do |multi_value|
        multi_value.base_value.invert_domain!
      end
    end

    def single_value_unified_with(rhs_value)
      to_single_value.single_value_unified_with(rhs_value)
    end

    def fork
      same_value = @descendants.find { |multi_value|
        multi_value.eql?(@base_value)
      }
      if same_value
        same_value
      else
        new_descendant = MultipleValue.new(@base_value.dup, self)
        @descendants.push(new_descendant)
        new_descendant
      end
    end

    def rollback!
      @descendants.pop
    end

    def delete_descendants!
      @descendants.clear
    end

    def ~
      ~to_single_value
    end

    def +@
      +to_single_value
    end

    def -@
      -to_single_value
    end

    def +(rhs_value)
      to_single_value + rhs_value.to_single_value
    end

    def -(rhs_value)
      to_single_value - rhs_value.to_single_value
    end

    def *(rhs_value)
      to_single_value * rhs_value.to_single_value
    end

    def /(rhs_value)
      to_single_value / rhs_value.to_single_value
    end

    def %(rhs_value)
      to_single_value % rhs_value.to_single_value
    end

    def &(rhs_value)
      to_single_value & rhs_value.to_single_value
    end

    def |(rhs_value)
      to_single_value | rhs_value.to_single_value
    end

    def ^(rhs_value)
      to_single_value ^ rhs_value.to_single_value
    end

    def <<(rhs_value)
      to_single_value << rhs_value.to_single_value
    end

    def >>(rhs_value)
      to_single_value >> rhs_value.to_single_value
    end

    def !
      !to_single_value
    end

    def <(rhs_value)
      to_single_value < rhs_value.to_single_value
    end

    def >(rhs_value)
      to_single_value > rhs_value.to_single_value
    end

    def ==(rhs_value)
      to_single_value == rhs_value.to_single_value
    end

    def !=(rhs_value)
      to_single_value != rhs_value.to_single_value
    end

    def <=(rhs_value)
      to_single_value <= rhs_value.to_single_value
    end

    def >=(rhs_value)
      to_single_value >= rhs_value.to_single_value
    end

    def logical_and(rhs_value)
      to_single_value.logical_and(rhs_value.to_single_value)
    end

    def logical_or(rhs_value)
      to_single_value.logical_or(rhs_value.to_single_value)
    end

    def must_be_equal_to?(value)
      single_value = value.to_single_value
      non_nil_values = effective_values.select { |multi_value|
        multi_value.base_value.exist?
      }
      return false if non_nil_values.empty?

      non_nil_values.all? do |multi_value|
        multi_value.base_value.must_be_equal_to?(single_value)
      end
    end

    def may_be_equal_to?(value)
      single_value = value.to_single_value
      effective_values.any? do |multi_value|
        multi_value.base_value.may_be_equal_to?(single_value)
      end
    end

    def must_not_be_equal_to?(value)
      single_value = value.to_single_value
      non_nil_values = effective_values.select { |multi_value|
        multi_value.base_value.exist?
      }
      return false if non_nil_values.empty?

      non_nil_values.all? do |multi_value|
        multi_value.base_value.must_not_be_equal_to?(single_value)
      end
    end

    def may_not_be_equal_to?(value)
      single_value = value.to_single_value
      effective_values.any? do |multi_value|
        multi_value.base_value.may_not_be_equal_to?(single_value)
      end
    end

    def must_be_less_than?(value)
      single_value = value.to_single_value
      non_nil_values = effective_values.select { |multi_value|
        multi_value.base_value.exist?
      }
      return false if non_nil_values.empty?

      non_nil_values.all? do |multi_value|
        multi_value.base_value.must_be_less_than?(single_value)
      end
    end

    def may_be_less_than?(value)
      single_value = value.to_single_value
      effective_values.any? do |multi_value|
        multi_value.base_value.may_be_less_than?(single_value)
      end
    end

    def must_be_greater_than?(value)
      single_value = value.to_single_value
      non_nil_values = effective_values.select { |multi_value|
        multi_value.base_value.exist?
      }
      return false if non_nil_values.empty?

      non_nil_values.all? do |multi_value|
        multi_value.base_value.must_be_greater_than?(single_value)
      end
    end

    def may_be_greater_than?(value)
      single_value = value.to_single_value
      effective_values.any? do |multi_value|
        multi_value.base_value.may_be_greater_than?(single_value)
      end
    end

    def must_be_undefined?
      effective_values.all? do |multi_value|
        multi_value.base_value.must_be_undefined?
      end
    end

    def may_be_undefined?
      effective_values.any? do |multi_value|
        multi_value.base_value.may_be_undefined?
      end
    end

    def must_be_true?
      non_nil_values = effective_values.select { |multi_value|
        multi_value.base_value.exist?
      }
      return false if non_nil_values.empty?

      non_nil_values.all? do |multi_value|
        multi_value.base_value.must_be_true?
      end
    end

    def may_be_true?
      effective_values.any? do |multi_value|
        multi_value.base_value.may_be_true?
      end
    end

    def must_be_false?
      non_nil_values = effective_values.select { |multi_value|
        multi_value.base_value.exist?
      }
      return false if non_nil_values.empty?

      non_nil_values.all? do |multi_value|
        multi_value.base_value.must_be_false?
      end
    end

    def may_be_false?
      effective_values.any? do |multi_value|
        multi_value.base_value.may_be_false?
      end
    end

    def coerce_to(type)
      MultipleValue.new(to_single_value.coerce_to(type), nil)
    end

    def to_enum
      to_single_value.to_enum
    end

    def to_single_value
      # NOTE: The base_value of the MultipleValue object must be a SingleValue.
      effective_values.map { |multi_value|
        multi_value.base_value
      }.reduce do |unified_value, single_value|
        unified_value.single_value_unified_with(single_value)
      end
    end

    def to_defined_value
      to_single_value.to_defined_value
    end

    def eql?(rhs_value)
      to_single_value.eql?(rhs_value.to_single_value)
    end

    def hash
      to_single_value.hash
    end

    def dup
      MultipleValue.new(to_single_value.dup, nil)
    end

    def effective_values
      @descendants.empty? ? [self] : @descendants
    end

    def descendants
      if @descendants.empty?
        [self]
      else
        @descendants.map { |multi_value| multi_value.descendants }.flatten.uniq
      end
    end

    protected
    attr_reader :ancestor
  end

  class VersionedValue < MultipleValue
    def initialize(original_value)
      # NOTE: `original_value.to_single_value' will be done in
      #       MultipleValue#initialize.
      super(original_value, nil)

      @version_controller = ValueVersionController.new(self)
    end

    def enter_versioning_group
      @version_controller.enter_new_versioning_group
    end

    def leave_versioning_group(raise_complement)
      @version_controller.copy_current_version if raise_complement
      @version_controller.merge_forked_versions
      @version_controller.leave_current_versioning_group
      invalidate_memo!
    end

    def begin_versioning
      @version_controller.begin_forking
    end

    def end_versioning
      @version_controller.end_forking
      invalidate_memo!
    end

    def rollback_latest_version!
      if @version_controller.rollback
        invalidate_memo!
      end
    end

    def rollback_all_versions!
      delete_descendants!
      _orig_overwrite!(@version_controller.original_value)
      @version_controller = ValueVersionController.new(self)
      invalidate_memo!
    end

    alias :_orig_overwrite! :overwrite!

    def overwrite!(value)
      @version_controller.fork_current_version
      super(value.to_single_value)
      @version_controller.mark_current_versioning_group_as_sticky
      invalidate_memo!
    end

    def narrow_domain!(operator_symbol, operand_value)
      @version_controller.fork_current_version
      super
      invalidate_memo!
    end

    def widen_domain!(operator_symbol, operand_value)
      @version_controller.fork_current_version
      super
      invalidate_memo!
    end

    def invert_domain!
      @version_controller.fork_current_version
      super
      invalidate_memo!
    end

    def coerce_to(type)
      VersionedValue.new(to_single_value.coerce_to(type))
    end

    def effective_values
      @version_controller.current_values
    end

    extend Memoizable

    memoize :to_single_value

    def invalidate_memo!
      forget_to_single_value_memo
    end

    private
    def compact_descendants!
      @descendants = @version_controller.current_values.reject { |multi_value|
        multi_value.equal?(self)
      }.uniq
    end
  end

  class ValueVersionController
    def initialize(original_value)
      @versioning_group_stack = [RootVersioningGroup.new(original_value)]
    end

    def original_value
      root_versioning_group.initial_values.first
    end

    def current_values
      current_versioning_group.current_values
    end

    def enter_new_versioning_group
      new_group = VersioningGroup.new(current_versioning_group.current_version)
      @versioning_group_stack.push(new_group)
    end

    def leave_current_versioning_group
      @versioning_group_stack.pop
    end

    def begin_forking
      current_versioning_group.begin_new_version
    end

    def end_forking
      current_versioning_group.end_current_version
    end

    def fork_current_version
      fork_all_versions
    end

    def rollback
      return false unless current_version.forked?

      # NOTE: This method must be called in the forking section.
      current_versioning_group.delete_current_version
      current_versioning_group.base_values.each do |multi_value|
        multi_value.rollback!
      end
      begin_forking
      mark_current_versioning_group_as_sticky

      true
    end

    def mark_current_versioning_group_as_sticky
      @versioning_group_stack.reverse_each do |group|
        if group.sticky?
          break
        else
          group.sticky = true
        end
      end
    end

    def copy_current_version
      return unless current_versioning_group.sticky?

      # NOTE: This method must be called between ending of the forking section
      #       and ending of the versioning group.
      current_values.each do |multi_value|
        base_value = multi_value.base_value
        already_exist = multi_value.descendants.any? { |desc_multi_value|
          !desc_multi_value.equal?(multi_value) &&
            desc_multi_value.eql?(base_value)
        }
        multi_value.fork unless already_exist
      end
    end

    def merge_forked_versions
      # NOTE: This method must be called between ending of the forking section
      #       and ending of the versioning group.
      base_version = current_versioning_group.base_version

      case
      when current_versioning_group.sticky?
        fork_all_versions
        base_version.values = base_version.values.map { |base_multi_value|
          base_multi_value.descendants
        }.flatten.uniq
      when current_versioning_group.versions_forked?
        # NOTE: When versions in the current versioning group have been forked,
        #       base_version of the current versioning group has already been
        #       forked.  So, it is safe to overwrite the base_version's values
        #       with the current versioning group's initial snapshot values.
        initial_values = current_versioning_group.initial_values
        values = base_version.values.zip(initial_values)
        values.each do |base_multi_value, initial_single_value|
          base_multi_value.delete_descendants!
          if base_multi_value.kind_of?(VersionedValue)
            base_multi_value._orig_overwrite!(initial_single_value)
          else
            base_multi_value.overwrite!(initial_single_value)
          end
        end
      else
        # NOTE: Nothing to be done when base_version of the current versioning
        #       group has not been forked.
      end
    end

    private
    def fork_all_versions
      parent_group = root_versioning_group
      active_versioning_groups.each do |active_group|
        active_group.base_version = parent_group.current_version
        active_group.fork_all_versions
        parent_group = active_group
      end
    end

    def root_versioning_group
      @versioning_group_stack.first
    end

    def active_versioning_groups
      @versioning_group_stack[1..-1]
    end

    def current_versioning_group
      @versioning_group_stack.last
    end

    def current_version
      current_versioning_group.current_version
    end

    class VersioningGroup
      def initialize(base_version, sticky = false)
        @base_version = base_version
        @sticky = sticky

        @initial_values = base_version.values.map { |multi_value|
          multi_value.base_value.dup
        }
        @version_stack = []
        @all_versions = []
      end

      attr_accessor :base_version
      attr_writer :sticky
      attr_reader :initial_values

      def sticky?
        @sticky
      end

      def versions_forked?
        @all_versions.any? { |version| version.forked? }
      end

      def base_values
        @base_version.values
      end

      def current_version
        @version_stack.empty? ? @base_version : @version_stack.last
      end

      def current_values
        current_version.values
      end

      def begin_new_version
        new_version = Version.new(base_values)
        @version_stack.push(new_version)
        @all_versions.push(new_version)
      end

      def end_current_version
        @version_stack.pop
      end

      def delete_current_version
        end_current_version
        @all_versions.pop
      end

      def fork_all_versions
        @all_versions.each { |version| version.fork_from(@base_version) }
      end
    end
    private_constant :VersioningGroup

    class RootVersioningGroup < VersioningGroup
      def initialize(original_value)
        super(Version.new([original_value], true), true)
      end
    end
    private_constant :RootVersioningGroup

    class Version
      def initialize(values, original = false)
        @values = values
        @state = original ? :original : :forking
      end

      attr_accessor :values

      def original?
        @state == :original
      end

      def forking?
        @state == :forking
      end

      def forked?
        @state == :forked
      end

      def fork_from(base_version)
        if forking?
          @values = base_version.values.map { |multi_value| multi_value.fork }
          @state = :forked
        end
      end
    end
    private_constant :Version
  end

  class FrozenValue < Value
    def initialize(versioned_value)
      @versioned_value = versioned_value
    end

    extend Forwardable

    def_delegator :@versioned_value, :scalar?
    def_delegator :@versioned_value, :array?
    def_delegator :@versioned_value, :composite?
    def_delegator :@versioned_value, :undefined?
    def_delegator :@versioned_value, :exist?
    def_delegator :@versioned_value, :multiple?
    def_delegator :@versioned_value, :enter_versioning_group
    def_delegator :@versioned_value, :leave_versioning_group
    def_delegator :@versioned_value, :begin_versioning
    def_delegator :@versioned_value, :end_versioning
    def_delegator :@versioned_value, :rollback_latest_version!
    def_delegator :@versioned_value, :rollback_all_versions!
    def_delegator :@versioned_value, :single_value_unified_with
    def_delegator :@versioned_value, :~
    def_delegator :@versioned_value, :+@
    def_delegator :@versioned_value, :-@
    def_delegator :@versioned_value, :+
    def_delegator :@versioned_value, :-
    def_delegator :@versioned_value, :*
    def_delegator :@versioned_value, :/
    def_delegator :@versioned_value, :%
    def_delegator :@versioned_value, :&
    def_delegator :@versioned_value, :|
    def_delegator :@versioned_value, :^
    def_delegator :@versioned_value, :<<
    def_delegator :@versioned_value, :>>
    def_delegator :@versioned_value, :!
    def_delegator :@versioned_value, :<
    def_delegator :@versioned_value, :>
    def_delegator :@versioned_value, :==
    def_delegator :@versioned_value, :!=
    def_delegator :@versioned_value, :<=
    def_delegator :@versioned_value, :>=
    def_delegator :@versioned_value, :logical_and
    def_delegator :@versioned_value, :logical_or
    def_delegator :@versioned_value, :must_be_equal_to?
    def_delegator :@versioned_value, :may_be_equal_to?
    def_delegator :@versioned_value, :must_not_be_equal_to?
    def_delegator :@versioned_value, :may_not_be_equal_to?
    def_delegator :@versioned_value, :must_be_less_than?
    def_delegator :@versioned_value, :may_be_less_than?
    def_delegator :@versioned_value, :must_be_greater_than?
    def_delegator :@versioned_value, :may_be_greater_than?
    def_delegator :@versioned_value, :must_be_undefined?
    def_delegator :@versioned_value, :may_be_undefined?
    def_delegator :@versioned_value, :must_be_true?
    def_delegator :@versioned_value, :may_be_true?
    def_delegator :@versioned_value, :must_be_false?
    def_delegator :@versioned_value, :may_be_false?
    def_delegator :@versioned_value, :coerce_to
    def_delegator :@versioned_value, :to_enum
    def_delegator :@versioned_value, :to_single_value
    def_delegator :@versioned_value, :to_defined_value
    def_delegator :@versioned_value, :eql?
    def_delegator :@versioned_value, :hash
    def_delegator :@versioned_value, :dup
  end

end
end
