package testutils

import (
	"bytes"
	"io/fs"
	"os"
	"path/filepath"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/termie/go-shutil"
)

// CompareTreesWithFiltering allows comparing a goldPath directory to p. Those can be updated via the dedicated flag.
//  It will filter dconf database and not commit it in the new golden directory.
func CompareTreesWithFiltering(t *testing.T, p, goldPath string, update bool) {
	t.Helper()

	// Update golden file
	if update {
		t.Logf("updating golden file %s", goldPath)
		require.NoError(t, os.RemoveAll(goldPath), "Cannot remove target golden directory")
		// Filter dconf generated DB files that are machine dependent
		require.NoError(t,
			shutil.CopyTree(
				p, goldPath,
				&shutil.CopyTreeOptions{Symlinks: true, Ignore: ignoreDconfDB, CopyFunction: shutil.Copy}),
			"Can’t update golden directory")
	}

	gotContent, err := treeContent(t, p, []byte("GVariant"))
	if err != nil {
		t.Fatalf("No generated content: %v", err)
	}
	goldContent, err := treeContent(t, goldPath, nil)
	if err != nil {
		t.Fatalf("No golden directory found: %v", err)
	}
	assert.Equal(t, goldContent, gotContent, "got and expected content differs")

	// Verify that each <DB>.d has a corresponding gvariant db generated by dconf update
	// search for dconfDir
	dconfDir := p
	err = filepath.WalkDir(dconfDir, func(p string, info fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if !info.IsDir() {
			return nil
		}
		if info.Name() == "db" {
			dconfDir = filepath.Dir(p)
		}
		return nil
	})
	require.NoError(t, err, "can't find dconf directory")

	dbs, err := filepath.Glob(filepath.Join(dconfDir, "db", "*.d"))
	require.NoError(t, err, "Checking pattern for dconf db failed")
	for _, db := range dbs {
		_, err = os.Stat(strings.TrimSuffix(db, ".db"))
		assert.NoError(t, err, "Binary version of dconf DB should exists")
	}
}

// treeContent builds a recursive file list of dir with their content
// It can ignore files starting with ignoreHeaders.
func treeContent(t *testing.T, dir string, ignoreHeaders []byte) (map[string]string, error) {
	t.Helper()

	r := make(map[string]string)

	err := filepath.Walk(dir, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			return err
		}

		content := ""
		if !info.IsDir() {
			d, err := os.ReadFile(path)
			if err != nil {
				return err
			}
			// ignore given header
			if ignoreHeaders != nil && bytes.HasPrefix(d, ignoreHeaders) {
				return nil
			}
			content = string(d)
		}
		r[strings.TrimPrefix(path, dir)] = content
		return nil
	})
	if err != nil {
		return nil, err
	}

	return r, nil
}

// ignoreDconfDB is a utility function that returns the list of binary dconf db files to ignore during copy with shutils.CopyTree
func ignoreDconfDB(src string, entries []os.FileInfo) []string {
	var r []string
	for _, e := range entries {
		if e.IsDir() {
			continue
		}
		d, err := os.ReadFile(filepath.Join(src, e.Name()))
		if err != nil {
			continue
		}

		if bytes.HasPrefix(d, []byte("GVariant")) {
			r = append(r, e.Name())
		}
	}
	return r
}
