/*
   Project: Adun

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
/** \file **/

#ifndef _ADUN_IOMANAGER_
#define _ADUN_IOMANAGER_

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "AdunKernel/AdCommandInterface.h"
#include "AdunKernel/AdCoreCommand.h"
#include "AdunKernel/AdServerInterface.h"
#include "AdunKernel/AdunDefinitions.h"
#include "AdunKernel/AdunObject.h"
#include "AdunKernel/AdunEnvironment.h"
#include "AdunKernel/AdunDataSource.h"

/*!
\ingroup Inter
AdIOManager manages input and output from the core. 
- Acts as the interface to AdServer
- Loads and processes the simulation data (options and dataSources)
- Opens, closes and provides access to file streams for the other framework objects
- Creates the output directory and associated files
AdIOManager is a singleton class i.e. only one instance of it can be "alive" at any one time.
This instance is created when the core is started and destroyed when it exits. 
Use appIOManager to get a reference to this instance - Never explicitly use any init methods.
*/

@interface AdIOManager: AdObject <AdCommandInterface>
{
	id fileManager;
	NSMutableDictionary* fileStreams;
	NSString *controllerDir;
	NSString *userHome;
	NSString* adunUserDir;
	NSString* resultsDir;
	NSString* outputDir;
	NSString* controllerOutputDir;
	NSProcessInfo *adunInfo;	
	NSMutableDictionary* simulationData; //!< The data for the current simulation
	id serverProxy;
	id core; 	//!< The simulator core
	id serverConnection;
	NSProtocolChecker* checkerInterface;
}
/**
Attempts to connect to the IOManager on localhost
*/
- (BOOL) connectToServer: (NSError**) error;
/**
Causes the ioManager to indicate to the server that it is 
ready to process requests
*/
- (void) acceptRequests;
/**
Closes the connection to the local AdServer instance
\param error If the connection is being closed for a reason other than
normal termination , the NSError object, error, will contain the reason. 
Otherwise it will be nil.
*/
- (void) closeConnection: (NSError*) error;
/**
Sends the data sets in array \e results to the server.
*/
- (void) sendControllerResults: (NSArray*) results;
/**
Returns the IOManager for the simulation
*/
+ (id) appIOManager;
/**
Loads simulation data encapsulated by a ULProcess object
into the Kernel. \e options is a dictionary with one key \e inputSourceType
whose associated value can be either CommandLine or Server indicating the source
of the ULProcess. 
\todo decide on return value
*/
- (id) loadProcessData: (NSDictionary*) dict;
- (FILE*) openFile: (NSString*) file usingName: (NSString*) name flag: (NSString*) fileFlag;
- (FILE*) getStreamForName: (NSString*) name;
- (void) closeStreamWithName: (NSString*) name;
- (void) closeAllStreams;
- (void) writeMatrix: (AdMatrix*) Matrix toStream: (FILE*) stream;
- (id) simulationOptions;
- (id) systemDataSources;
/**
Sets the output directory for the simulation controller to aPath. If \aPath is not
absolute the last path component of \e aPath is used for the directory name
and the location if given by the current directory path. The directory is created
if it doesnt exist. If a directory with the same name exists at the given location
AdIOManager attempts to remove it. If it cant it raises an NSInternalInconsistencyException.
An NSInternalInconsistencyException is also raised if the specified directory cannot be written
to or created.
*/
- (void) setControllerOutputDirectory: (NSString*) aPath;
/**
Sets the output directory of the simulation to \aPath. Behaviour is the same
as setControllerOutputDirectory: .
*/
- (void) setSimulationOutputDirectory: (NSString*) aPath;
/**
Returns the name of the directory to which the simulation writes its output
*/
- (NSString*) simulationOutputDirectory;
/*
Returns the directory to which the controller should write its output.
The name and location of the directory is set using the setControllerOutputDirectory: method.
*/
- (NSString*) controllerOutputDirectory;
- (NSString*) currentDirectoryPath;
- (NSString*) controllerDirectory;
- (NSString*) adunUserDirectory;
- (NSString*) resultsDirectory;
/**
Sets a reference to the AdCore instance of the simulator.
\note Could aquite the reference by implementing the
singleton pattern e.g like this object
*/
- (void) setCore: (id) core;
- (id) core;
@end

#endif

