/*
 * Userspace USB interface for Linux
 * Copyright (C) 2006 Wittawat Yamwong <wittawat@web.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef MY_USB_H
#define MY_USB_H

#include <stdint.h>

#define PACKED __attribute__((packed))

#define USB_DT_DEVICE                    1
#define USB_DT_CONFIGURATION             2
#define USB_DT_STRING                    3
#define USB_DT_INTERFACE                 4
#define USB_DT_ENDPOINT                  5
#define USB_DT_DEVICE_QUALIFIER          6
#define USB_DT_OTHER_SPEED_CONFIGURATION 7
#define USB_DT_INTERFACE_POWER           8

typedef struct PACKED USB_descriptor {
    uint8_t  bLength;
    uint8_t  bDescriptorType;
} USBDescriptor;

typedef struct PACKED USB_device_descriptor {
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint16_t bcdUSB;
    uint8_t  bDeviceClass;
    uint8_t  bDeviceSubClass;
    uint8_t  bDeviceProtocol;
    uint8_t  bMaxPacketSize0;
    uint16_t idVendor;
    uint16_t idProduct;
    uint16_t bcdDevice;
    uint8_t  iManufacturer;
    uint8_t  iProduct;
    uint8_t  iSerialNumber;
    uint8_t  bNumConfigurations;
} USBDeviceDescriptor;

typedef struct PACKED USB_configuration_descriptor {
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint16_t wTotalLength;
    uint8_t  bNumInterfaces;
    uint8_t  bConfigurationValue;
    uint8_t  iConfiguration;
    uint8_t  bmAttributes;
    uint8_t  bMaxPower;
} USBConfigurationDescriptor;

typedef struct PACKED USB_interface_descriptor {
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint8_t  bInterfaceNumber;
    uint8_t  bAlternateSetting;
    uint8_t  bNumEndpoints;
    uint8_t  bInterfaceClass;
    uint8_t  bInterfaceSubClass;
    uint8_t  bInterfaceProtocol;
    uint8_t  iInterface;
} USBInterfaceDescriptor;

typedef struct PACKED USB_endpoint_descriptor {
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint8_t  bEndpointAddress;
    uint8_t  bmAttributes;
    uint16_t wMaxPacketSize;
    uint8_t  bInterval;
} USBEndpointDescriptor;

struct USB_device;
struct USB_urb;
typedef struct USB_device USBDevice;
typedef uint32_t USBDeviceLocation;
typedef uint8_t USBEndpoint;
typedef void (*USBURBCallback)(struct USB_urb *);
typedef int (*USBFindDevicesCallback)(USBDeviceLocation, void *);
typedef uint16_t USBChar;

typedef enum {
    USBControl = 0,
    USBIsochronous = 1,/* TODO: */
    USBBulk = 2,
    USBInterrupt = 3
} USBTransferType;

typedef struct USB_urb {
    void *priv; /* Don't use this field! */

    USBTransferType type;
    USBEndpoint ep;
    int status;
    void *buffer;
    unsigned bufferSize;
    unsigned actualLength;
    USBURBCallback callback; /* The callback will be called in the context of
			      * usbGetCompleteUrb*() */

    void *userContext; /* The usb library won't touch this field. */
} USBURB;



/* timeout is in milliseconds. If timeout < 0, block until the operation
 * completes or an error occurs. */

void usbInit(const char *usbRoot);
void usbCleanup(void);
const char *usbGetRootDir(void);

void usbFindDevices(uint16_t vid, uint16_t pid,
		    USBFindDevicesCallback callback, void *data);
void usbScanDevices(USBFindDevicesCallback callback, void *data);
USBDevice *usbGetDevice(USBDeviceLocation);
void usbFreeDevice(USBDevice*);

const USBDeviceDescriptor *usbGetDeviceDescriptor(USBDevice *);
const USBConfigurationDescriptor *usbGetConfigurationDescriptor(
    USBDevice *, unsigned idx);
USBDeviceLocation usbGetDeviceLocation(USBDevice *);

int usbClaimInterface(USBDevice *, unsigned inf);
int usbReleaseInterface(USBDevice *, unsigned inf);

USBURB *usbNewUrb(void);
void usbDestroyUrb(USBURB *);
int usbSubmitUrb(USBDevice *, USBURB *);
int usbGetCompleteUrb(USBDevice *, USBURB **completed, int timeout);
int usbGetCompleteUrbNoIntr(USBDevice *dev, USBURB **completed, int timeout);
int usbCancelUrb(USBDevice *, USBURB *);

int usbSendBulkMsg(USBDevice *, USBEndpoint, void *data, unsigned len,
		   int timeout);
int usbSendControlMsg(USBDevice *, USBEndpoint,
		      uint8_t bRequest, uint8_t bmRequestType,
		      uint16_t wValue, uint16_t wIndex, uint16_t wLength,
		      void *data, int timeout);

int usbGetDescriptor(USBDevice *, uint8_t type, uint8_t descidx,
		     uint16_t wIndex, uint16_t wLength, void *data);
int usbGetFirstLanguageString(USBDevice *, unsigned idx,
			      void *buf, unsigned bufsize);
int usbResetDevice(USBDevice *);
int usbSetInterface(USBDevice *, uint8_t iface, uint8_t altsetting);
/*int usbSetConfiguration(USBDevice *, uint8_t configuration);*/
/*int usbGetConfiguration(USBDevice *);*/

#endif
