// ---------------------------------------------------------------------------
// - Rvector.hpp                                                             -
// - afnix:mth module - real vector definitions                              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_RVECTOR_HPP
#define  AFNIX_RVECTOR_HPP

#ifndef  AFNIX_RVI_HPP
#include "Rvi.hpp"
#endif
 
namespace afnix {

  /// This Rvector class is the default implementation of the real vector
  /// interface. Internally, the vector is represented as an array of t_real.
  /// @author amaury darsch

  class Rvector : public Rvi {
  public:
    /// generate a random vector by size
    /// @param size the vector size
    /// @param rmin the minimum value
    /// @param rmax the maximum value
    static Rvector random (const t_long size, 
			   const t_real rmin, const t_real rmax);

    /// add a vector with a scalar
    /// @param x the vector argument
    /// @param s the scalar argument
    friend Rvector operator + (const Rvector& x, const t_real s);

    /// add a vector with another one
    /// @param x the vector argument
    /// @param y the vector argument
    friend Rvector operator + (const Rvector& x, const Rvector& y);

    /// substract a vector with a scalar
    /// @param x the vector argument
    /// @param s the scalar argument
    friend Rvector operator - (const Rvector& x, const t_real s);

    /// substract a vector with another one
    /// @param x the vector argument
    /// @param y the vector argument
    friend Rvector operator - (const Rvector& x, const Rvector& y);

    /// multiply a vector with a scalar
    /// @param x the vector argument
    /// @param s the scalar argument
    friend Rvector operator * (const Rvector& x, const t_real s);

    /// divide a vector with a scalar
    /// @param x the vector argument
    /// @param s the scalar argument
    friend Rvector operator / (const Rvector& x, const t_real s);

  private:
    /// the vector element
    t_real* p_vtab;

  public:
    /// create a null vector
    Rvector (void);

    /// create a vector by size
    /// @param size the vector size
    Rvector (const t_long size);

    /// copy construct this vector
    /// @param that the vector to copy
    Rvector (const Rvector& that);

    /// destroy this vector
    ~Rvector (void);

    /// @return the class name
    String repr (void) const;

    /// @return a clone of this object
    Object* clone (void) const;

    /// assign a vector to this one
    /// @param that the vector to assign
    Rvector& operator = (const Rvector& that);

    /// set a vector by position
    /// @param pos the vector position
    /// @param val the value to set
    void set (const t_long pos, const t_real val);

    /// get a vector value by position
    /// @param pos the vector position
    t_real get (const t_long pos) const;

  private:
    // make the matrix class a friend
    friend class Rmatrix;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// operate this object with another object
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);
  };
}

#endif
