/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <iostream>
#include <iomanip>
#include <fstream>
#include <glib/gstdio.h>

#include <gtkmm/stock.h>
#include <gtkmm/actiongroup.h>

#include "gcs-bookmarklist.h"
#include "gcs-cellrendererswatch.h"
#include "gcs-debug.h"
#include "gcs-util.h"
#include "gcs-i18n.h"
#include "gcs-conf.h"

namespace gcs
{
    namespace Widgets
    {
        BookmarkList::BookmarkList(Glib::ustring fname) :
            m_file(fname),
            m_refPalette(Palette::create(fname)),
            m_refUIManager(Gtk::UIManager::create())
        {
            set_model(m_refPalette);
            Glib::RefPtr<Gtk::TreeSelection> refSel = get_selection();
            refSel->set_mode(Gtk::SELECTION_MULTIPLE);

            // display column headers
            set_headers_visible(true);

            // Create a instance of a custom CellRenderer for drawing color
            // swatches
            CellRendererSwatch *const pRenderer = Gtk::manage(new
                    CellRendererSwatch);
            // Create a new Column named "Favorites"
            Gtk::TreeViewColumn *const pColumn = Gtk::manage(new
                    Gtk::TreeViewColumn(_("Favorites")));
            // Pack the custom swatch renderer into the column
            pColumn->pack_start(*pRenderer, false);
            // associate the 'color' property of the custom renderer with the
            // value of m_columns.m_colColor so that the cell renderer can color
            // each swatch according to the bookmarked item
            pColumn->add_attribute(pRenderer->property_color(),
                    m_refPalette->get_color_column());
            // add a the model text column to the TreeView column -- This packs
            // both the swatch and the hexstring into the same treeview column
            pColumn->pack_start(m_refPalette->get_text_column());
            // append the column to the TreeView
            append_column(*pColumn);
            set_reorderable();
            // save the list when it gets re-ordered so that it stays in that
            // order the next time the user runs the app
            signal_drag_end().connect(sigc::hide(
                        sigc::mem_fun(*this, &BookmarkList::on_list_changed)));
            m_refPalette->signal_rows_reordered().connect(sigc::hide(sigc::hide(sigc::hide(
                        sigc::mem_fun(*this, &BookmarkList::on_list_changed)))));
            // save the palette when a color gets renamed
            m_refPalette->signal_row_changed().connect(sigc::hide(sigc::hide(
                        sigc::mem_fun(*this, &BookmarkList::on_list_changed))));

            //std::list<Gtk::TargetEntry> targets;
            //targets.push_back(Gtk::TargetEntry("application/x-color"));
            //drag_dest_set(targets);
            //signal_drag_data_received().connect(sigc::mem_fun(*this,
                        //&BookmarkList::on_drop_drag_data_received));

            // set up the popup menu
            Glib::RefPtr<Gtk::ActionGroup> refActions = Gtk::ActionGroup::create();
            // the name of the menu item in the popup in the favorites list
            refActions->add(Gtk::Action::create("RenameBookmark", Gtk::Stock::EDIT,
                        _("_Rename Color")),
                    sigc::mem_fun(*this, &BookmarkList::on_action_rename));
            m_refUIManager->insert_action_group(refActions);
            try
            {
                m_refUIManager->add_ui_from_file(AGAVE_UIDIR "/bookmarkspopup.ui");
                m_popupMenu = static_cast<Gtk::Menu*>(
                        m_refUIManager->get_widget("/BookmarkPopup"));
                assert(m_popupMenu);
            }
            catch(const Glib::Error& ex)
            {
                std::cerr << __FILE__ << ": " << ex.what() << std::endl;
                throw ex;
            }
        }


        void BookmarkList::add(ColorPtr clr)
        {
            Gtk::TreeModel::Row row = m_refPalette->add(clr);
            // select the newly added row
            Glib::RefPtr<Gtk::TreeView::Selection> sel = get_selection();
            sel->unselect_all();
            sel->select(row);
            // save the bookmarks to disk
            save_to_disk();
        }


        // returns the color of the currently selected row.  Will return NULL
        // if not exactly one row is selected
        ColorPtr BookmarkList::get_color(void)
        {
            ColorPtr pClr;
            Gtk::TreeModel::iterator iter = get_selected_iter();
            if (iter)
            {
                // need to use get_value() here instead of operator[]
                pClr = iter->get_value(m_refPalette->get_color_column());
            }
            // if number of selected rows is not exactly 1, this will be a NULL
            // pointer
            return pClr;
        }


        // clear out the bookmarks list
        void BookmarkList::clear(void)
        {
            get_selection()->unselect_all();
            m_refPalette->clear();
            save_to_disk();
        }


        // check if the list is empty
        bool BookmarkList::empty(void)
        {
            return m_refPalette->children().empty();
        }


        void BookmarkList::remove_selected(void)
        {
            Glib::RefPtr<Gtk::TreeSelection> sel = get_selection();
            std::list<Gtk::TreeModel::Path> paths = sel->get_selected_rows();
            std::list<Gtk::TreeModel::iterator> iterlist;

            // convert the paths to a list of iterators
            for (std::list<Gtk::TreeModel::Path>::iterator pathiter = paths.begin();
                    pathiter != paths.end(); pathiter++)
            {
                iterlist.push_back(m_refPalette->get_iter(*pathiter));
            }

            // de-select and remove the iterators from the treemodel
            for (std::list<Gtk::TreeModel::iterator>::iterator i = iterlist.begin();
                    i != iterlist.end(); i++)
            {
                sel->unselect(*i);
                m_refPalette->erase(*i);
            }
            save_to_disk();
        }


        gint BookmarkList::count_selected(void)
        {
            return get_selection()->count_selected_rows();
        }


        void BookmarkList::save_to_disk(Glib::ustring filename)
        {
            if (filename.empty())
            {
                filename = m_file;
            }

            if (!filename.empty())
            {
                Glib::ustring dirname(Glib::path_get_dirname(filename));
                debug("Dirname: ", dirname);
                try
                {
                    // FIXME: there should be a better way to do this
                    // will throw Glib::FileError if it doesn't exist 
                    Glib::Dir parent(dirname);
                }
                catch (Glib::FileError& e)
                {
                    debug("Creating directory for bookmarks, etc: ", dirname);
                    // create the directory for holding colorscheme data
                    // must have execute bit for creating files inside the directory
                    g_mkdir(dirname.c_str(), 0755);
                }
                std::ofstream fav_file(filename.c_str());
                if (fav_file.is_open())
                {
                    Glib::ustring name = Glib::get_real_name();
                    if (!name.size())
                    {
                        name = Glib::get_user_name();
                    }
                    name = Glib::ustring("Bookmarks for ") + name;
                    // write header
                    fav_file << "GIMP Palette" << std::endl;
                    fav_file << "Name: " << name << std::endl;
                    if (m_refPalette->get_columns())
                    {
                        fav_file << "Columns: " << m_refPalette->get_columns() << std::endl;
                    }
                    fav_file << "#" << std::endl;
                    fav_file << "# generated by GNOME Colorscheme" << std::endl;
                    fav_file << "#" << std::endl;

                    // some variables for use in the loop
                    ColorPtr p;
                    Glib::ustring color_name;
                    // write the colors
                    for (Gtk::TreeModel::iterator iter = m_refPalette->children().begin();
                            iter != m_refPalette->children().end();
                            ++iter)
                    {
                        p = (*iter).get_value(m_refPalette->get_color_column());
                        color_name = (*iter).get_value(m_refPalette->get_text_column());
                        fav_file << std::setw(3) << p->get_red() << " "
                            << std::setw(3) << p->get_green() << " "
                            << std::setw(3) << p->get_blue() << " "
                            << color_name << std::endl;
                    }
                }
                else
                {
                    std::cerr << "*** Error opening bookmarks for writing" << std::endl;
                }
            }
        }


        bool BookmarkList::on_button_press_event(GdkEventButton* event)
        {
            Gtk::TreeView::on_button_press_event(event);
            debug("button pressed: ", event->button);

            //Then do our custom stuff:
            if (event->type == GDK_BUTTON_PRESS && event->button == 3)
            {
                debug("Right button pressed");
                m_popupMenu->popup(event->button, event->time);
            }
            return true;
        }


        void BookmarkList::on_action_rename(void)
        {
            Dialogs::RenameEntry entry;
            Glib::ustring name;
            Gtk::TreeModel::iterator iter = get_selected_iter();
            // make sure it's valid
            if (iter)
            {
                // need to use get_value() here instead of operator[]
                name = iter->get_value(m_refPalette->get_text_column());
            }
            entry.set_name(name);
            if (entry.run() == Gtk::RESPONSE_OK)
            {
                if (iter)
                {
                    iter->set_value(m_refPalette->get_text_column(),
                            entry.get_name());
                }
            }
        }


        Gtk::TreeModel::iterator BookmarkList::get_selected_iter(void)
        {
            Gtk::TreeModel::iterator iter;
            Glib::RefPtr<Gtk::TreeSelection> refSel = get_selection();
            std::list<Gtk::TreeModel::Path> paths = refSel->get_selected_rows();
            if (paths.size() == 1)
            {
                // get an iterator pointing to the first (and only) selected row
                iter = m_refPalette->get_iter(*paths.begin());
            }
            return iter;
        }


        /*
        void BookmarkList::on_drop_drag_data_received(const
                Glib::RefPtr<Gdk::DragContext>& context, int x, int y,
                const Gtk::SelectionData& selection_data, guint info,
                guint time)
        {
            debug("== Drop received ==");
            boost::shared_ptr<Gdk::Color> c = get_dropped_color(selection_data);
            bool drag_success = false;

            if (c)
            {
                // create a gcs::Color from the Gdk::Color
                ColorPtr pClr = Color::create(*c);
                // set the application's current color
                add(pClr);
                bool drag_success = true;
            }
            context->drag_finish(drag_success, false, time);
        }
        */

    }   // namespace Widgets 

    namespace Dialogs
    {

        RenameEntry::RenameEntry(void) :
            pEntry(Gtk::manage(new Gtk::Entry)),
            // The instructions for the rename dialog
            pInstructions(Gtk::manage(new Gtk::Label(_("Enter a new name:"),
                            Gtk::ALIGN_LEFT)))
        {
            // the title of the color rename dialog
            set_title(_("Rename Color"));
            set_border_width(Conf::WINDOW_BORDER);
            add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
            add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
            set_has_separator(false);
            pInstructions->set_use_markup(true);

            // make it so that hitting enter in the text entry will 'click' OK
            pEntry->set_activates_default();
            set_default_response(Gtk::RESPONSE_OK);

            Gtk::VBox* pVbox = get_vbox();
            assert(pVbox);
            pVbox->set_spacing(Conf::UI_SPACING_SMALL);
            pVbox->set_border_width(Conf::UI_SPACING_SMALL);
            pVbox->pack_start(*pInstructions);
            pVbox->pack_start(*pEntry);

            show_all();
        }


        void RenameEntry::set_name(Glib::ustring name)
        {
            pEntry->set_text(name);
            pEntry->select_region(0, name.size());
        }


        Glib::ustring RenameEntry::get_name(void)
        {
            return pEntry->get_text();
        }

    } // namespace Dialogs

}   // namespace gcs
