--
-- Copyright (c) 2007, 2008 Tero Koskinen <tero.koskinen@iki.fi>
--
-- Permission to use, copy, modify, and distribute this software for any
-- purpose with or without fee is hereby granted, provided that the above
-- copyright notice and this permission notice appear in all copies.
--
-- THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
-- WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
-- MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
-- ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
-- WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
-- ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
-- OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
--

with Ada.Unchecked_Deallocation;

package body Ahven.Results is
   use Ahven.Results.Result_List;
   use Ahven.Results.Result_Info_List;

   -- Bunch of setters and getters.
   -- The implementation is straightforward.
   procedure Set_Test_Name (Info : in out Result_Info;
                            Name : Unbounded_String) is
   begin
      Info.Test_Name := Name;
   end Set_Test_Name;

   procedure Set_Routine_Name (Info : in out Result_Info;
                               Name : Unbounded_String) is
   begin
      Info.Routine_Name := Name;
   end Set_Routine_Name;

   procedure Set_Message (Info : in out Result_Info;
                          Message : Unbounded_String) is
   begin
      Info.Message := Message;
   end Set_Message;

   procedure Set_Test_Name (Info : in out Result_Info; Name : String) is
   begin
      Set_Test_Name (Info, To_Unbounded_String (Name));
   end Set_Test_Name;

   procedure Set_Routine_Name (Info : in out Result_Info; Name : String) is
   begin
      Set_Routine_Name (Info, To_Unbounded_String (Name));
   end Set_Routine_Name;

   procedure Set_Message (Info : in out Result_Info; Message : String) is
   begin
      Set_Message (Info, To_Unbounded_String (Message));
   end Set_Message;

   procedure Set_Long_Message (Info : in out Result_Info;
                               Message : Unbounded_String) is
   begin
      Info.Long_Message := Message;
   end Set_Long_Message;

   procedure Set_Long_Message (Info : in out Result_Info; Message : String) is
   begin
      Set_Long_Message (Info, To_Unbounded_String (Message));
   end Set_Long_Message;

   procedure Set_Execution_Time (Info : in out Result_Info;
                                 Elapsed_Time : Duration) is
   begin
      Info.Execution_Time := Elapsed_Time;
   end Set_Execution_Time;

   procedure Set_Output_File (Info : in out Result_Info;
                              Filename : Unbounded_String) is
   begin
      Info.Output_File := Filename;
   end Set_Output_File;

   procedure Set_Output_File (Info : in out Result_Info;
                              Filename : String) is
   begin
      Set_Output_File (Info, To_Unbounded_String (Filename));
   end Set_Output_File;

   function Get_Test_Name (Info : Result_Info) return Unbounded_String is
   begin
      return Info.Test_Name;
   end Get_Test_Name;

   function Get_Routine_Name (Info : Result_Info) return Unbounded_String is
   begin
      return Info.Routine_Name;
   end Get_Routine_Name;

   function Get_Message (Info : Result_Info) return Unbounded_String is
   begin
      return Info.Message;
   end Get_Message;

   function Get_Long_Message (Info : Result_Info) return Unbounded_String is
   begin
      return Info.Long_Message;
   end Get_Long_Message;

   function Get_Execution_Time (Info : Result_Info) return Duration is
   begin
      return Info.Execution_Time;
   end Get_Execution_Time;

   function Get_Output_File (Info : Result_Info) return Unbounded_String is
   begin
      return Info.Output_File;
   end Get_Output_File;

   procedure Add_Child (Collection : in out Result_Collection;
                        Child : Result_Collection_Access) is
   begin
      Append (Collection.Children, Child);
   end Add_Child;

   procedure Add_Error (Collection : in out Result_Collection;
                        Info : Result_Info) is
   begin
      Append (Collection.Errors, Info);
   end Add_Error;

   procedure Add_Failure (Collection : in out Result_Collection;
                          Info : Result_Info) is
   begin
      Append (Collection.Failures, Info);
   end Add_Failure;

   procedure Add_Pass (Collection : in out Result_Collection;
                       Info : Result_Info) is
   begin
      Append (Collection.Passes, Info);
   end Add_Pass;

   -- When Result_Collection is finalized, it recursively releases
   -- its all children.
   procedure Finalize (Collection : in out Result_Collection) is
      procedure Free is
        new Ada.Unchecked_Deallocation (Object => Result_Collection,
                                        Name   => Result_Collection_Access);

      Iter : Result_List.Iterator := First (Collection.Children);
      Ptr  : Result_Collection_Access := null;
   begin
      loop
         exit when not Is_Valid (Iter);

         Ptr := Data (Iter);
         Free (Ptr);

         Iter := Next (Iter);
      end loop;
      Remove_All (Collection.Children);

      -- No need to call Free for these three since
      -- they are stored as plain objects instead of pointers.
      Remove_All (Collection.Errors);
      Remove_All (Collection.Failures);
      Remove_All (Collection.Passes);
   end Finalize;

   procedure Set_Name (Collection : in out Result_Collection;
                       Name : Unbounded_String) is
   begin
      Collection.Test_Name := Name;
   end Set_Name;

   procedure Set_Parent (Collection : in out Result_Collection;
                         Parent : Result_Collection_Access) is
   begin
      Collection.Parent := Parent;
   end Set_Parent;

   function Test_Count (Collection : Result_Collection) return Natural is
      Count : Natural := Result_Info_List.Size (Collection.Errors) +
                         Result_Info_List.Size (Collection.Failures) +
                         Result_Info_List.Size (Collection.Passes);
      Iter  : Result_List.Iterator := First (Collection.Children);
   begin
      loop
         exit when not Is_Valid (Iter);

         Count := Count + Test_Count (Data (Iter).all);
         Iter := Next (Iter);
      end loop;
      return Count;
   end Test_Count;

   function Direct_Test_Count (Collection : Result_Collection)
     return Natural
   is
   begin
      return Size (Collection.Errors) +
             Size (Collection.Failures) +
             Size (Collection.Passes);
   end Direct_Test_Count;

   function Pass_Count (Collection : Result_Collection) return Natural is
      Count : Natural              := Size (Collection.Passes);
      Iter  : Result_List.Iterator := First (Collection.Children);
   begin
      loop
         exit when not Is_Valid (Iter);

         Count := Count + Pass_Count (Data (Iter).all);
         Iter := Next (Iter);
      end loop;
      return Count;
   end Pass_Count;

   function Error_Count (Collection : Result_Collection) return Natural is
      Count : Natural              := Size (Collection.Errors);
      Iter  : Result_List.Iterator := First (Collection.Children);
   begin
      loop
         exit when not Is_Valid (Iter);

         Count := Count + Error_Count (Data (Iter).all);
         Iter := Next (Iter);
      end loop;
      return Count;
   end Error_Count;

   function Failure_Count (Collection : Result_Collection) return Natural is
      Count : Natural              := Size (Collection.Failures);
      Iter  : Result_List.Iterator := First (Collection.Children);
   begin
      loop
         exit when not Is_Valid (Iter);

         Count := Count + Failure_Count (Data (Iter).all);
         Iter := Next (Iter);
      end loop;
      return Count;
   end Failure_Count;

   function Get_Test_Name (Collection : Result_Collection)
     return Unbounded_String is
   begin
      return Collection.Test_Name;
   end Get_Test_Name;

   function Get_Parent (Collection : Result_Collection)
     return Result_Collection_Access is
   begin
      return Collection.Parent;
   end Get_Parent;

   function Get_Execution_Time (Collection : Result_Collection)
     return Duration
   is
      Iter : Result_Info_List.Iterator;
      Total_Time : Duration := 0.0;
      Child_Iter : Result_List.Iterator;
   begin
      Iter := First (Collection.Passes);
      Pass_Loop:
      loop
         exit Pass_Loop when not Is_Valid (Iter);
         Total_Time := Total_Time + Get_Execution_Time (Data (Iter));
         Iter := Next (Iter);
      end loop Pass_Loop;

      Iter := First (Collection.Failures);
      Failure_Loop:
      loop
         exit Failure_Loop when not Is_Valid (Iter);
         Total_Time := Total_Time + Get_Execution_Time (Data (Iter));
         Iter := Next (Iter);
      end loop Failure_Loop;

      Iter := First (Collection.Errors);
      Error_Loop:
      loop
         exit Error_Loop when not Is_Valid (Iter);
         Total_Time := Total_Time + Get_Execution_Time (Data (Iter));
         Iter := Next (Iter);
      end loop Error_Loop;

      Child_Loop:
      loop
         exit Child_Loop when not Result_List.Is_Valid (Child_Iter);
         Total_Time := Total_Time +
                       Get_Execution_Time (Result_List.Data (Child_Iter).all);
         Child_Iter := Result_List.Next (Child_Iter);
      end loop Child_Loop;

      return Total_Time;
   end Get_Execution_Time;

   function First_Pass (Collection : Result_Collection)
     return Result_Info_Iterator is
   begin
      return First (Collection.Passes);
   end First_Pass;

   function First_Failure (Collection : Result_Collection)
     return Result_Info_Iterator is
   begin
      return First (Collection.Failures);
   end First_Failure;

   function First_Error (Collection : Result_Collection)
     return Result_Info_Iterator is
   begin
      return First (Collection.Errors);
   end First_Error;

   function Next (Iter : Result_Info_Iterator) return Result_Info_Iterator is
   begin
      return Result_Info_Iterator
        (Result_Info_List.Next (Result_Info_List.Iterator (Iter)));
   end Next;

   function Data (Iter : Result_Info_Iterator) return Result_Info is
   begin
      return Result_Info_List.Data (Result_Info_List.Iterator (Iter));
   end Data;

   function Is_Valid (Iter : Result_Info_Iterator) return Boolean is
   begin
      return Result_Info_List.Is_Valid (Result_Info_List.Iterator (Iter));
   end Is_Valid;

   function First_Child (Collection : in Result_Collection)
     return Result_Collection_Iterator is
   begin
      return First (Collection.Children);
   end First_Child;

   function Next (Iter : Result_Collection_Iterator)
     return Result_Collection_Iterator is
   begin
      return Result_Collection_Iterator
        (Result_List.Next (Result_List.Iterator (Iter)));
   end Next;

   function Is_Valid (Iter : Result_Collection_Iterator) return Boolean is
   begin
      return Result_List.Is_Valid (Result_List.Iterator (Iter));
   end Is_Valid;

   function Data (Iter : Result_Collection_Iterator)
     return Result_Collection_Access is
   begin
      return Result_List.Data (Result_List.Iterator (Iter));
   end Data;

   function Child_Depth (Collection : in Result_Collection) return Natural
   is
      function Child_Depth_Impl (Coll : in Result_Collection;
                                 Level : Natural) return Natural;

      function Child_Depth_Impl (Coll : in Result_Collection;
                                 Level : Natural)
        return Natural
      is
         Max     : Natural := 0;
         Current : Natural := 0;
         Iter    : Result_List.Iterator
           := Result_List.First (Coll.Children);
      begin
         loop
            exit when not Is_Valid (Iter);
            Current := Child_Depth_Impl (Data (Iter).all, Level + 1);
            if Max < Current then
               Max := Current;
            end if;
            Iter := Result_List.Next (Iter);
         end loop;
         return Level + Max;
      end Child_Depth_Impl;
   begin
      return Child_Depth_Impl (Collection, 0);
   end Child_Depth;

   package body Result_Info_List is
      procedure Remove (Ptr : Node_Access) is
         procedure Free is
           new Ada.Unchecked_Deallocation (Object => Node,
                                           Name   => Node_Access);
         My_Ptr : Node_Access := Ptr;
      begin
         Ptr.Next := null;
         Ptr.Prev := null;
         Free (My_Ptr);
      end Remove;

      procedure Append (Target : in out List; Node_Data : Result_Info) is
         New_Node : Node_Access  := null;
      begin
         New_Node := new Node'(Data => Node_Data,
            Next => null, Prev => Target.Last);

         if Target.Last = null then
            Target.Last := New_Node;
            Target.First := New_Node;
         else
            Target.Last.Next := New_Node;
            Target.Last := New_Node;
         end if;

         Target.Size := Target.Size + 1;
      end Append;

      procedure Remove_All (Target : in out List) is
         Current_Node : Node_Access := Target.First;
         Next_Node : Node_Access := null;
      begin
         while Current_Node /= null loop
            Next_Node := Current_Node.Next;
            Remove (Current_Node);
            Current_Node := Next_Node;
         end loop;

         Target.First := null;
         Target.Last := null;
         Target.Size := 0;
      end Remove_All;

      function Empty (Target : List) return Boolean is
      begin
         if Target.Size = 0 then
            return True;
         end if;
         return False;
      end Empty;

      function First (Target : List) return Iterator is
      begin
         if Target.Size = 0 then
            return null;
         end if;

         return Iterator (Target.First);
      end First;

      function Last (Target : List) return Iterator is
      begin
         if Target.Size = 0 then
            return null;
         end if;

         return Iterator (Target.Last);
      end Last;

      function Next (Iter : Iterator) return Iterator is
      begin
         if Iter = null then
            raise Invalid_Iterator;
         end if;
         return Iterator (Iter.Next);
      end Next;

      function Prev (Iter : Iterator) return Iterator is
      begin
         if Iter = null then
            raise Invalid_Iterator;
         end if;
         return Iterator (Iter.Prev);
      end Prev;

      function Data (Iter : Iterator) return Result_Info is
      begin
         return Iter.Data;
      end Data;

      function Is_Valid (Iter : Iterator) return Boolean is
      begin
         return Iter /= null;
      end Is_Valid;

      function Size (Target : List) return Natural is
      begin
         return Target.Size;
      end Size;

      procedure Initialize (Target : in out List) is
      begin
         Target.Last := null;
         Target.First := null;
         Target.Size := 0;
      end Initialize;

      procedure Finalize (Target : in out List) is
      begin
         Remove_All (Target);
      end Finalize;

      procedure Adjust (Target : in out List) is
         Target_Last : Node_Access := null;
         Target_First : Node_Access := null;
         Current : Node_Access := Target.First;
         New_Node : Node_Access;
      begin
         while Current /= null loop
            New_Node := new Node'(Data => Current.Data,
              Next => null, Prev => Target_Last);

            if Target_Last = null then
               Target_Last := New_Node;
               Target_First := New_Node;
            else
               Target_Last.Next := New_Node;
               Target_Last := New_Node;
            end if;

            Current := Current.Next;
         end loop;
         Target.First := Target_First;
         Target.Last := Target_Last;
      end Adjust;
   end Result_Info_List;

   package body Result_List is
      procedure Remove (Ptr : Node_Access) is
         procedure Free is
           new Ada.Unchecked_Deallocation (Object => Node,
                                           Name => Node_Access);
         My_Ptr : Node_Access := Ptr;
      begin
         Ptr.Next := null;
         Ptr.Prev := null;
         Free (My_Ptr);
      end Remove;

      procedure Append (Target : in out List;
                        Node_Data : Result_Collection_Access) is
         New_Node : Node_Access  := null;
      begin
         New_Node := new Node'(Data => Node_Data,
            Next => null, Prev => Target.Last);

         if Target.Last = null then
            Target.Last := New_Node;
            Target.First := New_Node;
         else
            Target.Last.Next := New_Node;
            Target.Last := New_Node;
         end if;

         Target.Size := Target.Size + 1;
      end Append;

      procedure Remove_All (Target : in out List) is
         Current_Node : Node_Access := Target.First;
         Next_Node : Node_Access := null;
      begin
         while Current_Node /= null loop
            Next_Node := Current_Node.Next;
            Remove (Current_Node);
            Current_Node := Next_Node;
         end loop;

         Target.First := null;
         Target.Last := null;
         Target.Size := 0;
      end Remove_All;

      function Empty (Target : List) return Boolean is
      begin
         if Target.Size = 0 then
            return True;
         end if;
         return False;
      end Empty;

      function First (Target : List) return Iterator is
      begin
         if Target.Size = 0 then
            return null;
         end if;

         return Iterator (Target.First);
      end First;

      function Last (Target : List) return Iterator is
      begin
         if Target.Size = 0 then
            return null;
         end if;

         return Iterator (Target.Last);
      end Last;

      function Next (Iter : Iterator) return Iterator is
      begin
         if Iter = null then
            raise Invalid_Iterator;
         end if;
         return Iterator (Iter.Next);
      end Next;

      function Prev (Iter : Iterator) return Iterator is
      begin
         if Iter = null then
            raise Invalid_Iterator;
         end if;
         return Iterator (Iter.Prev);
      end Prev;

      function Data (Iter : Iterator) return Result_Collection_Access is
      begin
         return Iter.Data;
      end Data;

      function Is_Valid (Iter : Iterator) return Boolean is
      begin
         return Iter /= null;
      end Is_Valid;

      procedure Initialize (Target : in out List) is
      begin
         Target.Last := null;
         Target.First := null;
         Target.Size := 0;
      end Initialize;

      procedure Finalize (Target : in out List) is
      begin
         Remove_All (Target);
      end Finalize;

      procedure Adjust (Target : in out List) is
         Target_Last : Node_Access := null;
         Target_First : Node_Access := null;
         Current : Node_Access := Target.First;
         New_Node : Node_Access;
      begin
         while Current /= null loop
            New_Node := new Node'(Data => Current.Data,
              Next => null, Prev => Target_Last);

            if Target_Last = null then
               Target_Last := New_Node;
               Target_First := New_Node;
            else
               Target_Last.Next := New_Node;
               Target_Last := New_Node;
            end if;

            Current := Current.Next;
         end loop;
         Target.First := Target_First;
         Target.Last := Target_Last;
      end Adjust;
   end Result_List;

end Ahven.Results;
