/**********************************************************************
 **
 ** Inclination - contains methods and attributes for an inclination 
 **               object for use on the builder port and during new player 
 **               login
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/


#ifndef INCLINATION_C
#define INCLINATION_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "inclination.h"
#include "lexer.h"
#include "objtype.h"
#include "builder.h"
#include "player.h"
#include "objtype.h"
#include "newfuncts.h"
#include "utils.h"
#include "global.h"
#include "specials.h"
#include "code.h"

/***********************************************************************
 ** Inclination (constructor) - loads inclination name and attributes
 **
 ** Parameters: inclination_name - the name to give to the inclination
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Inclination::Inclination(char *inclination_name)
{
   Strings tmp_name;

   obj_type = OBJ_TYPE_INCLINAT;
   tmp_name.assign_word(inclination_name, 1);

   set_name(tmp_name.str_show());

   set_description("Describe the inclination here.\n");
   set_special(NULL);

   str_offset = dex_offset = intel_offset = con_offset = magic_offset = 
      wis_offset = cha_offset = endur_offset = 0;
}


/***********************************************************************
 ** ~Inclination (destructor) - cleans up the Inclination
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Inclination::~Inclination()
{
}


/***********************************************************************
 ** set_description - sets the description string for this inclination
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Inclination::set_description(char *the_string)
{
   if (the_string == NULL)
      return;

   description = the_string;
}

/***********************************************************************
 ** get_description - gets the description string for this inclination
 **
 ** Parameters: None
 **
 ** Returns: pointer to the description string
 **
 ***********************************************************************/

char *Inclination::get_description(void)
{
   return description.str_show();
}



/***********************************************************************
 ** set_special - sets the special string indicating which special should 
 **               be run to set special attributes on players
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Inclination::set_special(char *the_string)
{
   if (the_string == NULL)
      return;

   the_special = the_string;
}

/***********************************************************************
 ** get_special - sets the special on the inclination
 **
 ** Parameters: None
 **
 ** Returns: pointer to the special string
 **
 ***********************************************************************/

char *Inclination::get_special(void)
{
   return the_special.str_show();
}


/***********************************************************************
 ** set_str_offset - sets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Inclination::set_str_offset(int the_num)
{
   str_offset = the_num;
}

/***********************************************************************
 ** get_str_offset - gets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Returns: the offset value
 **
 ***********************************************************************/

int Inclination::get_str_offset(void)
{
   return str_offset;
}


/***********************************************************************
 ** set_dex_offset - sets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Inclination::set_dex_offset(int the_num)
{
   dex_offset = the_num;
}

/***********************************************************************
 ** get_dex_offset - gets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Returns: the offset value
 **
 ***********************************************************************/

int Inclination::get_dex_offset(void)
{
   return dex_offset;
}

/***********************************************************************
 ** set_intel_offset - sets the offset, whether positive or negative on this
 **                    attribute
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Inclination::set_intel_offset(int the_num)
{
   intel_offset = the_num;
}

/***********************************************************************
 ** get_intel_offset - gets the offset, whether positive or negative on this
 **                    attribute
 **
 ** Returns: the offset value
 **
 ***********************************************************************/

int Inclination::get_intel_offset(void)
{
   return intel_offset;
}

/***********************************************************************
 ** set_con_offset - sets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Inclination::set_con_offset(int the_num)
{
   con_offset = the_num;
}

/***********************************************************************
 ** get_con_offset - gets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Returns: the offset value
 **
 ***********************************************************************/

int Inclination::get_con_offset(void)
{
   return con_offset;
}


/***********************************************************************
 ** set_cha_offset - sets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Inclination::set_cha_offset(int the_num)
{
   cha_offset = the_num;
}

/***********************************************************************
 ** get_cha_offset - gets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Returns: the offset value
 **
 ***********************************************************************/

int Inclination::get_cha_offset(void)
{
   return cha_offset;
}


/***********************************************************************
 ** set_wis_offset - sets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Inclination::set_wis_offset(int the_num)
{
   wis_offset = the_num;
}

/***********************************************************************
 ** get_wis_offset - gets the offset, whether positive or negative on this
 **                  attribute
 **
 ** Returns: the offset value
 **
 ***********************************************************************/

int Inclination::get_wis_offset(void)
{
   return wis_offset;
}


/***********************************************************************
 ** load_inclination - loads an inclination from a file into memory
 **
 ** Parameters: the_file - where we are getting the inclination from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Inclination::load_inclination(FILE *the_file, ErrLog *error_log, 
                                                            int is_builder)
{
   token_record *the_token;
   char         *tmp_charholder;
   Strings      holder;
   int          neg = 0;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');
 
       if (the_token->token_type != T_NUMERICAL)
       {
           error_log->log_err("Invalid format in inclination file", 
                                                         "load_inclination");
           return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* get the description */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_description(tmp_charholder);
   delete tmp_charholder;

   /* Set special value */
   the_token = get_token(the_file, '\0');     
   set_special(the_token->the_string);

   /* Set str_offset value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      neg = 1;
      the_token = get_token(the_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute str_offset for inclination %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_inclination");
      return -1;
   }
   if (neg)
      set_str_offset(-(atoi(the_token->the_string)));
   else
      set_str_offset(atoi(the_token->the_string));
   neg = 0;

   /* Set dex_offset value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      neg = 1;
      the_token = get_token(the_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute dex_offset for inclination %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_inclination");
      return -1;
   }
   if (neg)
      set_dex_offset(-(atoi(the_token->the_string)));
   else
      set_dex_offset(atoi(the_token->the_string));
   neg = 0;


   /* Set con_offset value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      neg = 1;
      the_token = get_token(the_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute con_offset for inclination %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_inclination");
      return -1;
   }
   if (neg)
      set_con_offset(-(atoi(the_token->the_string)));
   else
      set_con_offset(atoi(the_token->the_string));
   neg = 0;


   /* Set intel_offset value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      neg = 1;
      the_token = get_token(the_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf(
        "Invalid format for attribute intel_offset for inclination %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_inclination");
      return -1;
   }
   if (neg)
      set_intel_offset(-(atoi(the_token->the_string)));
   else
      set_intel_offset(atoi(the_token->the_string));
   neg = 0;

   /* Set wis_offset value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      neg = 1;
      the_token = get_token(the_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf(
        "Invalid format for attribute wis_offset for inclination %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_inclination");
      return -1;
   }
   if (neg)
      set_wis_offset(-(atoi(the_token->the_string)));
   else
      set_wis_offset(atoi(the_token->the_string));
   neg = 0;

   /* Set cha_offset value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      neg = 1;
      the_token = get_token(the_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf(
        "Invalid format for attribute cha_offset for inclination %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_inclination");
      return -1;
   }
   if (neg)
      set_cha_offset(-(atoi(the_token->the_string)));
   else
      set_cha_offset(atoi(the_token->the_string));
   neg = 0;

   return 1;
}


/***********************************************************************
 ** describe - describes the inclination to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Inclination::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GInclination: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GSpecial: \t&+g%s&*\n", get_special());
   the_builder->send_bldr("&+GStrOffset: \t&+w%d&*\n", get_str_offset());
   the_builder->send_bldr("&+GDexOffset: \t&+w%d&*\n", get_dex_offset());
   the_builder->send_bldr("&+GConOffset: \t&+w%d&*\n", get_con_offset());
   the_builder->send_bldr("&+GIntelOffset: \t&+w%d&*\n", get_intel_offset());
   the_builder->send_bldr("&+GWisOffset: \t&+w%d&*\n", get_wis_offset());
   the_builder->send_bldr("&+GChaOffset: \t&+w%d&*\n", get_cha_offset());
   the_builder->send_bldr("&+GDesc: \n&+w%s&*\n", get_description());
   the_builder->send_bldr("\n\n");
}


/***********************************************************************
 ** describe - describes the inclination to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Inclination::describe(Player *the_player)
{
   the_player->send_plr("\n&+GInclination: \t&+M%s&*\n", get_name());
   the_player->send_plr("&+GSpecial: \t&+g%s&*\n", get_special());
   the_player->send_plr("&+GStrOffset: \t&+w%d&*\n", get_str_offset());
   the_player->send_plr("&+GDexOffset: \t&+w%d&*\n", get_dex_offset());
   the_player->send_plr("&+GConOffset: \t&+w%d&*\n", get_con_offset());
   the_player->send_plr("&+GIntelOffset: \t&+w%d&*\n", get_intel_offset());
   the_player->send_plr("&+GWisOffset: \t&+w%d&*\n", get_wis_offset());
   the_player->send_plr("&+GChaOffset: \t&+w%d&*\n", get_cha_offset());
   the_player->send_plr("&+GDesc: \n&+w%s&*\n", get_description());
   the_player->send_plr("\n\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Inclination::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on an inclination.\n"
               "   desc, special, stroffset, dexoffset, inteloffset,\n"
               "   wisoffset, chaoffset, and conoffset\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "desc",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&description) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "special",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
             send_bldr("You need to specify an Special to set to.\n");
         return -1;
      }
 
      holder.assign_word(the_parsed->get_speech(), 1);

      set_special(holder.str_show());

      the_builder->send_bldr("Special on %s set to: %s\n", get_name(), 
                                                     get_special());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "stroffset",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (((!isdigit(*(the_parsed->get_speech()))) && 
          (*(the_parsed->get_speech()) != '-')) ||
            ((*(the_parsed->get_speech()) == '-') &&
             (!isdigit(*(the_parsed->get_speech() + 1)))))
      {
         the_builder->
                   send_bldr("You need to specify a number as StrOffset.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      set_str_offset(value);
      the_builder->send_bldr("StrOffset set to %d on inclination object %s.\n",
                                          get_str_offset(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "dexoffset",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (((!isdigit(*(the_parsed->get_speech()))) && 
          (*(the_parsed->get_speech()) != '-')) ||
            ((*(the_parsed->get_speech()) == '-') &&
             (!isdigit(*(the_parsed->get_speech() + 1)))))
      {
         the_builder->
                   send_bldr("You need to specify a number as DexOffset.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      set_dex_offset(value);
      the_builder->send_bldr("DexOffset set to %d on inclination object %s.\n",
                                          get_dex_offset(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "conoffset",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (((!isdigit(*(the_parsed->get_speech()))) && 
          (*(the_parsed->get_speech()) != '-')) ||
            ((*(the_parsed->get_speech()) == '-') &&
             (!isdigit(*(the_parsed->get_speech() + 1)))))
      {
         the_builder->
                   send_bldr("You need to specify a number as ConOffset.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      set_con_offset(value);
      the_builder->send_bldr("ConOffset set to %d on inclination object %s.\n",
                                          get_con_offset(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "inteloffset",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (((!isdigit(*(the_parsed->get_speech()))) && 
          (*(the_parsed->get_speech()) != '-')) ||
            ((*(the_parsed->get_speech()) == '-') &&
             (!isdigit(*(the_parsed->get_speech() + 1)))))
      {
         the_builder->
                   send_bldr("You need to specify a number as IntelOffset.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      set_intel_offset(value);
      the_builder->send_bldr("IntelOffset set to %d on inclination object %s.\n",
                                          get_intel_offset(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "chaoffset",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (((!isdigit(*(the_parsed->get_speech()))) && 
          (*(the_parsed->get_speech()) != '-')) ||
            ((*(the_parsed->get_speech()) == '-') &&
             (!isdigit(*(the_parsed->get_speech() + 1)))))
      {
         the_builder->
                   send_bldr("You need to specify a number as ChaOffset.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      set_cha_offset(value);
      the_builder->send_bldr("ChaOffset set to %d on inclination object %s.\n",
                                          get_cha_offset(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "wisoffset",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (((!isdigit(*(the_parsed->get_speech()))) && 
          (*(the_parsed->get_speech()) != '-')) ||
            ((*(the_parsed->get_speech()) == '-') &&
             (!isdigit(*(the_parsed->get_speech() + 1)))))
      {
         the_builder->
                   send_bldr("You need to specify a number as WisOffset.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      set_wis_offset(value);
      the_builder->send_bldr("WisOffset set to %d on inclination object %s.\n",
                                          get_wis_offset(), get_name());
      return 1;
   }


   the_builder->send_bldr("The attribute '%s' is not an inclination attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the inclination to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ***********************************************************************/
   
void Inclination::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nincl %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   fprintf(the_file, "^%s^\n", (get_description() == NULL) ? "" : 
                                                        get_description());
   fprintf(the_file, "%s\n", (get_special() == NULL) ? "none" : get_special());
   fprintf(the_file, "%d\n", get_str_offset());
   fprintf(the_file, "%d\n", get_dex_offset());
   fprintf(the_file, "%d\n", get_con_offset());
   fprintf(the_file, "%d\n", get_intel_offset());
   fprintf(the_file, "%d\n", get_wis_offset());
   fprintf(the_file, "%d\n", get_cha_offset());
}


/***********************************************************************
 ** is_modified - has this inclination been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/
   
int Inclination::is_modified(void)
{
   return modified;
}


/***********************************************************************
 ** set_modified - shall we set this? Yes
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Inclination::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** copy_object - copies the object to an inclination of a different name
 **
 ** Parameters: copy_from - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Inclination::copy_object(Entity *copy_obj)
{
   Inclination *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_INCLINAT)
      return 0;

   copy_from = (Inclination *) copy_obj;

   /******* set the action attributes *****/
   set_description(copy_from->get_description());
   set_special(copy_from->get_special());
   set_str_offset(copy_from->get_str_offset());
   set_dex_offset(copy_from->get_dex_offset());
   set_con_offset(copy_from->get_con_offset());
   set_intel_offset(copy_from->get_intel_offset());
   set_wis_offset(copy_from->get_wis_offset());
   set_cha_offset(copy_from->get_cha_offset());

   return 1;
}


/***********************************************************************
 ** set_values - sets the values of the player based on the inclination
 **
 ** Parameters: the_player - the player to set
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Inclination::set_values(Player *the_player)
{
   Specials    *special_obj;
   Strings     holder;
   in_params   fill_param;
   special_env the_environment;

   the_player->set_strength(the_player->get_strength() + get_str_offset());
   the_player->set_dexterity(the_player->get_dexterity() + get_dex_offset());
   the_player->set_intel(the_player->get_intel() + get_intel_offset());
   the_player->set_constitution(the_player->get_constitution() + get_con_offset());
   the_player->set_wisdom(the_player->get_wisdom() + get_wis_offset());
   the_player->set_charisma(the_player->get_charisma() + get_cha_offset());

   /* this is where the special, if it exists, is executed */
   if ((the_special.str_show() != NULL) &&
       (STRCASECMP(the_special.str_show(), "none")))
   {
      if ((special_obj = mainstruct->get_special(the_special.str_show()))  == NULL)
      {
         holder.sprintf("Special %s on inclination %s doesn't exist.", 
                          the_special.str_show(), get_name());
         mainstruct->log_error(holder.str_show(), "set_values");
         return -1;
      }

      fill_param.primary_obj = NULL;
      fill_param.secondary_obj = NULL;
      fill_param.this_obj = the_player;

      the_environment.trig_used = "oninclination";
      the_environment.target_str = get_name();
      the_environment.exec_vars = NULL;
      special_obj->run_special(the_player, &fill_param, 
                                               &the_environment);
   }

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Inclination::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Inclination::get_mem_size_dynamic()
{
   int  size = 0;
 
   size += description.get_mem_size_dynamic();   
   size += the_special.get_mem_size_dynamic();   
   return size;
}


#endif
