// ---------------------------------------------------------------------------
// - Instance.cpp                                                            -
// - aleph engine - aleph instance class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Method.hpp"
#include "Closure.hpp"
#include "Instance.hpp"
#include "Exception.hpp"

namespace aleph {
  // the instance eval quarks
  static const long QUARK_THIS  = String::intern ("this");
  static const long QUARK_META  = String::intern ("meta");
  static const long QUARK_INIT  = String::intern ("initialize");
  static const long QUARK_SUPER = String::intern ("super");

  // create a new instance with a class and some arguments

  Instance::Instance (Runnable* robj, Nameset* nset, Cons* args, Class* cls) {
    // save the meta class
    if (cls == nilp)
      throw Exception ("meta-error", "invlid nil meta class with instance");
    p_class = cls;
    Object::iref (cls);
    p_super = nilp;
    d_const = false;
    // create the instance local set
    Object::iref (p_iset = new Localset);
    // bind the this symbol
    p_iset->symcst (QUARK_THIS, this);
    // bind the default symbols
    const Qarray& mdata = cls->getmdata ();
    if (mdata.length () != 0) {
      long len = mdata.length ();
      for (long i = 0; i < len; i++)
	p_iset->symdef (mdata.get (i), (Object*) nilp);
    }
    // find the initial form
    Object* obj  = p_class->find (QUARK_INIT);
    Object* form = (obj == nilp) ? nilp : obj->eval (robj, nset);
    if (form != nilp) {
      try {
	p_iset->setparent (nset);
	Object::cref (form->apply (robj, p_iset, args));
	p_iset->setparent ((Nameset*) nilp);
      } catch (...) {
	Object::iref (this);
	Object::dref (p_iset);
	Object::dref (p_class);
	Object::dref (p_super);
	Object::tref (this);
	throw;
      }
    }
    Object::iref (this);
    p_iset->remove (QUARK_THIS);
    Object::tref (this);
  }

  // destroy this instance

  Instance::~Instance (void) {
    Object::dref (p_iset);
    Object::dref (p_class);
    Object::dref (p_super);
  }

  // return the class name

  String Instance::repr (void) const {
    return "Instance";
  }

  // make this instance a shared object

  void Instance::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_class != nilp) p_class->mksho ();
    if (p_super != nilp) p_super->mksho ();
    if (p_iset  != nilp) p_iset->mksho  (); 
  }

  // set the instance super value

  Object* Instance::setsuper (Object* object, const bool flag) {
    if (d_const == true)
      throw Exception ("const-error", "const violation with super member");
    Object::iref (object);
    Object::dref (p_super);
    p_super = object;
    d_const = flag;
    return object;
  }

  // create a const symbol in the instance local set

  Object* Instance::cdef (Runnable* robj, Nameset* nset, const long quark,
			  Object* object) {
    if (quark == QUARK_SUPER) return setsuper (object, true);
    return p_iset->cdef (robj, nset, quark, object);
  }

  // bind a symbol in the instance local set

  Object* Instance::vdef (Runnable* robj, Nameset* nset, const long quark,
			  Object* object) {
    // check for super
    if (quark == QUARK_SUPER) return setsuper (object, false);
    // look in the instance local set
    Object* obj = p_iset->find (quark);
    if (obj != nilp) return obj->vdef (robj, nset, object);
    // look in the class
    obj = p_class->find (quark);
    if (obj != nilp) return obj->vdef (robj, nset, object);
    // bind locally
    return p_iset->vdef (robj, nset, quark, object);
  }
  
  // evaluate an instance member

  Object* Instance::eval (Runnable* robj, Nameset* nset, const long quark) {
    // check for super 
    if (quark == QUARK_SUPER) return p_super;
    // check for meta
    if (quark == QUARK_META) return p_class;
    // check in the instance local set
    Object* obj = p_iset->find (quark);
    if (obj != nilp) {
      Object* result = obj->eval (robj, nset);
      if (dynamic_cast <Closure*> (result) == nilp) return result;
      return new Method (result, this);
    }
    // check in the class
    obj = p_class->find (quark);
    if (obj != nilp) {
      Object* result = obj->eval (robj, nset);
      if (dynamic_cast <Closure*> (result) == nilp) return result;
      return new Method (result, this);
    }
    // check in the super instance
    if (p_super != nilp) {
      Object* result = p_super->eval (robj, nset, quark);
      if (dynamic_cast <Closure*> (result) == nilp) return result;
      return new Method (result, this);
    }
    // look in the object
    Object* result = Object::eval (robj, nset, quark);
    if (dynamic_cast <Closure*> (result) == nilp) return result;
    return new Method (result, this);
  }

  // evaluate an object with a set of arguments by quark

  Object* Instance::apply (Runnable* robj, Nameset* nset, const long quark,
			   Cons* args) {
    Object* method = eval (robj, nset, quark);
    return apply (robj, nset, method, args);
  }
    
  // evaluate an object within this instance

  Object* Instance::apply (Runnable* robj, Nameset* nset, Object* object,
			   Cons* args) {
    // basic reject - as usual
    if (object == nilp) return nilp;
    // rebind the local set before the call
    Localset* lset = new Localset (p_iset);
    lset->setparent (nset);
    lset->symcst    (QUARK_THIS, this);
    Object* result = nilp;
    // let's make the call
    try {
      result = object->apply (robj, lset, args);
      lset->clear ();
      delete lset;
    } catch (...) {
      lset->clear ();
      delete lset;
      throw;
    }
    return result; 
  }
}
