// ---------------------------------------------------------------------------
// - NameTable.cpp                                                           -
// - standard object library - name table class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "NameTable.hpp"
#include "Exception.hpp"

namespace aleph {
  
  // the name table node
  struct s_node {
    // the object quark
    long d_quark;
    // the object 
    Object* p_object;
    // next record in the list
    s_node* p_next;
    // simple constructor
    s_node (void) {
      d_quark  = 0;
      p_object = nilp;
      p_next   = nilp;
    }
    // simple destructor
    ~s_node (void) {
      Object::dref (p_object);
      delete p_next;
    }
  };
  
  // find a node by quark given its root node
  static inline s_node* getnode (s_node* node, const long quark) {
    // simple check as fast as we can
    if (node == nilp) return nilp;
    // loop until we have a match
    while (node != nilp) {
      if (node->d_quark == quark) return node;
      node = node->p_next;
    }
    // no node found
    return nilp;
  }
  
  // extract a node by quark given its root node . This procedure remove the
  // node if it is found and maintain the link list.
  static inline s_node* rmnode (s_node** root, const long quark) {
    s_node* node = *root;
    // simple check as fast as we can
    if (node == nilp) return nilp;
    // first case for the root node
    if (node->d_quark == quark) {
      *root = node->p_next;
      node->p_next = nilp;
      return node;
    }
    // loop until we have a match
    while (node->p_next != nilp) {
      if (node->p_next->d_quark == quark) {
	s_node* result = node->p_next;
	node->p_next = result->p_next;
	result->p_next = nilp;
	return result;
      }
      node = node->p_next;
    } 
    // no node found
    return nilp;
  }
  
  // Create a new name table
  
  NameTable::NameTable (void) {
    p_table = nilp;
  }
  
  // delete this name table but not the objects, norr the parent
  
  NameTable::~NameTable (void) {
    delete p_table;
  }

  // return the class name

  String NameTable::repr (void) const {
    return "NameTable";
  }

  // make this name table a shared object

  void NameTable::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    s_node* node = p_table;
    while (node != nilp) {
      Object* obj = node->p_object;
      if (obj != nilp) obj->mksho ();
      node = node->p_next;
    }
  }

  // set or create an object in this table
  
  void NameTable::add (const long quark, Object* object) {
    // protect the object
    Object::iref (object);
    // check for shared
    if ((p_shared != nilp) && (object != nilp)) object->mksho ();
    // look for existing symbol
    s_node* node = getnode (p_table,quark);
    if (node != nilp) {
      Object::dref (node->p_object);
      node->p_object = object;
      return;
    }
    // the node does not exist, create it 
    node           = new s_node;
    node->d_quark  = quark;
    node->p_object = object;
    node->p_next   = p_table;
    p_table          = node;
  }
  
  // add an object by name

  void NameTable::add (const String& name, Object* object) {
    add (name.toquark (), object);
  }

  // get an object by name. If the name is not found, nilp is returned
  
  Object* NameTable::get (const long quark) const {
    // look for the node and find symbol
    s_node* node = getnode (p_table, quark);
    if (node != nilp) return node->p_object;;
    return nilp;
  }

  // get an object by quark, if not found nil is returned
  
  Object* NameTable::get (const String& name) const {
    return get (name.toquark ());
  }

  // get an object by quark, if not found an exception is raised

  Object* NameTable::lookup (const long quark) const {
    // look for the node and find symbol
    s_node* node = getnode (p_table,quark);
    if (node != nilp) return node->p_object;;
    throw Exception ("name-error", "name not found", String::qmap (quark));
  }
  
  // get an object by name or throw an exception

  Object* NameTable::lookup (const String& name) const {
    return lookup (name.toquark ());
  }

  // return true if a quark exists in this table

  bool NameTable::exists (const long quark) const {
    // look for the node and find symbol
    s_node* node = getnode (p_table,quark);
    if (node != nilp) return true;
    return false;
  }
  
  // return true if the name exists in the table

  bool NameTable::exists (const String& name) const {
    return exists (name.toquark ());
  }

  // remove an object by quark
  
  void NameTable::remove (const long quark) {
    // extract the node
    s_node* node = rmnode (&p_table,quark);
    delete node;
  }

  // remove an object by name
  
  void NameTable::remove (const String& name) {
    remove (name.toquark ());
  }

  // clear this name table

  void NameTable::clear (void) {
    delete p_table;
    p_table = nilp;
  }
}
