// ---------------------------------------------------------------------------
// - Thread.hpp                                                              -
// - standard object library - thread class definition                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_THREAD_HPP
#define  ALEPH_THREAD_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

#ifndef  ALEPH_RUNNABLE_HPP
#include "Runnable.hpp"
#endif

namespace aleph {

  /// The Thread class is the thread management class. When a thread is 
  /// created, an instance of a Runnable object is ready to be run. Calling 
  /// the constructor will automatically invoke the run method.
  /// @author amaury darsch

  class Thread : public virtual Object  {
  public:
    /// the type of thread
    enum t_tmode {NORMAL, DAEMON};

  private:
    /// the runnable object
    Runnable* p_robj;
    /// the thread id
    void* p_tid;
    /// the thread mode
    t_tmode d_mode;

  public:
    /// create a new thread by calling the run method
    /// @param mode the thread mode
    /// @param robj the parallel object
    Thread (t_tmode mode, Runnable* robj);

    /// destroy this thread
    ~Thread (void);

    /// @return the class name
    String repr (void) const;

    /// @return true if the master thread
    static bool ismaster (void);

    /// set the main thread runnable
    /// @param robj the runnable object
    static void setrobj (Runnable* robj);

    /// @return the thread runnable object
    static Runnable* getrobj (void);

    /// wait for one thread to terminate
    /// @param thr the thread to wait for
    static void wait (const Thread& thr);

    /// wait for all thread to terminate 
    static void waitall (void);

    /// make the current thread terminate itself
    static void exit (void);

    /// evaluate a thread member by quark
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to evaluate
    Object* eval (Runnable* robj, Nameset* nset, const long quark);

    /// apply this thread object with a set of arguments and a quark
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  private:
    // make the copy constructor private
    Thread (const Thread&);
    // make the assignment operator private
    Thread& operator = (const Thread&);
  };
}

#endif
