# $Id: Logging.pm,v 1.15 2003/08/26 09:39:30 bengen Exp $

#
# AMAVIS logging module
#

# This module honors the following configuration directives:
#
# In section [Logging]: 
#  syslog           facility (e.g. mail|info) or no/off
#  syslog loglevel  (between 0 and 7 -- for now)
#  logfile          (e.g. /var/log/amavis.log)
#  logfile loglevel (same as syslog loglevel)

package AMAVIS::Logging;

use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use POSIX qw(strftime geteuid setuid uname setsid);
use POSIX ":sys_wait_h";
use Fcntl qw(:flock);
use Sys::Syslog qw(:DEFAULT setlogsock);
use IO::File;

require Exporter;
@AMAVIS::Logging::ISA = qw(Exporter);

# Currently these symbols are just taken from syslog(3). This may
# change in the future.
@AMAVIS::Logging::EXPORT = qw( LOG_EMERG LOG_ALERT LOG_CRIT LOG_ERR LOG_WARNING
	      LOG_NOTICE LOG_INFO LOG_DEBUG writelog);

sub LOG_EMERG {0}
sub LOG_ALERT {1}
sub LOG_CRIT {2}
sub LOG_ERR {3}
sub LOG_WARNING {4}
sub LOG_NOTICE {5}
sub LOG_INFO {6}
sub LOG_DEBUG {7}

use vars qw(
	    $cfg_syslog
	    $cfg_syslog_loglevel
	    $cfg_logfile
	    $cfg_logfile_loglevel

	    $log_fd

	    @LOG_PRIORITY
	   );

@LOG_PRIORITY=('emerg',
	       'alert',
	       'crit',
	       'err',
	       'warning',
	       'notice',
	       'info',
	       'debug');

# Log routines
# Need work. Quite a bit.

# Logging setup
sub init_logging {
  my $args = shift;
  $$args{'log'} = '';
  $cfg_syslog = ($AMAVIS::cfg->val('logging', 'syslog') or 'no');

  if ($cfg_syslog !~ /^ *no *$|^ *off$ *$/ ) {
    $cfg_syslog_loglevel = $AMAVIS::cfg->val('logging', 'syslog loglevel');

    if (not defined $cfg_syslog_loglevel) {$cfg_syslog_loglevel = 1};
    setlogsock('unix');
    openlog('amavis', 'pid,ndelay,cons,nowait', $cfg_syslog);
  }

  $cfg_logfile = $AMAVIS::cfg->val('logging', 'logfile');
  $cfg_logfile_loglevel = $AMAVIS::cfg->val('logging', 'logfile loglevel');

  if ($cfg_logfile) {
    if ($cfg_logfile =~ /^\s*stderr\s*$/i) {
      $log_fd = IO::File->new(">&STDERR")
	or die "Error opening STDERR for logging: $!\n";
    }
    else {
      $log_fd = IO::File->new(">> $cfg_logfile")
	or die "Error opening '$cfg_logfile' for logging: $!\n";
    }
  }

  if ((not defined $log_fd) && 
      (!$cfg_syslog or ($cfg_syslog =~ /^\s*(no|off|)\s*$/i))) {
    # will have to use crystal ball for debugging...
    warn "Warning: No logging enabled.\n";
  }
  else {
    writelog($args, LOG_DEBUG, "Logging system initialized");
  }

  # Set "signals" for warn and die statements
  $SIG{__WARN__} = sub {
    foreach my $line (split(/\n/, $_[0])) {
      writelog(undef,LOG_WARNING, '***WARN*** '.$line);
    }
  };

  $SIG{__DIE__} = sub {
    die @_ if $^S;

    foreach my $line (split(/\n/, $_[0])) {
      writelog(undef,LOG_EMERG, '***EMERG*** '.$line);
    }
  };

}

# Write message to log file.
sub writelog {
  my $args = shift;
  my $level = shift;
  my $errmsg = shift;

  chomp $errmsg;
  $errmsg =~ s/[\000-\037]//g;

  my $datestamp = strftime("%b %e %H:%M:%S", localtime);
  my $hostname = (uname())[1];
  my $line = "$datestamp $hostname amavis[$$]: $errmsg\n";

  if (defined $args) {
    $$args{'log'}.= $line;
  }

  if ($cfg_syslog !~ /^ *no *$|^ *off$ *$/ ) {
    # syslog(LOG_PRIORITY[$cfg_syslog_loglevel], $errmsg);
    # Argh. This is debug code only...
    # print STDERR "$level\nmail|".$LOG_PRIORITY[$level]."\n\t$errmsg\n";
    if ($errmsg !~ /^\s*$/ && $level <= $cfg_syslog_loglevel) {
      syslog('mail|'.$LOG_PRIORITY[$level] ,$errmsg);
    }
  }

  if (defined $log_fd) {
    if ($errmsg !~ /^\s*$/ && $level <= $cfg_logfile_loglevel) {
      flock $log_fd, LOCK_EX;
      print($log_fd $line);
      flock $log_fd, LOCK_UN;
    }
  }
}

1;
