# $Id: Milter.pm,v 1.14 2003/10/09 05:07:01 bengen Exp $

#
# MTA module for Milter setup, using external C program amavis-milter
#

package AMAVIS::MTA::Milter;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;

# For receiving mail
use IO::Socket;

use POSIX qw(setsid);

use vars qw(
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line

	    $cfg_daemon
	    $cfg_pidfile
	    $cfg_umask

	    $cfg_amavis_socket

	    $cfg_amavis_milter
	    $cfg_amavis_milter_pidfile

	    $cfg_amavis_milter_debug
	    $cfg_amavis_milter_logfile

	    $cfg_milter_socket

	    $cfg_sendmail_binary
	    $cfg_sendmail_args

	    $hostname

	    $server
	    $conn

	    $saved_args
	    $mta_result

	    $server_pid
	    $running
	    $signame
	   );

sub init {
  my $self = shift;
  my $args = shift;

  $cfg_daemon = $AMAVIS::cfg->val('Milter', 'daemon');
  $cfg_pidfile = $AMAVIS::cfg->val('Milter', 'pidfile');
  $cfg_umask = ($AMAVIS::cfg->val('Milter', 'umask') || '000');

  $cfg_amavis_socket=$AMAVIS::cfg->val('Milter', 'amavis socket');
  if (! defined $cfg_amavis_socket) {
    writelog($args,LOG_CRIT, __PACKAGE__.
	     ": Milter socket for AMaViS not specified");
    return 0;
  }
  $cfg_milter_socket=$AMAVIS::cfg->val('Milter', 'milter socket');
  if (! defined $cfg_milter_socket) {
    writelog($args,LOG_CRIT, __PACKAGE__.
	     ": Milter socket for client not specified");
    return 0;
  }
  $cfg_amavis_milter=($AMAVIS::cfg->val('Milter', 'amavis milter') ||
		      '/usr/sbin/amavis-milter');
  if  (! -x $cfg_amavis_milter) {
    writelog($args,LOG_CRIT,__PACKAGE__.
	     ": $cfg_amavis_milter not executable");
    return 0;
  }

  $cfg_sendmail_binary=($AMAVIS::cfg->val('Milter', 'sendmail') ||
			'/usr/sbin/sendmail');
  if  (! -x $cfg_sendmail_binary) {
    writelog($args,LOG_CRIT,__PACKAGE__.
	     ": $cfg_sendmail_binary not executable");
    return 0;
  }
  $cfg_sendmail_args=$AMAVIS::cfg->val('Milter', 'args');

  $cfg_amavis_milter_pidfile=$AMAVIS::cfg->val('Milter', 
					       'amavis-milter pidfile');
  $cfg_amavis_milter_debug=$AMAVIS::cfg->val('Milter', 'amavis-milter debug');
  $cfg_amavis_milter_logfile=$AMAVIS::cfg->val('Milter', 
					       'amavis-milter logfile');

  $hostname=hostname();

  my $pid;
  if ((defined $cfg_daemon) && $cfg_daemon eq 'yes') {
    if (!defined ($pid = fork)) {
      writelog($args,LOG_ERR, __PACKAGE__.": fork() failed.");
      return 0;
    }
    # If all went well...
    if ($pid) {
      # We are the parent
      # The parent will (should) only have children that have been
      # forked for accepting socket connections.
      exit 0;
    }
    # We are the child.
    # So we become a daemon.
    setsid();
    chdir("/");
    (open(STDIN, "< /dev/null") &&
     open(STDOUT, "> /dev/null") &&
     open(STDERR, "> /dev/null")) || do {
       writelog($args,LOG_ERR,__PACKAGE__.
		": Error closing stdin, stdout, or stderr: $!");
       return 0;
     };
  }

  $server_pid=$$;

  if (-e $cfg_amavis_socket) {
    unlink $cfg_amavis_socket || do {
      writelog($args,LOG_ERR,__PACKAGE__.
	       ": Could not remove Socket $cfg_amavis_socket: $!");
      return 0;
    };
  }

  $server = IO::Socket::UNIX->new(Local=>$cfg_amavis_socket, 
				  Listen=>20,
				  Type=>SOCK_STREAM) or do {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to create socket: $!");
    return 0;
  };

  # Start amavis-milter

  if (-e $cfg_amavis_milter_pidfile) {
    $self->kill_amavis_milter($args);
  }

  # NOTE: Order of parameters is important, since setting the logfile
  # in amavis-milter is done in the argument parsing loop, but
  # amavis-milter already tries to log stuff in this loop. Furrfu!
  #
  # amavis-milter neeeds to die. Soon.
  my @command = ($cfg_amavis_milter);
  if ((defined $cfg_amavis_milter_debug) &&
      (defined $cfg_amavis_milter_logfile)) {
    push (@command, 
	  '-l', $cfg_amavis_milter_logfile,
	  '-d', $cfg_amavis_milter_debug);
  }
  push (@command, '-D', '-x',
	'-p', $cfg_milter_socket,
	'-a', $cfg_amavis_socket,
	'-r', $AMAVIS::cfg_unpack_dir);

  $ENV{'BASH_ENV'}='';
  $ENV{'PATH'}='';
  writelog($args,LOG_DEBUG, __PACKAGE__.
	   ": Starting ".join(' ',@command));
  my $oldumask = umask $cfg_umask;
  my $ret=system(@command);
  if ($ret != 0) {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": Unable to start $cfg_amavis_milter auxiliary program. ".
	     "Error code: ".($ret>>8));
    return 0;
  }
  umask $oldumask;

  # create PID file.
  my $pidfile=IO::File->new(">$cfg_pidfile");
  unless (defined $pidfile) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to create PID file.");
    return 0;
  }
  else {
    $pidfile->print("$$\n");
    $pidfile->close;
  }
  $0='amavisd';

  $SIG{TERM} = \&inthandler;

  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  if ($$ == $server_pid) {
    writelog($args,LOG_DEBUG,__PACKAGE__.
	     ": Received SIG$signame. Cleaning up.");
    $self->kill_amavis_milter($args);
    unlink $cfg_pidfile;
  }
  return 1;
}

# Create temp dir and write mail
sub get_directory($) {
  my $self = shift;
  my $args = shift;

  writelog($args,LOG_DEBUG, "Waiting for connection.");
  $running = 1;
  # Main loop for accepting amavisd connections
  while(eval {$conn=$server->accept();}) {
    my $pid;
    writelog($args,LOG_DEBUG, "Accepting connection.");
    if (!defined ($pid = fork)) {
      writelog($args,LOG_ERR, "fork() failed.");

      # FIXME: Tell the client something?
      $conn->send("1",0);
      $conn->shutdown(2);
      $conn->close();
      next;
    }
    # If all went well...
    if ($pid) {
      # We are the parent
      $SIG{CHLD} = 'IGNORE';
      writelog($args,LOG_DEBUG, __PACKAGE__.
	       ": fork() successful, child's PID=$pid.");
      $conn->close();
      writelog($args,LOG_DEBUG, "Continuing ...");
      next;
    }
    else {
      # We are the child
      # Our children won't be automatically reaped.
      $SIG{CHLD} = 'DEFAULT';
      $SIG{TERM} = 'IGNORE';

      # Make sure that no result has been set for the message.
      undef $mta_result;

      # email.txt will be written here.
      handle_client_connection($conn,$args) || do {
	writelog($args,LOG_ERR,__PACKAGE__.
		 ": Something failed. Forget about all this.");
	$conn->shutdown(2);
	$conn->close();
	die;
      };

      # Return successfully
      return 1;
    }
  }

  # We reach this point only if there was a problem in the accept loop. 
  $$args{'directory'}='END';
  return 0;
}

# Called from within AMAVIS.pm to continue message delivery
sub accept_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");

  $conn->send(0,0);
  $conn->shutdown(2);
  $conn->close();

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to throw message away
sub drop_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Dropping message");

  $conn->send(99,0);
  $conn->shutdown(2);
  $conn->close();

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to freeze message delivery
sub freeze_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  $conn->send(1,0);
  $conn->shutdown(2);
  $conn->close();

  return 1;
}

# Called from Notify::*.pm, i.e. for sending warining messages
sub send_message( $$$ ) {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my @sendmail_args;

  push @sendmail_args, split(/\s+/,$cfg_sendmail_args);
  push @sendmail_args, $sender;
  push @sendmail_args, @recipients;

  writelog($args,LOG_DEBUG, __PACKAGE__.": Running $cfg_sendmail_binary ".
	   join(' ',@sendmail_args));

  open(MAIL, "|-") || exec($cfg_sendmail_binary, @sendmail_args);
  print MAIL $message;
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": $cfg_sendmail_binary @sendmail_args exited with ".($?>>8));
    return 0;
  }


  # Return successfully
  return 1;
}

sub handle_client_connection {
  my $conn = shift;
  my $args = shift;

  my $from=undef;
  my @to=();

  my $yval="\1";

  my $ret;
  my $inbuff;
  my $TEMPDIR;
  my @LDAARGS;

  $ret = $conn->recv($inbuff, 8192, 0);
  $TEMPDIR = $inbuff;
  # FIXME! Untainting should be more careful.
  if ($TEMPDIR =~ /^(\Q$AMAVIS::cfg_unpack_dir\E\/amavis-unpack-.*)$/) {
    # untaint the directory option...
    $$args{'directory'} = $1;
    mkdir "$$args{'directory'}/parts", 0777 or do {
      writelog($args,LOG_ERR,__PACKAGE__.
	       "Couldn't create directory $$args{'directory'}/parts: $!");
      return 0;
    };
    my $fh=IO::File->new("<$$args{'directory'}/email.txt") or do {
      writelog($args,LOG_ERR,__PACKAGE__.
	       "Couldn't open $$args{'directory'}/email.txt: $!");
      return 0;
    };
  } else {
    writelog($args,LOG_ERR,__PACKAGE__.": Invalid directory $TEMPDIR");
    return 0;
  }
  $conn->send($yval, 0) or do {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": Failed to send response to client: $!");
  };
  $ret = $conn->recv($inbuff, 8192, 0);
  $$args{'sender'} = $inbuff;
  $$args{'sender'} = "<>" if (!$$args{'sender'});

  $conn->send($yval, 0) or do {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": Failed to send response to client: $!");
  };

  my $outvar = \@{$$args{'recipients'}};
  while (1) {
    $ret = $conn->recv($inbuff, 8192, 0);
    last if ($inbuff eq "\3");

    ($inbuff eq "\2") ? $outvar = \@LDAARGS : push(@$outvar, $inbuff);

    $conn->send($yval, 0) or do {
      writelog($args,LOG_ERR,__PACKAGE__.
	       ": Failed to send response to client: $!");
    };
  }

  writelog($args,LOG_DEBUG, "Sender: $$args{'sender'}");
  writelog($args,LOG_DEBUG, "Recipient(s): "
	   .join (' ',@{$$args{'recipients'}}));


  my $fh=IO::File->new("$$args{'directory'}/email.txt") or do {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": Could not open $$args{'directory'}/email.txt: $!");
    return 0;
  };
  while (<$fh>) {
    last if /^ *$/;
    $$args{'headers'}.=$_;
  }
  $fh->seek(0,0);
  $$args{'filehandle'} = $fh;

  # Return successfully
  return 1;
}

sub kill_amavis_milter {
  my $self=shift;
  my $args=shift;
  my $fh=IO::File->new($cfg_amavis_milter_pidfile);
  my $milter_pid=<$fh>;
  foreach ($milter_pid) {
    /^(.*)$/ && do {
      $milter_pid=$1;
    }
  }
  writelog($args,LOG_DEBUG,__PACKAGE__.
	   ": Attempting to kill amavis-milter auxiliary program".
	   " ($milter_pid)");
  kill 'QUIT', $milter_pid;
  unlink $cfg_amavis_milter_pidfile;
}

sub inthandler {
  $signame = shift;
  $running=0;
  die "Somebody sent me a SIG$signame";
}

1;
