# $Id: AMAVIS.pm,v 1.66 2004/07/27 13:42:45 braeucup Exp $

#
# AMAVIS main module
#

# Documentation of the message description hash:
# ----------------------------------------------
# status: $
# directory: $
# sender: $
# recipients: @
# virus: %
#   $av: @         (For each $av, found viruses are inserted here)
# policy: %
#   $policy: @     (For each $policy module, found offenses are inserted
#                   here)
# unpacked_size    (Counters for size so-far...
# unpacked_files             ... and number of files)

package AMAVIS;
use strict;
use vars qw($VERSION);
$VERSION='0.1.6.7';

require Exporter;
@AMAVIS::ISA = qw(Exporter);
# Currently these symbols are just taken from syslog(3). This may
# change in the future.
@AMAVIS::EXPORT = qw(
		     get_secure_filename cmd_pipe

		     $cfg_maxlevels
		     $cfg_maxfiles
		     $cfg_maxspace

		     $cfg_uid
		     $cfg_gid
		     $cfg_chroot
		    );

use AMAVIS::Logging;

use Config::IniFiles;
use Sys::Syslog qw(:DEFAULT setlogsock);

use POSIX qw(strftime uname getuid geteuid);
use POSIX ":sys_wait_h";

use AMAVIS::Magic;
use IO::File;
use IO::Handle;

# Message digesting modules

use vars qw(
	    $umask
	    $conffile

	    $cfg
	    $mta
	    $mta_in $mta_out 

	    @av @policy

	    $cfg_syslog

	    $cfg_syslog_facility $cfg_syslog_loglevel
	    $cfg_logfile $cfg_logfile_loglevel

	    $log_fd

	    $cfg_maxlevels
	    $cfg_maxfiles
	    $cfg_maxspace

	    $log

	    @extractors
	    %type_extractors
	    @notifiers

	    $magic
	    $cfg_magicfile

	    $cfg_uid
	    $cfg_gid
	    $cfg_chroot

	    $cfg_unpack_dir
	    $cfg_quarantine_dir
	    $cfg_problem_dir

	    $cfg_cleanup

	    $cfg_unpack_error_notify
	   );


$conffile = '/etc/amavis-ng/amavis.conf';

sub init {
  my $self = shift;
  my $args = shift;
  # Read config file
  my %options = @_;

  if (defined $options{'configfile'}) {
    $conffile=$options{'configfile'}
  }

  $cfg = new Config::IniFiles(-file => "$conffile",
			      -nocase =>1) or do {
    die("Unable to open config file $conffile: ".
	join "\n",@Config::IniFiles::errors);
  };

  AMAVIS::Logging::init_logging();
  writelog($args,LOG_INFO,"Starting AMaViS $VERSION");

  $umask = $cfg->val('global', 'umask');
  if (defined $umask) {
    umask $umask;
  }
  else {
    umask '002';
  }

  $cfg_uid = ($cfg->val('security', 'uid') || $>);
  $cfg_uid = (($cfg_uid =~ /^\d+$/) ? $cfg_uid : getpwnam($cfg_uid));
  $cfg_gid = ($cfg->val('security', 'gid') || $));
  $cfg_gid = (($cfg_gid =~ /^\d+$/) ? $cfg_gid : getgrnam($cfg_gid));

  $cfg_chroot = ($cfg->val('security', 'chroot') || '');

  $cfg_maxlevels = ($cfg->val('security', 'maxlevels') or '20');
  $cfg_maxfiles = ($cfg->val('security', 'maxfiles') or '1000');
  $cfg_maxspace = ($cfg->val('security', 'maxspace') or '30M');

  $cfg_cleanup = ($cfg->val('paths', 'cleanup') or 'no');

  # kB...TB to bytes for maxspace
  if ($cfg_maxspace =~ /^\s*(\d+)\s*([KkMmGgTt])[Bb]?\s*$/ ) {
    my $num = $1; 
    foreach ($2) {
      /[Kk]/ && do {
	$cfg_maxspace = $num * 1024;
	last;
      };
      /[Mm]/ && do {
	$cfg_maxspace = $num * 1048576;
	last;
      };
      /[Gg]/ && do {
	$cfg_maxspace = $num * 1073741824;
	last;
      };
      /[Tt]/ && do {
	$cfg_maxspace = $num * 1099511627776;
	last;
      };
    }
  }

  $cfg_magicfile=($cfg->val('global','magic file')
		  || '/usr/share/amavis-ng/magic.mime');

  # Unpack error handling
  $cfg_unpack_error_notify = ($cfg->val('MIME', 'error')
		     || $cfg->val('MIME', 'error action'));

  # Unpacking directory
  $cfg_unpack_dir = ($cfg->val('paths', 'unpack')
		     ||$cfg->val('paths', 'unpack dir'));
  unless (defined $cfg_unpack_dir) {
    $cfg_unpack_dir = '/tmp';
    writelog($args,LOG_WARNING, __PACKAGE__.
	     ': no unpack dir specified, using /tmp');
  }
  # Quarantine path for rejected messages
  $cfg_quarantine_dir = ($cfg->val('paths', 'quarantine') ||
			 $cfg->val('paths', 'quarantine dir') || 
			 '/var/spool/amavis-ng/quarantine');
  # Quarantine path for messages that we had problems unpacking with
  $cfg_problem_dir = ($cfg->val('paths', 'problem') ||
		      $cfg->val('paths', 'problem dir') ||
		      '/var/spool/amavis-ng/problems');
  unless (defined $cfg_problem_dir) {
    if (defined $cfg_quarantine_dir) {
      $cfg_problem_dir = $cfg_quarantine_dir;
    }
    else {
      writelog($args,LOG_WARNING, __PACKAGE__.
	     " : no problem dir specified, using unpack dir $cfg_unpack_dir");
      $cfg_problem_dir = $cfg_unpack_dir;
    }
  }

  # Make sure that paths don't end with slash.
  foreach ($cfg_unpack_dir,
	   $cfg_quarantine_dir,
	   $cfg_problem_dir,
	   $cfg_chroot) {
    /^(.*?)\/+$/ && do {
      $_=$1;
    };
  }

  # Set up file type detection.
  $magic=AMAVIS::Magic->new($cfg_magicfile) or do {
    writelog($args,LOG_CRIT, __PACKAGE__.
	     ": Couldn't init AMAVIS::Magic (File::MMagic)");
    die __PACKAGE__.": Couldn't init AMAVIS::Magic (File::MMagic)";
  };
  # for the time being: Work around missing methods if necessary.
  eval {
    $magic->removeSpecials();
    $magic->removeFileExts();
  };
  if ($@) {
    writelog($args,LOG_DEBUG, "Old version of File::MMagic. $@");
    $magic->{SPECIALS} = {};
    $magic->{FILEEXTS} = {};
  }
  $magic->read_magic_file;

  # MTA initialization
  $mta = 'AMAVIS::MTA::'.$cfg->val('global', 'mail-transfer-agent');
  eval "use $mta $VERSION;1" or do {
    writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't load $mta");
    die __PACKAGE__.": Couldn't load $mta: $!";
  };
  $mta->init($args) or do {
    writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't init $mta");
    die __PACKAGE__.": Couldn't init $mta: $!";
  };

  # Used for get_secure_filename
  $$args{'filename_counter'}=0;

  # Extractors initialization
  @extractors=split / *, */,$cfg->val('global','extractors');
  unless (defined $extractors[0]) {
    shift @extractors;
  }
  foreach my $extractor (@extractors) {
    $extractor = 'AMAVIS::Extract::'.$extractor;
    eval "use $extractor $VERSION;1" or do {
      writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't load $extractor");
      die __PACKAGE__.": Couldn't load $extractor: $!";
    };
    $extractor->init($args,\%type_extractors) or do {
      writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't init $extractor");
      die __PACKAGE__.": Couldn't init $extractor: $!";
    };
  }

#   # Policy filters initialization
#   @policy = $cfg->val('global', 'policy');
#   unless (defined $policy[0]) {
#     shift @policy;
#   }
#   foreach my $policy (@policy) {
#     $policy = 'AMAVIS::Policy::'.$policy;
#     eval "use $policy $VERSION;1" or do {
#       writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't load $policy");
#       die __PACKAGE__.": Couldn't load $policy";
#     };
#     $policy->init($args) or do {
#       writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't init $policy");
#       die __PACKAGE__.": Couldn't init $policy";
#     };
#   }

  # Virus scanners initialization
  @av = $cfg->val('global', 'virus-scanner');
  unless (defined $av[0]) {
    shift @av;
  }
  foreach my $av (@av) {
    $av =~ /^\s*(.*)\s*$/;
  }
  foreach my $av (@av) {
    $av = 'AMAVIS::AV::'.$av;
    eval "use $av $VERSION;1" or do {
      writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't load $av");
      die __PACKAGE__.": Couldn't load $av: $!";
    };
    $av->init($args) or do {
      writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't init $av");
      die __PACKAGE__.": Couldn't init $av: $!";
    };
  }

  @notifiers = split (/ *, */, ($cfg->val('global','notifiers') || ''));
  unless (defined $notifiers[0]) {
    shift @notifiers;
  }
  foreach my $notifier (@notifiers) {
    $notifier = 'AMAVIS::Notify::'.$notifier;
    eval "use $notifier $VERSION;1" or do {
      writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't load $notifier");
      die __PACKAGE__.": Couldn't load $notifier: $!";
    };
    $notifier->init($args) or do {
      writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't init $notifier");
      die __PACKAGE__.": Couldn't init $notifier: $!";
    };
  }

  unless (defined @{$$args{'virus_scanners'}}) {
    push @{$$args{'virus_scanners'}}, '(none)';
  }

  my $uid = (getpwuid(getuid()) || "unknown");
  my $euid = (getpwuid(geteuid()) || "unknown");

  # Make sure that PATH is not tainted.
  $ENV{PATH} = "/bin:/usr/bin";
  delete @ENV{qw(IFS CDPATH ENV BASH_ENV)};   # Make %ENV safer

  writelog($args,LOG_DEBUG, __PACKAGE__.
           ": Running as UID/EUID $uid(".getuid().")/$euid(".geteuid().")");

  writelog($args,LOG_DEBUG, __PACKAGE__." initialized");
  return 1;
}

# "Universal" extraction routine
# Determine mime type of each extracted file, then call extract
# routine for that mime type
sub extract {
  my $self=shift;
  my $args=shift;

  my $newfilesfound;
  my $level;

  for ($level=0; $level<=$cfg_maxlevels; $level++) {
    $newfilesfound=0;
    foreach my $filename (keys % {$$args{'contents'}}) {
      unless (defined $ {$ {$$args{'contents'}}{$filename}}{'type'}) {
	# The filetype isn't known yet, this means the file is new.
	# First determine the filetype, then.
	my $filetype=$magic->checktype_filename("$$args{'directory'}/parts/$filename");
	my $insecure_filetype = $ {$ {$$args{'contents'}}{$filename}}{insecure_type};

	# If an "insecure filetype" (from a MIME message) exists, do
	# we believe the MIME structure? We should, in some cases,
	# otherwise bounces that get sent back as MIME messages will
	# result in errors while unpacking.
	if (defined $ {$ {$$args{'contents'}}{$filename}}{insecure_type}) {
	  foreach ($ {$ {$$args{'contents'}}{$filename}}{insecure_type}) {
	    (/message\/rfc-?822-headers/
	     || /message\/delivery-status/
	     || /message\/partial/
	     || /text\/plain/) && do {
	       $filetype = $ {$ {$$args{'contents'}}{$filename}}{insecure_type};
	     };
	  }
	}

	# Store the filetype.
	$ {$ {$$args{'contents'}}{$filename}}{'type'} = $filetype;
	writelog($args,LOG_INFO, __PACKAGE__.
		 ": Determined $filename to be type $filetype");
	if (defined $type_extractors{$filetype}) {
	  # We know how to unpack this sucker...
	  my $res=$type_extractors{$filetype}->extract($args, $filename);
	  unless ($res) {
	    writelog($args,LOG_ERR, __PACKAGE__.
		     ": Error while unpacking $filename as $filetype");
	    if ((defined $insecure_filetype) && 
		(defined $type_extractors{$insecure_filetype})) {
	      writelog($args,LOG_INFO, __PACKAGE__.
		       ": Attempting to unpack $filename as ".
		       "$insecure_filetype");
	      $res=$type_extractors{$insecure_filetype}->extract($args,
								 $filename);
	      unless ($res) {
		writelog($args,LOG_ERR, __PACKAGE__.
			 ": Error while unpacking $filename as ".
			 "$insecure_filetype. Giving up.");
		return 0;
	      }
	    }
	    else {
	      writelog($args,LOG_ERR, __PACKAGE__.
		       ": Giving up");
	      return 0;
	    }
	  }
	  $newfilesfound=1;
	}
	else {
	  writelog($args,LOG_INFO, "Not attempting to unpack $filename");
	}
      }
    }
    last unless ($newfilesfound);
  }
  if ($level >= $cfg_maxlevels) {
    writelog($args,LOG_ERR,"Mail has too many archive levels");
    return 0;
  }
  return 1;
}

sub get_secure_filename {
  my $args=shift;
  return sprintf("%.8x",$$args{'filename_counter'}++);
}

sub process_message {
  my $self = shift;
  my $args = shift;

  $$args{'status'} = 'unknown';
  # Create temporary directory and have MTA module write the message
  # there, if necessary.

  unless (defined $$args{'directory'}) {
    $mta->get_directory($args) or do {
      if ($$args{'directory'} ne 'END') {
	writelog($args,LOG_CRIT, 
		 "Error: Couldn't get directory for extracting");
      }
      return 0;
    };
  }
  writelog($args,LOG_INFO, "Unpacking message in $$args{'directory'}");

  # Check whether tmp directory is really a directory and rwx for EUID.
  unless ((-d $cfg_unpack_dir) &&
	  (-r $cfg_unpack_dir) &&
	  (-w $cfg_unpack_dir) &&
	  (-x $cfg_unpack_dir)) {
    writelog($args, LOG_CRIT,
	     "$cfg_unpack_dir is not directory with rwx permissions.");
    $mta->freeze_message($args);
    return 0;
  }

  my $securename=get_secure_filename($args);

  # Initialize unpacked_size to size of mail file...
  my @stat = stat("$$args{'directory'}/email.txt");
  $$args{'unpacked_size'} = $stat[7];
  # ...and unpacked_files to 1.
  $$args{'unpacked_files'} = 1;

  # The first file to be analyzed is the mail file
  link( "$$args{'directory'}/email.txt",
	"$$args{'directory'}/parts/$securename") 
    or do {
      writelog($args,LOG_CRIT, "Unable to link parts/$securename to email.txt");
      $mta->freeze_message($args);
      return 0;
    };

  # Add filename to contents field of message.
  $ {$$args{'contents'}}{$securename} = {};

  $self->extract($args) or do {
    writelog($args,LOG_CRIT, __PACKAGE__.
	     ": Error while unpacking message");
    if ($cfg_unpack_error_notify eq "drop+notify") {
      foreach my $notifier (@notifiers) {
        $notifier->notify_unpack($args) or writelog($args,LOG_CRIT, __PACKAGE__.
		 ": Error while notifying $notifier");
      }
      $mta->drop_message($args);
    } else {
      $mta->freeze_message($args);
    }
    return 0;
  };

#   foreach my $policy (@policy) {
#     $policy->scan($args) or
#       writelog($args,LOG_CRIT,"Error while scanning for policy");
#   }

  foreach my $av (@av) {
    eval {
      $av->scan($args)
    } or do {
      writelog($args,LOG_CRIT,"Error while scanning for viruses with $av: $@");
      $mta->freeze_message($args);
      return 0;
    };
  }

  # Decide what to do with the message
  if (defined $$args{'found_viruses'}) {
    # Log what viruses have been found.
    foreach my $scanner (@{$$args{'virus_scanners'}}) {
      writelog($args,LOG_WARNING, "$scanner found:");
      foreach my $virus (@{$$args{'found_viruses'}{$scanner}}) {
	writelog($args,LOG_WARNING, " $virus");
      }
    }

    # Tell MTA module to drop the message
    $mta->drop_message($args);

    # Quarantine message
    if (defined $cfg_quarantine_dir) {
      $self->quarantine_message($args, $cfg_quarantine_dir) or do {
        writelog($args,LOG_CRIT, __PACKAGE__.
  	       ": Error while quarantining message");
      };
    }

    # Complain to everybody who cares if policy check failed or a
    # virus was found
    foreach my $notifier (@notifiers) {
      $notifier->notify($args) or
	writelog($args,LOG_CRIT, __PACKAGE__.
		 ": Error while notifying $notifier");
    }
  }
  else {
    $mta->accept_message($args) or do {
      writelog($args,LOG_CRIT, __PACKAGE__.
	       ": Error while accepting message");
      $mta->freeze_message($args);
      return 0;
    };
  }

  # Return successfully
  return 1;
}

sub quarantine_message {
  my $self = shift;
  my $args = shift;
  my $dir = shift;
  # quarantine message

  unless (defined $dir) {
    writelog($args,LOG_DEBUG, __PACKAGE__.
	     ": No directory for quarantine specified.");
    return 0;
  }

  $$args{'quarantine_file'}=sprintf("%.8x-%.4x",time,$$);
  writelog($args,LOG_INFO,
	   'Quarantining infected message to '.
	   "$dir/$$args{'quarantine_file'}");

  my $fh=$$args{'filehandle'};
  $fh->seek(0,0);
  my $quarantinefile=IO::File->new(">$dir/$$args{'quarantine_file'}.msg");

  unless (defined $quarantinefile) {
    writelog($args,LOG_CRIT, "Quarantining failed: $!");
    return 0;
  }

  $quarantinefile->print("X-Quarantined-From: $$args{'sender'}\n".
			 "X-Quarantined-To: ".
			 join(', ',@ { $$args{'recipients'}}).
			 "\n");

  while (<$fh>) {
    $quarantinefile->print($_) or do {
      writelog($args,LOG_CRIT, "Quaranining failed: $!");
      return 0;
    };
  }

  # write part of logfile to quarantine dir.
  my $logfile = IO::File->new(">$dir/".
			      "$$args{'quarantine_file'}.log");

  unless ((defined $logfile) &&
	  ($logfile->print($$args{'log'})) &&
	  ($logfile->close())) {
    writelog($args,LOG_CRIT, "Writing log to quarantine dir failed");
    return 0;
  }
  return 1;
}

sub quarantine_problem_message {
  my $self = shift;
  my $args = shift;

  my $result=$self->quarantine_message($args, $cfg_problem_dir);
  if ($result) {
    my $sender = $cfg->val('Notify', 'mail from');
    $sender = 'postmaster' unless defined($sender);
    my $error_address=$cfg->val('Notify', 'admin');
    my @error_addresses=split / *, */, $AMAVIS::cfg->val('Notify', 'admin');
    my $message=<<"EOF";
From: $sender
To: $error_address
Subject: Message put into problems directory

The message has been quarantined as
$$args{'quarantine_file'}.msg

The corresponding logfile has been written to
$$args{'quarantine_file'}.log

EOF
    $message.="\n".('-'x 72)."\nMessage headers follow:\n";
    $message.=$$args{'headers'};

    $message.="\n".('-'x 72)."\nLog file entry follows:\n";
    $message.=$$args{'log'};

    $mta->send_message($args, $message, '<>', @error_addresses);
  }
  else {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": Error while storing message in $cfg_problem_dir .");
    return 0;
  }
}

# Open file descriptor from
sub cmd_pipe {
  my $args = shift;
  my $handle = IO::Handle->new(); # _from_fd(\*HANDL, 'r');
  my $pid = open($handle, "-|");
  unless (defined $pid ) {
    writelog($args,LOG_ERR, __PACKAGE__.": Can't fork: $!");
    die "Can't fork: $!";
  }
  if ($pid) {           # parent
    return $handle;
  } else {
    # Drop privileges for fork if EUID=root
    if ($> == 0) {
      if (defined $cfg_gid) {
	writelog($args,LOG_DEBUG,__PACKAGE__.": Dropping GID");
	$)=$cfg_gid;
	if ($) != $cfg_gid) {
	  writelog($args,LOG_ERR, __PACKAGE__.": Can't drop GID to $cfg_gid");
	  die;
	}
      }
      if (defined $cfg_uid) {
	writelog($args,LOG_DEBUG,__PACKAGE__.": Dropping UID");
	$>=$cfg_uid;
	if ($> != $cfg_uid) {
	  writelog($args,LOG_ERR, __PACKAGE__.": Can't drop UID to $cfg_uid");
	  die;
	}
      }
    }
    $ENV{PATH} = "/bin:/usr/bin";
    $ENV{TERM} = 'dumb';
    exec @_
      or die "can't exec $_[0]: $!";
  }
}

# Cleanup routine
# Removes working directory
sub cleanup {
  my $self = shift;
  my $args = shift;

  if ($cfg_cleanup eq 'yes') {
    writelog($args,LOG_INFO, __PACKAGE__.": Cleaning up.");
    # Remove unpacked files, parts/ directory, email.txt, unpacking
    # directory
    foreach my $filename (map("$$args{'directory'}/parts/$_",
			      (keys % {$$args{'contents'}})),
			  "$$args{'directory'}/parts/",
			  "$$args{'directory'}/email.txt",
			  $$args{'directory'}) {
      eval{
	if (-d $filename) {
	  writelog($args,LOG_DEBUG,"Removing $filename");
	  rmdir $filename or
	    writelog($args,LOG_ERR,__PACKAGE__.
		     ": Error removing $filename: $!");
	}
	elsif (-f $filename || -l $filename) {
	  writelog($args,LOG_DEBUG,"Removing $filename");
	  unlink $filename or
	    writelog($args,LOG_ERR,__PACKAGE__.
		     ": Error removing $filename: $!");
	}
      };
    }
  }

  $mta->cleanup($args) or do {
    writelog($args,LOG_CRIT, __PACKAGE__.": Couldn't cleanup for $mta");
    return 0;
  };

  writelog($args,LOG_INFO, __PACKAGE__.": Done.");
  # Return successfully
  return 1;
}

1;
