# $Id: Maildrop.pm,v 1.4 2005/01/13 00:14:33 braeucup Exp $

#
# MTA module for use as a maildrop xfilter
# By: John Laur <johnl@blurbco.com>
# Based on AMAVIS::MTA::Exim and AMAVIS::MTA::Maildrop
#

package AMAVIS::MTA::Maildrop;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;

use vars qw(
            $cfg_sendmail_binary
	    $cfg_sendmail_args
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line
	   );

sub init {
  my $self = shift;
  my $args = shift;

  # Get the configuration for sending notificatin messages
  $cfg_sendmail_binary = $AMAVIS::cfg->val('Maildrop', 'sendmail');
  $cfg_sendmail_args = $AMAVIS::cfg->val('Maildrop', 'args');
  if ($cfg_sendmail_binary eq '') {
    writelog($args,LOG_CRIT, "Location of sendmail binary not specified");
    return 0;
  }
  if (! -x $cfg_sendmail_binary) {
    writelog($args,LOG_CRIT, "$cfg_sendmail_binary not executable");
    return 0;
  }
  
  # Get the configuration for adding a custom header to the message
  if (($AMAVIS::cfg->val('global', 'x-header') || '') eq 'true') {
    $cfg_x_header = 1
  };
  $cfg_x_header_tag = $AMAVIS::cfg->val('global', 'x-header-tag');
  $cfg_x_header_line = $AMAVIS::cfg->val('global', 'x-header-line');
  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  return 1;
}

# Create temp dir and write mail
sub get_directory {
  my $self = shift;
  my $args = shift;

  # Create temp directory. Try creating $prefix[date]-[pid] 
  # If unsuccessful after 10 tries, abort
  my $prefix = "$AMAVIS::cfg_unpack_dir/amavis-unpack-";
  my $i = 0;
  my $message_id;
  while (1) {
    $message_id = sprintf("%.8x-%.4x",time,$$);
    unless (defined mkpath ($prefix.$message_id, 0, 0770)) {
      if (++$i > 10) {
	return 0;
      }
      else {
	next;
      }
    }
    last;
  }
  $$args{'message_id'}=$message_id;
  my $directory = $prefix.$message_id;
  mkdir "$directory/parts", 0777;
  $$args{'directory'} = $directory;

  # Open message file that is later going to be disected and scanned
  my $output = IO::File->new("+>$directory/email.txt");

  # Read message from STDIN
  my $message = IO::File->new("<-");
  my $headers=1;
  $$args{'headers'}='';
  while (<$message>) {
    $headers=0 if /^ *$/;
    if ($headers==1) {
      $$args{'headers'}.=$_;

      if (/^From: *(.*)$/) {
        $$args{'sender'}=$1;
        writelog($args,LOG_DEBUG,__PACKAGE__.": Sender is $$args{'sender'}");
      }
      if (/^To: *(.*)$/) {
        @{$$args{'recipients'}}=split /\s*,\s*/,$1;
        writelog($args,LOG_DEBUG,__PACKAGE__.": Recipients are "
          .join(', ',@{$$args{'recipients'}}));
      }
    }
    print $output $_;
  }
  close($message);

  # Message file has been written, reset file pointer and put it into
  # the record.
  $output->seek(0,0);
  $$args{'filehandle'} = $output;

  # Return successfully
  return 1;
}

# Accept the message for delivery and add header if needed.
sub accept_message {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");

  #Output the message to stdout
  print "$cfg_x_header_tag: $cfg_x_header_line\n" if $cfg_x_header;
  
  while (my $line=$$args{'filehandle'}->getline()) {
    print $line;
  }

  $$args{'status'} = 'accept';
  
  # Return successfully
  return 1;
}

# Just "forget" the mail.
sub drop_message {
  my $self = shift;
  my $args = shift;
  # Doing nothing should suffice
  writelog($args,LOG_WARNING, __PACKAGE__.": Dropping message");

  # Return successfully
  return 1;
}

# Maildrop can't freeze since it's supposed to deliver the message,
# so we try to quarantine the mail. Note, that often, maildrop
# gets run as a normal user, so writing to the system quarantine
# may fail!
sub freeze_message {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  # Instead of freezing, drop the message, but store a copy in
  # $cfg_problem_dir -- analogous to AMAVIS::quarantine_message()
  if (AMAVIS->quarantine_problem_message($args)) {
    return 1;
  } else {
    writelog($args,LOG_ERR,__PACKAGE__.": Could not freeze message");
    return 0;
  }
}

# Called from Notify::*.pm, i.e. for sending warining messages
# If maildrop is running as a normal user, notifications from a 'postmaster'
# address will likely not be accepted!
sub send_message {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my @cfg_sendmail_args;

  push @cfg_sendmail_args, split(/\s+/,$cfg_sendmail_args);

  writelog($args,LOG_DEBUG, __PACKAGE__.": Running $cfg_sendmail_binary ".
    join(' ',@cfg_sendmail_args));

  open(MAIL, "|-") || exec($cfg_sendmail_binary, @cfg_sendmail_args);
  print MAIL $message;
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR,
      __PACKAGE__.": $cfg_sendmail_binary @cfg_sendmail_args exited with ".($?>>8));
  }

  # Return successfully
  return 1;
}

1;
