# $Id: Postfix.pm,v 1.12 2005/01/13 00:14:33 braeucup Exp $

#
# MTA module for Postfix setup
#
# So far, this is mostly identical to the Sendmail module. 
#
# The only setup that has been tested so far is the "simple content
# filtering example" as described in FILTER_README in the Postfix
# distribution.
#

package AMAVIS::MTA::Postfix;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;

use vars qw(
	    $cfg_postfix_binary
	    $cfg_postfix_args
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line
	   );

sub init {
  my $self = shift;
  my $args = shift;
  $cfg_postfix_binary = $AMAVIS::cfg->val('Postfix', 'postfix');
  $cfg_postfix_args = $AMAVIS::cfg->val('Postfix', 'args');
  if ($cfg_postfix_binary eq '') {
    writelog($args,LOG_CRIT, "Location of postfix binary not specified");
    return 0;
  }
  if (! -x $cfg_postfix_binary) {
    writelog($args,LOG_CRIT, "$cfg_postfix_binary not executable");
    return 0;
  }
  $cfg_postfix_binary = $AMAVIS::cfg->val('Postfix', 'postfix');
  $cfg_postfix_args = $AMAVIS::cfg->val('Postfix', 'args');
  if (($AMAVIS::cfg->val('global', 'x-header') || '') eq 'true') {
    $cfg_x_header = 1
  };
  $cfg_x_header_tag = $AMAVIS::cfg->val('global', 'x-header-tag');
  $cfg_x_header_line = $AMAVIS::cfg->val('global', 'x-header-line');
  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  return 1;
}

# Create temp dir and write mail
sub get_directory($) {
  my $self = shift;
  my $args = shift;

  # Create temp directory. Try creating $prefix[date]-[pid] 
  # If unsuccessful after 10 tries, abort
  my $prefix = "$AMAVIS::cfg_unpack_dir/amavis-unpack-";
  my $i = 0;
  my $message_id;
  while (1) {
    $message_id = sprintf("%.8x-%.4x",time,$$);
    unless (defined mkpath ($prefix.$message_id, 0, 0770)) {
      if (++$i > 10) {
	return 0;
      }
      else {
	next;
      }
    }
    last;
  }
  $$args{'message_id'}=$message_id;
  my $directory = $prefix.$message_id;
  mkdir "$directory/parts", 0777;
  $$args{'directory'} = $directory;

  # Open message file that is later going to be disected and scanned
  my $output = IO::File->new("+>$directory/email.txt");

  # Get message from STDIN
  my $headers=1;
  $$args{'headers'}='';
  while (<STDIN>) {
    if (/^ *$/) {
      $headers=0;
    }
    if ($headers==1) {
      next if (/^From /);
      $$args{'headers'}.=$_;
    }
    print $output $_;
  }

  if ($#ARGV < 1) {
    writelog($args,LOG_ERR, __PACKAGE__.": Missing arguments to postfix");
    return 0;
  }

  writelog($args,LOG_DEBUG, "Called as amavis ".join(' ',@ARGV));

  foreach (@ARGV) {
    /^-f$/ && next; # ignore "-f"
    /^-d$/ && next; # ignore "-d"
    /^(.*)$/; # untaint sender or recipient
    unless (defined $$args{'sender'}) {
      my $sender = $1;
      if ($sender =~ /^$/) {
        $sender= "<>";
      }
      writelog($args,LOG_DEBUG, "Sender: $sender");
      $$args{'sender'} = $sender;
    }
    else {
      my $recipient = $1;
      writelog($args,LOG_DEBUG, "Recipient: $recipient"); 
      push @{$$args{'recipients'}}, $recipient;
    }
  }

  # Message file has been written, reset file pointer and put it into
  # the record.

  $output->seek(0,0);
  $$args{'filehandle'} = $output;

  # Return successfully
  return 1;
}

# Generate a copy of the scanned message and pipe it to mailer.
sub accept_message($) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");
  my @postfix_args;

  push @postfix_args, split(/\s+/,$cfg_postfix_args);
  push @postfix_args, $$args{'sender'};
  push @postfix_args, @{$$args{'recipients'}};

  open(MAIL, "|-") || exec($cfg_postfix_binary, @postfix_args);
  if ($cfg_x_header) {
    print MAIL "$cfg_x_header_tag: $cfg_x_header_line\n";
  }
  while (my $line=$$args{'filehandle'}->getline()) {
    print MAIL $line;
  }
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR, __PACKAGE__.": $cfg_postfix_binary exited with ".($?>>8));
  }

  $$args{'status'} = 'accept';

  # Return successfully
  return 1;
}

# Just "forget" the mail.
sub drop_message($) {
  my $self = shift;
  my $args = shift;
  # Doing nothing should suffice
  writelog($args,LOG_WARNING, __PACKAGE__.": Dropping message");

  # Return successfully
  return 1;
}

sub freeze_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  if (AMAVIS->quarantine_problem_message($args)) {
    # Return successfully
    return 1;
  }
  else {
    writelog($args,LOG_ERR,__PACKAGE__.": Could not freeze message");
    return 0;
  }
}

# Called from Notify::*.pm, i.e. for sending warining messages
sub send_message($$) {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my @postfix_args;

  push @postfix_args, split(/\s+/,$cfg_postfix_args);
  push @postfix_args, $sender;
  push @postfix_args, @recipients;

  writelog($args,LOG_DEBUG, __PACKAGE__.": Running $cfg_postfix_binary ".
	   join(' ',@postfix_args));

  open(MAIL, "|-") || exec($cfg_postfix_binary, @postfix_args);
  print MAIL $message;
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR,
	     __PACKAGE__.": $cfg_postfix_binary @postfix_args exited with ".($?>>8));
  }

  # Return successfully
  return 1;
}

1;
