// ----------------------------------------------------------------------------
//
//  Copyright (C) 2006-2010 Fons Adriaensen <fons@kokkinizita.net>
//    
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// ----------------------------------------------------------------------------


#include <string.h>
#include <math.h>
#include "decoder.h"


const float Decoder::_g_norm [11] = { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };

const float Decoder::_g_sn3d [11] =
{
    1.0f,
    1.732051f,
    1.732051f,
    1.732051f,
    2.236068f,
    2.236068f,
    2.236068f,
    2.236068f,
    2.236068f,
    2.645751f, 
    2.645751f, 
};

const float Decoder::_g_fuma [11] =
{
    1.414214f,
    1.732051f,
    1.732051f,
    1.732051f,
    2.236068f,
    1.936492f,
    1.936492f,
    1.936492f,
    1.936492f,
    2.091650f,
    2.091650f,
};



Decoder::Decoder (void) :
    _test (0)
{
}


Decoder::~Decoder (void)
{
    delete[] _test;
}


void Decoder::init (int frate)
{
    int    i, k, f;
    float  w;

    // Create delay lines.
    _frate = frate;
    _dsize = FRAG_SIZE * ((int)(ceilf (frate / (34.0f * FRAG_SIZE))) + 1); 
    for (i = 0; i < MAXOP; i++) _chan [i]._dline = new float [_dsize]; 
    reset ();

    // Create test signal.
    k = frate / 7;
    k -= k % FRAG_SIZE;
    _tsize = k;
    _test = new float[k];
    f = (800 * k) / frate;
    w = 2 * M_PI / k;
    for (i = 0; i < k; i++) _test [i] = 0.1f * sinf (i * f * w + sinf (i * w));
}


void Decoder::reset (void)
{
    int      i;
    Channel  *X;

    // Reset everything.
    memset (_sig0, 0, FRAG_SIZE * sizeof (float));
    memset (_sig1, 0, FRAG_SIZE * sizeof (float));
    memset (_sig2, 0, FRAG_SIZE * sizeof (float));
    memset (_sig3, 0, FRAG_SIZE * sizeof (float));
    for (i = 0, X = _chan; i < MAXOP; i++, X++)
    {
	memset (X->_dline, 0, _dsize * sizeof (float)); 
	X->_delay = 0;
	X->_gcorr = 1.0f;
	X->_gain0 = 0.0f;
	X->_nffilt1.reset ();
	X->_nffilt2.reset ();
	X->_nffilt3.reset ();
    }
    for (i = 0; i < 3; i++) _nffilt1 [i].reset ();
    for (i = 0; i < 5; i++) _nffilt2 [i].reset ();
    for (i = 0; i < 2; i++) _nffilt3 [i].reset ();
    for (i = 0; i < 9; i++)
    {
        _xover [i].reset ();
	_scale [i] = 1.0f;
    }
    _nsig1 = 0;
    _nsig2 = 0;
    _nsig3 = 0;
    _nchan = 1;
    _nband = 1;
    _comp  = 0;
    _inp0  = TEST_OFF;
    _inp1  = TEST_OFF;
    _solo  = 0;
    _mute  = 0;
    _gain  = 0;
    _idel  = 0;
    _itst  = 0;
}


void Decoder::set_config (AD_conf *C)
{
    int           i, j, k;
    float         dm, da;
    const float   *A, *B;
    Speaker       *S;

    reset ();
    _nchan = C->_dec.nspkr;
    _nband = C->_dec.nband;

    // Determing number of inputs for each order.
    switch ((C->_dec.h_ord << 4) | C->_dec.v_ord)
    {
    case 0x10:
	_nsig1 = 2;
	break;
    case 0x11:
	_nsig1 = 3;
	break;
    case 0x20:
	_nsig1 = 2;
	_nsig2 = 2;
	break;
    case 0x21:
	_nsig1 = 3;
	_nsig2 = 2;
	break;
    case 0x22:
	_nsig1 = 3;
	_nsig2 = 5;
	break;
    case 0x30:
	_nsig1 = 2;
	_nsig2 = 2;
	_nsig3 = 2;
	break;
    case 0x31:
	_nsig1 = 3;
	_nsig2 = 2;
	_nsig3 = 2;
	break;
    }

    // Calculate scaling coefficients.
    switch (C->_dec.scale)
    {
    case AD_conf::SN3D: A = _g_sn3d; break;
    case AD_conf::FUMA: A = _g_fuma; break;
    default: A = _g_norm;
    }
    switch (C->_opt.scale)
    {
    case AD_conf::SN3D: B = _g_sn3d; break;
    case AD_conf::FUMA: B = _g_fuma; break;
    default: B = _g_norm;
    }
    j = 0;
    _scale [j++] = B [0] / A [0];
    k = 1;
    for (i = 0; i < _nsig1; i++, k++) _scale [j++] = B [k] / A [k];
    k = (_nsig2 == 2) ? 7 : 4;
    for (i = 0; i < _nsig2; i++, k++) _scale [j++] = B [k] / A [k];
    k = 9;
    for (i = 0; i < _nsig3; i++, k++) _scale [j++] = B [k] / A [k];

    // Set crossover filter frequency.
    if (_nband == 2)
    {
	for (i = 0; i < 1 + _nsig1 + _nsig2 + _nsig3; i++)
	{
            _xover [i].init (C->_opt.xfreq / _frate);
	}
    }

    // Find maximum and average speaker distance.
    dm = da = 0;
    for (i = 0, S = C->_speakers; i < _nchan; i++, S++)
    {
	da += S->_r;
	if (dm < S->_r) dm = S->_r;
    } 
    da /= _nchan;

    // Ininitialise delay lines.
    if (C->_opt.delay) 
    {
	_comp |= COMP_DELAY;
	for (i = 0, S = C->_speakers; i < _nchan; i++, S++)
	{
	    k = int ((dm - S->_r) * _frate / 340.0f + 0.5f);
	    if (k > _dsize - FRAG_SIZE) k = _dsize - FRAG_SIZE;
	    _chan [i]._delay = k;
	}
    }

    // Correct gains for speaker distance.
    if (C->_opt.level) 
    {
	_comp |= COMP_LEVEL;
	for (i = 0, S = C->_speakers; i < _nchan; i++, S++)
	{
	    _chan [i]._gcorr = S->_r / dm;
	}
    }

    // Initialise near field compensation.
    // If on inputs, use average distance.
    if (C->_opt.nfeff == AD_conf::INPUT) 
    {
	_comp |= COMP_NFEIP;
	for (i = 0; i < _nsig1; i++) _nffilt1 [i].init (340.0f / (da * _frate));
	for (i = 0; i < _nsig2; i++) _nffilt2 [i].init (340.0f / (da * _frate));
	for (i = 0; i < _nsig3; i++) _nffilt3 [i].init (340.0f / (da * _frate));
    }
    // If on outputs, use correct distance.
    else if (C->_opt.nfeff == AD_conf::OUTPUT) 
    {
	_comp |= COMP_NFEOP;
	for (i = 0, S = C->_speakers; i < _nchan; i++, S++)
	{
	    _chan [i]._nffilt1.init (340.0f / (S->_r * _frate));
	    _chan [i]._nffilt2.init (340.0f / (S->_r * _frate));
	    _chan [i]._nffilt3.init (340.0f / (S->_r * _frate));
	}
    }
}


void Decoder::set_matrix (AD_conf *C)
{
    int      i;
    float    g, b;
    Channel  *X;
    Matrow   *R;

    if (_nband == 2) b = powf (10.0f, C->_opt.ratio / 40.0f);
    else b = 1;

    for (i = 0, X = _chan, R = C->_lfmatrix; i < _nchan; i++, X++, R++)
    {
	g = C->_lfgain [0] / b;
	X->_c0lf = R->_w * g;
	g = C->_lfgain [1] / b;
        X->_c1lf [0] = R->_x * g;
        X->_c1lf [1] = R->_y * g;
        X->_c1lf [2] = R->_z * g;
        g = C->_lfgain [2] / b;
	if (_nsig2 == 2)
	{
	    X->_c2lf [0] = R->_u * g;
	    X->_c2lf [1] = R->_v * g;
	}
        else if (_nsig2 == 5)
	{
	    X->_c2lf [0] = R->_r * g;
	    X->_c2lf [1] = R->_s * g;
	    X->_c2lf [2] = R->_t * g;
	    X->_c2lf [3] = R->_u * g;
	    X->_c2lf [4] = R->_v * g;
	}
        g = C->_lfgain [3] / b;
	if (_nsig3 == 2)
	{
 	    X->_c3lf [0] = R->_p * g;
	    X->_c3lf [1] = R->_q * g;
	}
    }

    if (_nband == 1) return;

    for (i = 0, X = _chan, R = C->_hfmatrix; i < _nchan; i++, X++, R++)
    {
	g = C->_hfgain [0] * b;
	X->_c0hf = R->_w * g;
	g = C->_hfgain [1] * b;
        X->_c1hf [0] = R->_x * g;
        X->_c1hf [1] = R->_y * g;
        X->_c1hf [2] = R->_z * g;
	g = C->_hfgain [2] * b;
	if (_nsig2 == 2)
	{
	    X->_c2hf [0] = R->_u * g;
	    X->_c2hf [1] = R->_v * g;
	}
        else if (_nsig2 == 5)
	{
	    X->_c2hf [0] = R->_r * g;
	    X->_c2hf [1] = R->_s * g;
	    X->_c2hf [2] = R->_t * g;
	    X->_c2hf [3] = R->_u * g;
	    X->_c2hf [4] = R->_v * g;
	}
	g = C->_hfgain [3] * b;
	if (_nsig3 == 2)
	{
 	    X->_c3hf [0] = R->_p * g;
	    X->_c3hf [1] = R->_q * g;
	}
    }
}


void Decoder::process (int nf, float *ip[], float *op[], float *ts, float *lm)
{
    int       i, j, c, n, n1, n2;
    uint64_t  m;
    float     g0, g1, d, s, t;
    float     *p1, *p2, *p3;
    Channel   *C;

    while (nf)
    {
	n = (nf > FRAG_SIZE) ? FRAG_SIZE : nf;
	j = 0;

	// Scale and bandsplit W.
	p1 = ip [j];
        p2 = _ip0lf;
	p3 = _ip0hf;
	for (i = 0; i < n; i++) p2 [i] = _scale [j] * p1 [i];
	if (_nband > 1) _xover [j].process (n, p2, p2, p3);
	ip [j++] += n;

	// Scale, filter and bandsplit X,Y,Z.
	for (c = 0; c < _nsig1; c++)
	{
	    p1 = ip [j];
            p2 = &(_ip1lf [c][0]);
            p3 = &(_ip1hf [c][0]);
	    if (_comp & COMP_NFEIP) _nffilt1 [c].process (n, p1, p2, _scale [j]);
            else for (i = 0; i < n; i++) p2 [i] = _scale [j] * p1 [i];
  	    if (_nband > 1) _xover [j].process (n, p2, p2, p3);
 	    ip [j++] += n;
	}	    

	// Scale, filter and bandsplit R,S,T,U,V.
	for (c = 0; c < _nsig2; c++)
	{
	    p1 = ip [j];
            p2 = &(_ip2lf [c][0]);
            p3 = &(_ip2hf [c][0]);
	    if (_comp & COMP_NFEIP) _nffilt2 [c].process (n, p1, p2, _scale [j]);
            else for (i = 0; i < n; i++) p2 [i] = _scale [j] * p1 [i];
  	    if (_nband > 1) _xover [j].process (n, p2, p2, p3);
  	    ip [j++] += n;
	}	    

	// Scale, filter and bandsplit P,Q.
	for (c = 0; c < _nsig3; c++)
	{
	    p1 = ip [j];
            p2 = &(_ip3lf [c][0]);
            p3 = &(_ip3hf [c][0]);
	    if (_comp & COMP_NFEIP) _nffilt3 [c].process (n, p1, p2, _scale [j]);
            else for (i = 0; i < n; i++) p2 [i] = _scale [j] * p1 [i];
  	    if (_nband > 1) _xover [j].process (n, p2, p2, p3);
  	    ip [j++] += n;
	}	    

	// Get current signal selection.
        switch (_inp0)
        {
	case TEST_OFF:
	    p1 = _sig0;
	    break;
	case TEST_INT:
	    p1 = _test + _itst;
	    break;
	case TEST_EXT:
	    p1 = ts;
	    break;
        }

	// Loop over all outputs.
	for (c = 0, C = _chan; c < _nchan; c++, C++)
	{
	    // Decoding matrix, per order output.
	    if (_nband > 1) matr2band (n, C);
	    else            matr1band (n, C);

	    // Optional NF filtering, sum orders.
	    if (_comp & COMP_NFEOP)
	    {
                C->_nffilt1.process_add (n, _sig1, _sig0, 1);
		if (_nsig2) C->_nffilt2.process_add (n, _sig2, _sig0, 1);
		if (_nsig3) C->_nffilt3.process_add (n, _sig3, _sig0, 1);
	    }
	    else
	    {
		// Just add
		if (_nsig3) 
		{
		    for (i = 0; i < n; i++) _sig0 [i] += _sig1 [i] + _sig2 [i] + _sig3 [i];
		}
		else if (_nsig2)
		{
		    for (i = 0; i < n; i++) _sig0 [i] += _sig1 [i] + _sig2 [i];
		}
		else
		{
		    for (i = 0; i < n; i++) _sig0 [i] += _sig1 [i];
		}
	    }

	    // Mute, solo and volume logic.
	    g0 = C->_gain0;
	    g1 = 0;
            m = 1LL << c;
	    if ((_inp1 == _inp0) && (m & ~_mute) && ((m & _solo) || (!_inp0 && !_solo)))
	    {		
		g1 = _gain;
		if (_comp & COMP_LEVEL) g1 *= C->_gcorr;
	    }
	    C->_gain0 = g1;
            d = (g1 - g0) / n;
            if (fabsf (d) < 1e-9f) d = 0;

	    // Copy to delay line with gain fade.
	    p2 = C->_dline + _idel;
	    s = 0;
   	    for (i = 0; i < n; i++)
	    {
		g0 += d;
  	        p2 [i] = t = g0 * p1 [i];
		s += t * t;
	    }
	    s = sqrtf (2 * s / n);

	    // Update level measurement.
	    lm [c] = 0.98f * lm [c] + 1e-10f;
	    if (s > lm [c]) lm [c] = s;

	    // Find read indices and block sizes for delay.
	    if (_comp & COMP_DELAY)
	    {
		i = _idel - C->_delay;
	        if (i < 0) i += _dsize;
		p2 = C->_dline + i;
		n1 = _dsize - i;
		if (n1 > n) n1 = n;
   	        n2 = n - n1;
	    }
	    else
            {
		n1 = n;
		n2 = 0;
	    }

            // Copy delayed signal to output buffer.
	    p3 = op [c];
	    memcpy (p3, p2, n1 * sizeof (float));
	    p2 = C->_dline;
	    p3 += n1;
	    memcpy (p3, p2, n2 * sizeof (float));
	    p3 += n2;
	    op [c] = p3;
	}

	// Prepare for next fragment.
	nf -= n;
	ts += n;
	_idel += n;
	if (_idel >= _dsize) _idel = 0;
	_itst += n;
	if (_itst >= _tsize) _itst = 0;
        _inp0 = _inp1;
    }
}


void Decoder::matr1band (int n, Channel *C)
{
    int i;

    // Single band decoder matrix, separate outputs per order.
    for (i = 0; i < n; i++) 
    {
	_sig0 [i] = C->_c0lf * _ip0lf [i];
    }		
    if (_nsig1 == 2)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig1 [i] = C->_c1lf [0] * _ip1lf [0][i]
		      + C->_c1lf [1] * _ip1lf [1][i];
	}
    }
    else if (_nsig1 == 3)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig1 [i] = C->_c1lf [0] * _ip1lf [0][i]
		      + C->_c1lf [1] * _ip1lf [1][i]
		      + C->_c1lf [2] * _ip1lf [2][i];
	}
    }
    if (_nsig2 == 2)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig2 [i] = C->_c2lf [0] * _ip2lf [0][i]
		      + C->_c2lf [1] * _ip2lf [1][i]; 
	}
    }
    else if (_nsig2 == 5)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig2 [i] = C->_c2lf [0] * _ip2lf [0][i]
		      + C->_c2lf [1] * _ip2lf [1][i]
		      + C->_c2lf [2] * _ip2lf [2][i]
		      + C->_c2lf [3] * _ip2lf [3][i]
		      + C->_c2lf [4] * _ip2lf [4][i];
	}
    }
    if (_nsig3 == 2)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig3 [i] = C->_c3lf [0] * _ip3lf [0][i]
		      + C->_c3lf [1] * _ip3lf [1][i]; 
	}
    }
}


void Decoder::matr2band (int n, Channel *C)
{
    int i;

    // Dual band decoder matrix, separate outputs per order.
    for (i = 0; i < n; i++) 
    {
	_sig0 [i] = C->_c0lf * _ip0lf [i] + C->_c0hf * _ip0hf [i];
    }		
    if (_nsig1 == 2)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig1 [i] = C->_c1lf [0] * _ip1lf [0][i] + C->_c1hf [0] * _ip1hf [0][i]
		      + C->_c1lf [1] * _ip1lf [1][i] + C->_c1hf [1] * _ip1hf [1][i];
	}
    }
    else if (_nsig1 == 3)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig1 [i] = C->_c1lf [0] * _ip1lf [0][i] + C->_c1hf [0] * _ip1hf [0][i]
		      + C->_c1lf [1] * _ip1lf [1][i] + C->_c1hf [1] * _ip1hf [1][i]
		      + C->_c1lf [2] * _ip1lf [2][i] + C->_c1hf [2] * _ip1hf [2][i];
	}
    }
    if (_nsig2 == 2)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig2 [i] = C->_c2lf [0] * _ip2lf [0][i] + C->_c2hf [0] * _ip2hf [0][i]
		      + C->_c2lf [1] * _ip2lf [1][i] + C->_c2hf [1] * _ip2hf [1][i];
	}
    }
    else if (_nsig2 == 5)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig2 [i] = C->_c2lf [0] * _ip2lf [0][i] + C->_c2hf [0] * _ip2hf [0][i]
	 	      + C->_c2lf [1] * _ip2lf [1][i] + C->_c2hf [1] * _ip2hf [1][i]
		      + C->_c2lf [2] * _ip2lf [2][i] + C->_c2hf [2] * _ip2hf [2][i]
		      + C->_c2lf [3] * _ip2lf [3][i] + C->_c2hf [3] * _ip2hf [3][i]
		      + C->_c2lf [4] * _ip2lf [4][i] + C->_c2hf [4] * _ip2hf [4][i];
	}
    }
    if (_nsig3 == 2)
    {
	for (i = 0; i < n; i++) 
	{
	    _sig3 [i] = C->_c3lf [0] * _ip3lf [0][i] + C->_c3hf [0] * _ip3hf [0][i]
		      + C->_c3lf [1] * _ip3lf [1][i] + C->_c3hf [1] * _ip3hf [1][i]; 
	}
    }
}
