// ----------------------------------------------------------------------------
//
//  Copyright (C) 2006-2010 Fons Adriaensen <fons@kokkinizita.net>
//    
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// ----------------------------------------------------------------------------


#include <string.h>
#include "jclient.h"


static const char *ipnames [MAXIP] = { "0w", "1x", "1y", "1z", "2r", "2s", "2t", "2u", "2v", "3p", "3q" };
static const int ipind30 [7] = { 0, 1, 2, 7, 8, 9, 10 };
static const int ipind31 [8] = { 0, 1, 2, 3, 7, 8, 9, 10 };
static const int ipind22 [9] = { 0, 1, 2, 3, 4, 5, 6, 7, 8 };



Jclient::Jclient (const char *jname, const char *jserv) :
    A_thread ("Jclient"),
    _jack_client (0),
    _jack_testip (0),
    _jname (jname),
    _state (ST_BYPASS),
    _n_inp (0),
    _n_out (0),
    _gain (0),
    _mute (0),
    _solo (0),
    _test (0)
{
    init_jack (jname, jserv);   
    _decoder.init (_fsamp);
    memset (_level, 0, MAXOP * sizeof (float));
}


Jclient::~Jclient (void)
{
    if (_jack_client) close_jack ();
}


void Jclient::init_jack (const char *jname, const char *jserv)
{
    int            i;
    char           s [16];
    jack_status_t  stat;
    int            opts;

    opts = JackNoStartServer;
    if (jserv) opts |= JackServerName;
    if ((_jack_client = jack_client_open (jname, (jack_options_t) opts, &stat, jserv)) == 0)
    {
        fprintf (stderr, "Can't connect to JACK\n");
        exit (1);
    }

    jack_set_process_callback (_jack_client, jack_static_process, (void *) this);
    jack_on_shutdown (_jack_client, jack_static_shutdown, (void *) this);

    if (jack_activate (_jack_client))
    {
        fprintf(stderr, "Can't activate JACK.\n");
        exit (1);
    }

    _jname = jack_get_client_name (_jack_client);
    _fsamp = jack_get_sample_rate (_jack_client);
    _fsize = jack_get_buffer_size (_jack_client);
    for (i = 0; i < MAXIP; i++)
    {
	sprintf (s, "in.%s", ipnames [i]);
        _jack_ipports [i] = jack_port_register (_jack_client, s, JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
	_ipmap [i] = 0;
    }
    _jack_testip = jack_port_register (_jack_client, "test", JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
}


void Jclient::close_jack ()
{
    jack_port_unregister (_jack_client, _jack_testip);
    jack_deactivate (_jack_client);
    jack_client_close (_jack_client);
}


void Jclient::jack_static_shutdown (void *arg)
{
    return ((Jclient *) arg)->jack_shutdown ();
}


void Jclient::jack_shutdown (void)
{
    send_event (EV_EXIT, 1);
}


int Jclient::jack_static_process (jack_nframes_t nframes, void *arg)
{
    return ((Jclient *) arg)->jack_process (nframes);
}


int Jclient::jack_process (jack_nframes_t nframes)
{
    unsigned int   i, s;
    int            e;
    float          *tip;
    float          *ipp [MAXIP];
    float          *opp [MAXOP];

    s = _state;
    e = get_event_nowait ();
    if (e != EV_TIME) switch (e)
    {
    case EV_GO_BYPASS:  s = ST_BYPASS;  break;
    case EV_GO_SILENCE: s = ST_SILENCE; break;
    case EV_GO_PROCESS: s = ST_PROCESS; break;
    }

    if (_state != ST_BYPASS)
    {
        for (i = 0; i < _n_out; i++) opp [i] = (float *) jack_port_get_buffer (_jack_opports [i], nframes);
	if (_state == ST_PROCESS)
	{
 	    tip = (float *) jack_port_get_buffer (_jack_testip, nframes);
  	    for (i = 0; i < _n_inp; i++)
	    {
                ipp [i] = (float *) jack_port_get_buffer (_ipmap [i], nframes);
	    }
	    _decoder.set_gain (s == ST_PROCESS ? _gain : 0);
	    _decoder.set_mute (_mute);
	    _decoder.set_solo (_solo);
	    _decoder.set_test (_test);
	    _decoder.process (nframes, ipp, opp, tip, _level);
	}
	else
	{
	    for (i = 0; i < _n_out; i++) memset (opp [i], 0, nframes * sizeof (float));
	}
    }

    if (e != EV_TIME)
    {
        _state = s;
        send_event (e, 1);
    }
    return 0;
}


void Jclient::set_config (AD_conf *C)
{
    _decoder.set_config (C);
}


void Jclient::set_matrix (AD_conf *C)
{
    _decoder.set_matrix (C);
}


void Jclient::set_ipports (AD_conf *C)
{
    int         i, n;
    const int   *p;

    switch ((C->_dec.h_ord << 4) | C->_dec.v_ord)
    {
    case 0x10:
	p = ipind30;
	n = 3;
	break;
    case 0x11:
	p = ipind31;
	n = 4;
	break;
    case 0x20:
	p = ipind30;
	n = 5;
	break;
    case 0x21:
	p = ipind31;
	n = 6;
	break;
    case 0x22:
	p = ipind22;
	n = 9;
	break;
    case 0x30:
	p = ipind30;
	n = 7;
	break;
    case 0x31:
	p = ipind31;
	n = 8;
	break;
    default:
	p = 0;
	n = 0;
    }

    for (i = 0; i < n; i++) _ipmap [i] = _jack_ipports [p [i]];
    while (i < 11) _ipmap [i++] = 0;
    _n_inp = n;
}


void Jclient::create_opports (AD_conf *C)
{
    unsigned int i;
    char s [64];

    delete_opports ();
    _n_out = C->_dec.nspkr;
    for (i = 0; i < _n_out; i++)
    {
	sprintf (s, "out_%s", C->_speakers [i]._label);
        _jack_opports [i] = jack_port_register (_jack_client, s, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
    }
}


void Jclient::delete_opports (void)
{
    unsigned int i;

    for (i = 0; i < _n_out; i++)
    {
	jack_port_unregister (_jack_client, _jack_opports [i]);
    }
    _n_out = 0;
}


void Jclient::connect_opports (AD_conf *C)
{
    unsigned int i;
    char s [64];

    disconn_opports ();
    for (i = 0; i < _n_out; i++)
    {
	sprintf (s, "%s:out_%s", _jname, C->_speakers [i]._label);
	jack_connect (_jack_client, s, C->_speakers [i]._jackp);
    }
}


void Jclient::disconn_opports (void)
{
    unsigned int i;

    for (i = 0; i < _n_out; i++)
    {
	jack_port_disconnect (_jack_client, _jack_opports [i]);
    }
}


