#!/usr/bin/env python3
#
# Copyright (C) 2021 The Android Open Source Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

""" This script generates the Android.run-test.bp build file"""

import os, textwrap

def main():
  test_dir = os.path.dirname(__file__)
  with open(os.path.join(test_dir, "Android.run-test.bp"), mode="wt") as f:
    f.write(textwrap.dedent("""
      // This file was generated by {}
      // It is not necessary to regenerate it when tests are added/removed/modified.
    """.format(os.path.basename(__file__))).lstrip())
    for mode in ["host", "target", "jvm"]:
      names = []
      # Group the tests into shards based on the last two digits of the test number.
      # This keeps the number of generated genrules low so we don't overwhelm soong,
      # but it still allows iterating on single test without recompiling all tests.
      for shard in ["{:02}".format(i) for i in range(100)]:
        name = "art-run-test-{mode}-data-shard{shard}".format(mode=mode, shard=shard)
        names.append(name)
        f.write(textwrap.dedent("""
          java_genrule {{
              name: "{name}",
              out: ["{name}.zip"],
              srcs: ["*{shard}-*/**/*"],
              defaults: ["art-run-test-data-defaults"],
              cmd: "$(location run-test-build.py) --out $(out) --mode {mode} --shard {shard} " +
                  "--bootclasspath $(location :art-run-test-bootclasspath)",
          }}
          """.format(name=name, mode=mode, shard=shard)))
      srcs = ("\n"+" "*8).join('":{}",'.format(n) for n in names)
      f.write(textwrap.dedent("""
        java_genrule {{
            name: "art-run-test-{mode}-data-merged",
            defaults: ["art-run-test-data-defaults"],
            out: ["art-run-test-{mode}-data-merged.zip"],
            srcs: [
                {srcs}
            ],
            tools: ["merge_zips"],
            cmd: "$(location merge_zips) $(out) $(in)",
        }}
        """).format(mode=mode, srcs=srcs))

if __name__ == "__main__":
  main()
