/* Copyright (C) 2002, 2003, 2004, 2005 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */

#ifndef NDEBUG
#include <iostream>
#endif

void ConnectDialog::init()
{
#ifndef NDEBUG
  std::cerr << "Initialising ConnectDialog" << std::endl;
#endif
  wallet = boost::shared_ptr< KWallet::Wallet >
    ( KWallet::Wallet::openWallet( KWallet::Wallet::LocalWallet(), winId() ) );

  if ( wallet ) {
    ERRORMACRO( wallet->createFolder( "anyMeal" ), Error, ,
                "Could not create folder \"anyMeal\" of the wallet "
                "\"kdewallet\"." );
    ERRORMACRO( wallet->setFolder( "anyMeal" ), Error, ,
                "Could not open folder \"anyMeal\" of the wallet "
                "\"kdewallet\"." );

    // First read user-map:
    // users: server x database -> (users)
    wallet->readMap( "users", users );

    // Read the database-map:
    // servers: server -> (database)
    // The slots will fill in all entries now.
    if ( wallet->readMap( "databases", servers ) == 0 ) {
      for ( QMap< QString, QString >::const_iterator i = servers.begin();
            i != servers.end(); i++ )
        serverCombo->insertItem( i.key() );
    };

    // Read the map of sockets:
    // sockets: socket -> (database)
    // The slots will fill in all entries now.
    if ( wallet->readMap( "sockets", sockets ) == 0 ) {
      for ( QMap< QString, QString >::const_iterator i = sockets.begin();
            i != sockets.end(); i++ )
        socketCombo->insertItem( i.key() );
    };
    
  };

  if ( serverCombo->count() == 0 )
    serverCombo->insertItem( "localhost" );

  if ( socketCombo->count() == 0 ) {
    const char *home = getenv( "HOME" );
#ifndef NDEBUG
    std::cerr << "HOME = " << home << std::endl;
#endif
    if ( home != NULL )
      socketCombo->insertItem( QString( home ) +
                               "/.kde/share/apps/anymeal/db/mysql.sock" );
  };
  serverWidgetStack->raiseWidget( serverTypeCombo->currentItem() );
}

void ConnectDialog::createDataSource()
{
  DatabaseWizard databaseWizard( this, "databaseWizard" );

  if ( databaseWizard.exec() == QDialog::Accepted ) {

    serverTypeCombo->setCurrentItem( databaseWizard.serverTypeCombo->
                                     currentItem() );
    serverCombo->setCurrentText( databaseWizard.serverEdit->text() );
    socketCombo->setCurrentText( databaseWizard.socketEdit->text() );
    databaseCombo->setCurrentText( databaseWizard.databaseEdit->text() );
    userCombo->setCurrentText( databaseWizard.databaseUserEdit->text() );
    passWordCheckBox->setChecked( databaseWizard.databasePassWordCheckBox->
                                  isChecked() );
    passWordEdit->setText( databaseWizard.databasePassWordEdit->text() );
    accept();

  };
}

void ConnectDialog::setDatabase( const QString &text )
{
  if ( wallet ) {
    QString key( sourceKey() + '/' + text );
    if ( users.contains( key ) ) {
      QStringList list( QStringList::split( ',', users[ key ] ) );
#ifndef NDEBUG
      std::cerr << "Selecting database '" << key
                << "' with following known users: ";
      for ( int i=0; i<(signed)list.size(); i++ )
        std::cerr << ' ' << list[ i ];
      std::cerr << " (extracted from " << users[ key ] << ')'
                << std::endl;
#endif
      userCombo->clear();
      userCombo->insertStringList( list );
    } else {
      userCombo->clear();
      const char *user = getenv( "USER" );
      if ( user != NULL )
        userCombo->insertItem( user );
#ifndef NDEBUG
      std::cerr << "No users-entry for '" << key << '\'' << std::endl;
#endif
    };
  } else {
    userCombo->clear();
    const char *user = getenv( "USER" );
    if ( user != NULL )
      userCombo->insertItem( user );
  };
}

void ConnectDialog::updateWallet()
{
  if ( wallet ) {

    if ( serverTypeCombo->currentItem() == 0 ) {
      QStringList list;
      if ( servers.contains( serverCombo->currentText() ) )
        list = QStringList::split( ',',
                                   servers[ serverCombo->currentText() ] );
      if ( list.find( databaseCombo->currentText() ) == list.end() )
        list.push_back( databaseCombo->currentText() );
      servers[ serverCombo->currentText() ] = list.join( "," );
      wallet->writeMap( "databases", servers );
    } else {
      QStringList list;
      if ( sockets.contains( socketCombo->currentText() ) )
        list = QStringList::split( ',',
                                   sockets[ socketCombo->currentText() ] );
#ifndef NDEBUG
      std::cerr << "sockets = " << sockets[ socketCombo->currentText() ]
                << std::endl;
#endif
      if ( list.find( databaseCombo->currentText() ) == list.end() )
        list.push_back( databaseCombo->currentText() );
      sockets[ socketCombo->currentText() ] = list.join( "," );
#ifndef NDEBUG
      std::cerr << "updated sockets = " << sockets[ socketCombo->currentText() ]
                << std::endl;
#endif
      wallet->writeMap( "sockets", sockets );
    };

    {
      QString key( sourceKey() + '/' + databaseCombo->currentText() );
      QStringList list;
      if ( users.contains( key ) )
        list = QStringList::split( ',', users[ key ] );
      if ( list.find( userCombo->currentText() ) == list.end() )
        list.push_back( userCombo->currentText() );
      users[ key ] = list.join( "," );
      wallet->writeMap( "users", users );
    };
    
    {
      QString key( QString( "password/" ) +
                   sourceKey() + '/' + userCombo->currentText() );
      if ( passWordCheckBox->isChecked() )
        wallet->writePassword( key, passWordEdit->text() );
      else
        wallet->writePassword( key, "" );
    };

  };
}


void ConnectDialog::setServer( const QString &server )
{
  if ( serverTypeCombo->currentItem() == 0 ) {
    if ( servers.contains( server ) ) {
      QStringList list( QStringList::split( ',', servers[ server ] ) );
      databaseCombo->clear();
      databaseCombo->insertStringList( list );
#ifndef NDEBUG
      std::cerr << "Selecting server '" << server
                << "' with following known databases: ";
      for ( int i=0; i<(signed)list.size(); i++ )
        std::cerr << ' ' << list[ i ];
      std::cerr << " (extracted from " << servers[ server ] << ')'
                << std::endl;
#endif
    } else {
      databaseCombo->clear();
      databaseCombo->insertItem( "anymeal" );
    };
  };
}


void ConnectDialog::setSocket( const QString &socket )
{
  if ( serverTypeCombo->currentItem() == 1 ) {
    if ( sockets.contains( socket ) ) {
      QStringList list( QStringList::split( ',', sockets[ socket ] ) );
      databaseCombo->clear();
      databaseCombo->insertStringList( list );
#ifndef NDEBUG
      std::cerr << "Selecting socket '" << socket
                << "' with following known databases: ";
      for ( int i=0; i<(signed)list.size(); i++ )
        std::cerr << ' ' << list[ i ];
      std::cerr << " (extracted from " << sockets[ socket ] << ')'
                << std::endl;
#endif
    } else {
      databaseCombo->clear();
      databaseCombo->insertItem( "anymeal" );
    };
  };
  std::string socketstr( (const char *)socket );
  unsigned int pos = socketstr.find_last_of( "/" );
  if ( pos != std::string::npos )
    folderEdit->setText( std::string( socketstr, 0, pos ).c_str() );

}


void ConnectDialog::setUser( const QString &text )
{
  if ( wallet ) {
    // Read password using constructed variable-name.
    // password: server x user -> password
    QString key( QString( "password/" ) + sourceKey() + '/' + text );
    QString password;
    wallet->readPassword( key, password );
    passWordEdit->setText( password );
    passWordCheckBox->setChecked( !password.isEmpty() );
  } else {
    passWordCheckBox->setChecked( false );
    passWordEdit->clear();
  };
}


void ConnectDialog::setOkEnabled()
{
  okButton->setEnabled
    ( ( ( serverTypeCombo->currentItem() == 0 &&
          !serverCombo->currentText().isEmpty() ) ||
        ( serverTypeCombo->currentItem() == 1 &&
          !socketCombo->currentText().isEmpty() &&
          !folderEdit->text().isEmpty() ) ) &&
      !userCombo->currentText().isEmpty() &&
      ( !passWordCheckBox->isChecked() ||
        !passWordEdit->text().isEmpty() ) );
}


void ConnectDialog::setServerType( int type )
{
  assert( serverTypeCombo->count() == 2 );
  assert( serverTypeCombo->text( 0 ) == i18n( "Network" ) );
  assert( serverTypeCombo->text( 1 ) == i18n( "Embedded" ) );
  switch ( type ) {
  case 0:
    setServer( serverCombo->currentText() );
    break;
  case 1:
    setSocket( socketCombo->currentText() );
    break;
  default:
    assert( false );
    break;
  };
}


QString ConnectDialog::sourceKey()
{
  return serverTypeCombo->currentItem() == 0 ?
    serverCombo->currentText() :
    socketCombo->currentText();
}


void ConnectDialog::selectSocket()
{
  QString fileName =
    KFileDialog::getOpenFileName( ":dbfolder",
                                  i18n( "*.sock|Socket (*.sock)\n"
                                        "*|All Files(*)" ),
                                  this,
                                  i18n( "Select database socket" ) );
  if ( fileName != QString::null ) {
    socketCombo->setCurrentText( fileName );
  };
}


void ConnectDialog::selectFolder()
{
  QString folder =
    KFileDialog::getExistingDirectory( ":dbfolder",
                                       this,
                                       i18n( "Select database folder" ) );
  if ( folder != QString::null ) {
    folderEdit->setText( folder );
  };
}
