
#include "debug.h"

#include "aclogger.h"
#include "acfg.h"
#include "lockable.h"
#include "filereader.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <glob.h>

#include <vector>
#include <iostream>

#include <stdio.h>

using namespace MYSTD;

#define TIMEFORMAT "%a %d/%m"

namespace aclog
{

FILE *fErr(NULL), *fStat(NULL);
static pthread_mutex_t mx = PTHREAD_MUTEX_INITIALIZER;

bool open()
{
	if(acfg::logdir.empty())
		return true;
	
	string apath(acfg::logdir+"/apt-cacher.log"), epath(acfg::logdir+"/apt-cacher.err");
	
	fErr = fopen(epath.c_str(), "a");
	fStat = fopen(apath.c_str(), "a");

	return fStat && fErr;
}


void transfer(bool bDirIn, uint64_t nCount, const char *szClient, const char *szPath)
{
	if(!fStat)
		return;
	lockguard g(&mx);

	fprintf(fStat, acfg::verboselog ? "%lu|%c|%"PRIu64"|%s|%s\n" : "%lu|%c|%"PRIu64"\n",
			(unsigned long) time(NULL), bDirIn ? 'I' : 'O', nCount,
					szClient, szPath);

	if(acfg::debug || acfg::verbose)
		fflush(fStat);
}

/*
 void aclogger::err(MYSTD::string &client, MYSTD::string &msg) {
 err(client.c_str(), msg.c_str());
 }
 */

void err(const char *msg, const char *client)
{
	if(!fErr)
		return;
	
	lockguard g(&mx);

	static char buf[32];
	const time_t tm=time(NULL);
	ctime_r(&tm, buf);
	buf[24]=0;
	if (client)
		fprintf(fErr, "%s|%s: %s\n", buf, client, msg);
	else
		fprintf(fErr, "%s|%s\n", buf, msg);
#ifdef DEBUG
	if(acfg::debug>=10)
	{
		fprintf(stderr, "%s|%s\n", buf, msg);
		fflush(stderr);
	}
	fflush(fErr);
#endif
	
}

void flush()
{
	lockguard g(&mx);
	if(fErr)
		fflush(fErr);
	if(fStat)
		fflush(fStat);
}

void close()
{
	lockguard g(&mx);
	if(acfg::verbose || acfg::debug) cerr << "Closing logs...\n";
	if(fErr) fclose(fErr);
	if(fStat) fclose(fStat);
}

typedef struct {
	time_t from, to;
	double count, ratioSent;
} tRowData;

void GetStats(vector<tRowData> & out)
{
	string sDataFile=acfg::cachedir+sPathSep+"_stats_dat";
	struct stat statbuf;
	
	if ( 0!= stat(sDataFile.c_str(), &statbuf) 
			|| statbuf.st_mtime < time(NULL)-86000)
	{ // needs to be created/updated
		glob_t globbuf;

		memset(&globbuf, 0, sizeof(glob_t));
		glob((acfg::logdir+"/apt-cacher*.log").c_str(), GLOB_DOOFFS | GLOB_NOSORT,
				NULL, &globbuf);

		//cout << "wo? " << (acfg::logdir+"/*.log.*").c_str() <<endl;
		
		map<time_t,pair<uint64_t,uint64_t> > mapPeriod2InOut;
		time_t now=time(NULL);
		for (unsigned int i=0; i<globbuf.gl_pathc; i++)
		{
			if(acfg::debug)
				cerr << "Reading log file: " << globbuf.gl_pathv[i] <<endl; 
			filereader reader;
			if (!reader.OpenFile(globbuf.gl_pathv[i]))
			{
				aclog::err("Error opening a log file");
				continue;
			}
			string sLine;
			tStrVec tokens;
			while(reader.GetOneLine(sLine))
			{
				// cout << "got line: " << sLine <<endl;
				tokens.clear();
				if(Tokenize(sLine, "|", tokens)<3)
					 continue;
				 
				 // cout << "having: " << tokens[0] << ", " << tokens[1] << ", " << tokens[2]<<endl;
				 
				 time_t when=strtoul(tokens[0].c_str(),0,10);

				 time_t delta=now-when;
				 if(when>now || delta>=3600*24*28)
					 continue;
				 
#define RANGEMARK 1234
				 time_t key; // map to particular weeks or DOW
				 if(delta>=7*24*3600) // account for weeks
					key = RANGEMARK + delta / (7*3600*24); 
				 else // account for days
					 key = delta / (24*3600);
				 
				 unsigned long count=strtoul(tokens[2].c_str(),0,10);
				 
				 switch(* tokens[1].c_str()) {
				 case('I'): mapPeriod2InOut[key].first+=count; break;
				 case('O'): mapPeriod2InOut[key].second+=count; break;
				 default: return; // what the...
				 }
			}
		}
		globfree(&globbuf);
		
		// Gathering...
		
		map<time_t,pair<uint64_t,uint64_t> >::iterator it;
		for(it=mapPeriod2InOut.begin(); it!=mapPeriod2InOut.end(); it++)
		{
			tRowData data;
			data.count=it->second.first+it->second.second;
			data.ratioSent=double(it->second.second)/double(data.count);
			if(it->first < RANGEMARK)
				data.from=data.to= now - it->first * 24 * 3600;
			else
			{
				data.to = now - (it->first-RANGEMARK)*7*24*3600;
				data.from=data.to-7*24*3600;
			}
			out.push_back(data);
			// TODO: serialize, into the dat cache file
		}

	}
	else
	{ // deserialize
		filereader reader;
		string sLine;
		tStrVec tokens;
		
		if(!reader.OpenFile(sDataFile))
		{
			aclog::err("Error opening stats file");
			return;
		}
		while(reader.GetOneLine(sLine) && Tokenize(sLine, SPACECHARS, tokens)==4)
		{
			tRowData data;
			data.from=strtoul(tokens[0].c_str(),0,10);
			data.to=strtoul(tokens[1].c_str(),0,10);
			data.count=atof(tokens[2].c_str());
			data.ratioSent=atof(tokens[3].c_str());
			out.push_back(data);
			tokens.clear();
		}
	}
}
	
}

#include "maintenance.h"

reportgen::reportgen(int fd) : maintenance(fd)
{
}

void reportgen::Run(const string &s)
{
	SendDecoration(true);
	GenReport();
	SendDecoration(false);
}

void reportgen::GenReport()
{
	filereader reader;
	string sTempFile=acfg::confdir+sPathSep+"report.html";
	if(!reader.OpenFile(sTempFile))
	{
		SendMsg(string("Unable to open template file, ")+sTempFile);
		return;
	}
	next_line:
	string sLine;
	//double factor;
	int nRatWid(200);
	vector<char> buf(4000);
	string sName=GetHostname();
	
	while(reader.GetOneLine(sLine))
	{
		SetStyle(sLine);
		tStrPos repl;
		while (stmiss!=(repl=sLine.find("$SERVERIP")))
		{
			sLine.replace(repl, 9, sName);
			repl+=sName.length();
		}
		
		tStrPos pos=0, posPrev=0;
		while(true)
		{
			pos=sLine.find('@', posPrev);
			if (pos==stmiss)
			{
				SendMsg(sLine.substr(posPrev)+sCRLF);
				break;
			}
			SendMsg(sLine.substr(posPrev, pos-posPrev));
			pos++;
			if (pos>=sLine.length())
			{
				SendMsg("<font size=+2 color=red>Template error detected</font>");
				return;
			}
			switch (sLine[pos])
			{
			case ('H'):
			{
				gethostname(&buf[0], buf.size());
				SendMsg(&buf[0]);
				break;
			}
			/* bloat
			case ('F'):
			{
				factor=atof(sLine.c_str()+pos+1);
				if(FP_NORMAL!=fpclassify(factor))
					factor=1.0;
				break;
			}
			*/
			case ('W'):
			{
				nRatWid=atoi(sLine.c_str()+pos+1);
				goto next_line;
			}
			case ('T'):
			{
				vector<aclog::tRowData> data;
				aclog::GetStats(data);
				for(vector<aclog::tRowData>::iterator it=data.begin(); it!=data.end(); it++)
				{
					char tbuf[50];
					size_t zlen(0);
					ctime_r(&it->from, tbuf);
					struct tm *tmp=localtime(& it->from);
					if (!tmp) goto time_error;
					zlen=strftime(tbuf, sizeof(tbuf), TIMEFORMAT, tmp);
					if ( !zlen ) goto time_error;
					
					if (it->from!=it->to)
					{
						tmp=localtime(& it->to);
						if(!tmp) goto time_error;
						if ( 0 == strftime(tbuf+zlen, sizeof(tbuf)-zlen, " - " TIMEFORMAT , tmp))
							goto time_error;
					}
					
										
					snprintf(&buf[0], buf.size(), (sLine.substr(pos+1)+sCRLF).c_str(),	
							tbuf, it->count/1048576, int(it->ratioSent*nRatWid), int((1.0-it->ratioSent)*nRatWid));
					SendMsg(&buf[0]);
					continue;
					time_error:
					SendMsg(" Invalid time value detected, check the stats database. ");
				}
				goto next_line;
			}
			default:
				//out+="<b>Unknown template variable</b>";
				SendMsg(string("@")+sLine[pos]);
				break;
			}
			posPrev=pos+1;
		}

	}
}
