
//#define LOCAL_DEBUG
#include "debug.h"

#include "expiration.h"
#include "lockable.h"
#include "acfg.h"
#include "meta.h"
#include "filereader.h"
#include "fileitem.h"
#include "dlcon.h"
#include "dirwalk.h"
#include "header.h"
#include "job.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <glob.h>

#include <map>
#include <string>
#include <iostream>

using namespace MYSTD;

expiration::expiration(int fd) :
	maintenance(fd), now(0), nDropPrefLen(0), m_nErrorCount(0), m_repFileStream(NULL)
{
	m_szDecoFile="maint.html";
}

expiration::~expiration()
{

	if(m_repFileStream)
	{
		fclose(m_repFileStream);
		m_repFileStream=0;
	}
}
void expiration::ProcessOthers(const string & sPath, const struct stat &)
{
	// NOOP
}

void expiration::ProcessRegular(const string & sPath, const struct stat &)
{

	//string sRelName=fileNode->sPath.substr(acfg::cachedir.length()+1);

	rechecks::eFileKind kind;
	if(sPath.length()<5 || sPath.substr(sPath.length()-5, 5)!=".head")
		kind = rex.getFiletype(sPath.c_str());
	else
		kind = rechecks::FILE_PKG; // threat .head like package files here, will need to cleanup as well

	if (rechecks::FILE_PKG == kind)
	{
		tStrPos pos=sPath.rfind(sPathSep[0]);
		// must be there... 

		// First file will shadow all others with the same base name!
		
		// don't double search even on misses, detect when STL initialized it
		pair<time_t,string> & val = m_trash[sPath.substr(pos+1)];
		if (val.first==0)
		{
			val.first=now;
			val.second=sPath.substr(nDropPrefLen, pos-nDropPrefLen);
		}
	}
	else if (rechecks::FILE_INDEX == kind 
			&& sPath.compare(acfg::cachedir.size()+1, 7, "_import"))
	{
		char sum[33];
		sum[32]=0;
		
		// don't run again on the same index files, if this should ever happen
		if (filereader::GetMd5String(sPath, sum))
		{
			for(tStrVec::iterator it=m_seenList.begin();it!=m_seenList.end();it++)
				if(*it==sum)
					return;
			m_seenList.push_back(sum);
		}
		
		m_volatileFiles.push_back(sPath);
	}
}

void expiration::_LoadTrashMapFromFile()
{
	filereader reader;
	reader.OpenFile(acfg::cachedir+sPathSep+"_expending_dat");
	
	string sLine;
	
	// stuff for user info
	time_t oldest(0), newest(0);
	
	while(reader.GetOneLine(sLine))
	{
		tStrVec elems;
		if(Tokenize(sLine, "\t", elems)==3)
		{
			time_t timestamp=atoll(elems[0].c_str());
			m_trash[elems[2]]=pair<time_t,string>(timestamp, elems[1]);
			
			if(oldest==0)
			{
				oldest=newest=timestamp;
			}
			else
			{
				if(timestamp < oldest)
					oldest=timestamp;
				if(timestamp > newest)
					newest=timestamp;
			}
		}
	}
	/*
	cout << "Unreferenced: ";
	for(trashmap_t::iterator it=m_trash.begin(); it!=m_trash.end(); it++)
		fprintf(stdout, "%lu\t%s\t%s\n",  it->second.first, it->second.second.c_str(), it->first.c_str());
	*/
	
	if(m_trash.size()>0)
	{
		// pretty printing for impatient users
		char buf[200];
		
		// fix range, map to days behind
		time_t now=time(0);
		if(oldest>now)
			oldest=now;
		if(newest>now)
			newest=now;
		oldest=(now-oldest)/86400;
		newest=(now-newest)/86400;
		oldest = (oldest>acfg::extreshhold ? oldest-acfg::extreshhold : 0);
		newest = (newest>acfg::extreshhold ? newest-acfg::extreshhold : 0);
		if(oldest==newest)
			oldest++;
		const char *szTheS=(oldest>1)?"s":"";
				
		snprintf(buf, _countof(buf), "Previously detected: %lu rotten package file(s), "
				"to be deleted in about %lu-%lu day%s<br>\n", 
				(unsigned long) m_trash.size(),
				newest, oldest, szTheS);
		SendMsg(buf);
	}
}


inline void expiration::_UpdateVolatileFiles()
{

	// TODO: implement pdiff reconstruction
	
	for(tStrVec::iterator it=m_volatileFiles.begin(); it!=m_volatileFiles.end(); it++)
	{
		
		// keep related .head files safe for now
		tStrPos pos=it->rfind(*szPathSep);
		m_trash.erase( it->substr(pos+1)+".head" );
		
		
		dlcon dl;
		tFileItemPtr fi;
		acfg::tHostiVec *pBackends(NULL);
		
		string sErr;
		string sRepName, sFilePath;
		
		string sKeyPath=it->substr(acfg::cachedir.size()+1, stmiss);

		// keypath must represent a previosly resolved URL, being including either repname or real host which works
		tHttpUrl url;
#ifdef WIN32
#error rewrite for the backslashes or change reliably in the walker to slashes
#endif
		if(!url.SetHttpUrl(sKeyPath))
		{
			sErr=sKeyPath+" does not contain a valid repository or host name.";
			goto rep_update_prob;
		}
			
		fi=GetFileItem(sKeyPath);
		if(fi)
			fi->Setup(true);
		else
		{
			sErr=" could not create file item handler.";
			goto rep_update_prob;
		}
		
		SendMsg(string("Checking/Updating ")+sKeyPath+"...<br>\n");
		pBackends = acfg::GetBackendVec(& url.sHost);
		if (pBackends)
			url.sPath.erase(0, 1); // get only the path string residual
		else 
		{ // try something different, do we have the original URL?
			const header *pHead=fi->GetHeader();
			// must be well formated to be valid
			if (pHead && pHead->h[header::XORIG] && 
					0==strncmp(pHead->h[header::XORIG], "http://", 7) &&
					url.SetHttpUrl(pHead->h[header::XORIG]))
				pBackends = acfg::GetBackendVec(&url.sHost);
		}
		if(pBackends)
			dl.AddJob(fi, pBackends, url.sPath);
		else // use as hostname, and pray
			dl.AddJob(fi, url);
		
		dl.WorkLoop(true);
		{
			lockguard g(*fi);
			if(fi->status == FIST_ERROR)
				goto rep_update_prob;
		}
		
		if(fi)
		{
			DelUser(sKeyPath);
			fi.reset();
		}
		
		continue;
		
		rep_update_prob:
		m_nErrorCount++;
		
		SendMsg("Error while redownloading this file!<br> ");
		if (fi)
		{
			{
				lockguard g(*fi);
				const header *pErHead=fi->GetHeader();
				if(pErHead)
					SendMsg(string("Reason: ")+pErHead->frontLine+"<br>");
			}
			DelUser(sKeyPath);
			fi.reset();
		}
		
	}
}

inline void expiration::_ParseVolatileFilesAndHandleEntries()
{
	for(tStrVec::const_iterator it=m_volatileFiles.begin(); it!=m_volatileFiles.end(); it++)
	{
		if(it->empty())
			continue;
		
		SendMsg(string("Parsing metadata in ")+*it+"<br>\n");
		filereader reader;
		if(!reader.OpenFile(*it))
		{
			SendMsg("<font color=red>ERROR: unable to open or read this file</font><br>");
			continue;
		}
		string sLine, key, val;
		tStrVec vSumSizeName;
		string sFile=reader.GetPureFilename();
		bool bScanFilenameField=(sFile=="Packages");
		bool bAppendGz=(sFile=="Index");
		bool bNoEof(true);
		NoCaseStringMap pinfo;
		
		do
		{
			bNoEof=reader.GetOneLine(sLine);
			//cout << "file: " << *it << " line: "  << sLine<<endl;
			sFile.clear();
			vSumSizeName.clear();
			if(bScanFilenameField)
			{
				if(sLine.empty())
				{
					_HandlePkgEntry(*it, pinfo);
					pinfo.clear();
					continue;
				}
				
				if(_ParseLine(sLine, key, val))
					pinfo[key]=val;
			}
			// Index, Sources?
			else if( sLine[0]==' ' // minimum protection against crap therein 
				&& 3==Tokenize(sLine, "\t ",	vSumSizeName) 
				&&	vSumSizeName[0].length()>30
				&& atoi(vSumSizeName[1].c_str()))
			{
				pinfo["Filename"]=bAppendGz?vSumSizeName[2]+".gz":vSumSizeName[2];
				pinfo["Size"]=vSumSizeName[1];
				pinfo["SHA1"]=vSumSizeName[0];
				_HandlePkgEntry(*it, pinfo);
			}
		}
		while(bNoEof);
		
		if(!reader.CheckGoodState(false))
		{
			SendMsg("<font color=red>An error occured while reading this file, some contents may have been ignored.</font><br>");
			continue;
		}
//		cout << "found package files: "<< m_trash.size()<<endl;
	}
}

void expiration::_HandlePkgEntry(const string &sReferer, const NoCaseStringMap & info)
{
	// cares only about filenames for now, may check other things in future to be more exact
	NoCaseStringMap::const_iterator it=info.find("Filename");
	if(it==info.end())
		return;
	const string & sFilename=it->second;
	tStrPos pos=sFilename.rfind(*SZPATHSEP);
	// might have path components, act only on basename
	if(pos!=stmiss)
	{
		m_trash.erase(sFilename.substr(pos+1));
		m_trash.erase(sFilename.substr(pos+1)+".head");
	}
	else
	{
		m_trash.erase(sFilename);
		m_trash.erase(sFilename+".head");
	}
}

inline void expiration::_RemoveAndStoreStatus()
{
	string sDbFile=acfg::cachedir+sPathSep+"_expending_dat";
	FILE *f = fopen(sDbFile.c_str(), "w");
	if(!f)
	{
		SendMsg("Unable to open _expending.dat for writting, trying to recreate... ");
		unlink(sDbFile.c_str());
		f=fopen(sDbFile.c_str(), "w");
		if(f)
			SendMsg("OK<br>");
		else
		{
			SendMsg("<font color=red>FAILED. ABORTING. Check filesystem and file permissions.");
			return;
		}
	}
	
	time_t now=time(NULL);
	
	//cout << "Unreferenced: ";
	for(trashmap_t::iterator it=m_trash.begin(); it!=m_trash.end(); it++)
	{
		//if(acfg::debug)
		//	cerr << "Vgl: " << it->second.first << " < " << (now-acfg::extreshhold*86400) <<endl; 
		if(it->second.first < (now-acfg::extreshhold*86400))
		{
			string sWhat=acfg::cachedir+sPathSep+it->second.second+sPathSep+it->first;
			SendMsg(string("Removing ")+it->second.second+sPathSep+it->first+"<br>\n");
			unlink(sWhat.c_str());
			unlink((sWhat+".head").c_str());
		}
		else
			fprintf(f, "%lu\t%s\t%s\n",  it->second.first, it->second.second.c_str(), it->first.c_str());
	}
	fclose(f);
	
}



void expiration::Action(const string &)
{
	_LoadTrashMapFromFile();
	SendMsg("Locating potentially expired files in the cache...<br>\n");

	nDropPrefLen=acfg::cachedir.size()+1;
	now=time(NULL);
	DirectoryWalk(acfg::cachedir, this);
	cout << "found package files: " << m_trash.size()<<endl;

	SendMsg("Redownloading index files...<br>\n");
	_UpdateVolatileFiles();
	if(m_bErrAbort && m_nErrorCount>0)
	{
		SendMsg("<font color=\"red\">Found errors during processing, aborting as requested.</font>"
				"<!-- TELL:THE:ADMIN -->");
		return;
	}
	SendMsg("Identifying fresh files...<br>\n");
	_ParseVolatileFilesAndHandleEntries();
	_RemoveAndStoreStatus();
	_PurgeMaintLogs();
	SendMsg("Done.");

}


time_t nMaintId(0), nLastId(0);
lockable idLock;

void expiration::DumpLog(time_t id)
{
	filereader reader;
	size_t size;
	const char *data(NULL);

	if (id<=0)
		goto log_na;

	char buf[PATH_MAX];
	snprintf(buf, sizeof(buf), "%s"SZPATHSEP"maint_%lu.log",
			acfg::logdir.c_str(), id);
	if (!reader.OpenFile(buf))
		goto log_na;

	data=reader.GetBuffer(size);
	maintenance::SendMsg(data, size);
	return;

	log_na:
	SendMsg(string("Log not available"));
}

void expiration::Run(const string &cmd)
{
	SendDecoration(true, "200", m_szDecoFile);

	static lockable onlyOneCanRun;

	if(!onlyOneCanRun.tryLock())
	{
		if(cmd.find("&readonly")==stmiss)
		{
			string msg("<font color=blue>A maintenance task is already running!</font><br>"
					"<ul><li><a href=");
			msg+=cmd+"&readonly#bottom";
			msg+=">View current log messages</a></li><li><a href=/";
			msg+=acfg::reportpage+">Return to main page</a></li></ul>";
			SendMsg(msg);
		}
		else
		{
			time_t id(0);
			{
				lockguard g(idLock);
				id=nMaintId;
			}
			DumpLog(id);
			maintenance::SendMsg("<br>To be continued... (reload the page to update this view)");
		}
	}
	else
	{
		// locking worked but not intended to act now
		if(cmd.find("&readonly")!=stmiss)
		{
			string msg("<font color=blue>Previous maintainenance task is completed!</font><br>"
						"<ul><li><a href=/");
			msg	+=acfg::reportpage+">Return to main page</a></li></ul><p>Last log contents:<br>\n";
			SendMsg(msg);

			DumpLog(nLastId);
		}
		else
		{
			/*****************************************************
			 * This is the worker part
			 *****************************************************/
			{
				lockguard g(idLock);
				nMaintId=time(NULL);
			}

			char buf[PATH_MAX];
			snprintf(buf, sizeof(buf), "%s"SZPATHSEP"maint_%lu.log",
					acfg::logdir.c_str(), nMaintId);
			m_repFileStream=fopen(buf, "w");

			Action(cmd);

			SendMsg(string("<br><a href=\"/")+acfg::reportpage
					+"\">Return to main page</a>");

			{
				lockguard g(idLock);
				nLastId=nMaintId;
				nMaintId=0;
			}

			if(m_repFileStream)
			{
				fclose(m_repFileStream);
				m_repFileStream=0;
			}
		}
		onlyOneCanRun.unlock();
	}
	SendDecoration(false, "", m_szDecoFile);
}

void expiration::SendMsg(const string &x)
{
	if (m_repFileStream)
	{
		fwrite(x.data(), sizeof(char), x.length(), m_repFileStream);
		fflush(m_repFileStream);
	}
	maintenance::SendMsg(x);
}


void expiration::_PurgeMaintLogs()
{
	glob_t globbuf;
	memset(&globbuf, 0, sizeof(glob_t));
	string pattern=acfg::logdir+SZPATHSEP"maint_*.log";
	glob(pattern.c_str(), GLOB_DOOFFS | GLOB_NOSORT, NULL, &globbuf);
	if (globbuf.gl_pathc>2)
	{
		SendMsg("Found required cleanup tasks: purging maintanence logs...<br>\n");
	}
	for (unsigned int i=0; i<globbuf.gl_pathc; i++)
	{
		cerr << "Checking "<<globbuf.gl_pathv[i];
		// skip our log
		time_t id=atol(globbuf.gl_pathv[i]+acfg::logdir.size()+7);
		cerr << "id ist: "<< id<<endl;
		if (id==nMaintId)
			continue;
		cerr << "Remove: "<<globbuf.gl_pathv[i]<<endl;
		unlink(globbuf.gl_pathv[i]);
	}
	globfree(&globbuf);

}
