
#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"

#include "fileitem.h"
#include "header.h"
#include "acfg.h"
#include "acbuf.h"

#include <iostream>
#include <sstream>
#include <utime.h>
#include <errno.h>

using namespace MYSTD;

header const * fileitem::GetHeader()
{
	return &m_head;
}

fileitem::fileitem(MYSTD::string sPath) :
	condition(),
	m_bIsDynType(true),
	status(FIST_FRESH),
	m_nIncommingCount(0),
	m_nFileSize(0),	
	m_sPath(acfg::cachedir+sPathSep+sPath),
	m_sKey(sPath),
	m_filefd(-1)
{
   ldbg("item created, m_sPath: "<< m_sPath);
}

uint64_t fileitem::GetTransferCount()
{
	setLockGuard;
	uint64_t ret=m_nIncommingCount;
	m_nIncommingCount=0;
	return ret;
}

int fileitem::GetFileFd() {
	setLockGuard;
	if(status<FIST_DLGOTHEAD)
		return -1;
	ldbg("Opening " << m_sPath);
	int fd=open(m_sPath.c_str(), O_RDONLY);
	
#ifdef HAS_ADVISE
	// optional, experimental
	if(status==FIST_COMPLETE)
		posix_fadvise(fd, 0, m_nFileSize, POSIX_FADV_SEQUENTIAL);
#endif

	return fd;
}


FiStatus fileitem::Setup(bool bIsDynType) {
	
	setLockGuard;
	if(status>FIST_FRESH)
		return status;
	
	status=FIST_INITED;
	
	m_bIsDynType = bIsDynType;
	
	if(m_head.LoadFromFile(m_sPath+".head") >0 && m_head.type==header::ANSWER )
	{
		if(200 != m_head.getStatus())
			goto error_clean;
		
		struct stat stbuf;
		if (0==::stat(m_sPath.c_str(), &stbuf))
			m_nFileSize=stbuf.st_size;
					
		// some optimizations and plausibility checks
		const char *p=m_head.h[header::CONTENT_LENGTH];
		// should always have it, even in chunked mode it stored during fixation
		if(!p) goto error_clean;
		
		off_t nContLen=atol(p);
		// accept length for non-volatile files, are they complete?
		if(!bIsDynType && stbuf.st_size == nContLen) 
			status=FIST_COMPLETE;
		else if(nContLen< m_nFileSize)
			goto error_clean;
			
		// can never be set finished (unchecked) unless in offline mode
		if(bIsDynType && !acfg::offlinemode)
			status=FIST_INITED;
		
	}
	return status;

	error_clean:
			unlink((m_sPath+".head").c_str());
			m_head.clear();
			m_nFileSize=0;
			status=FIST_INITED;
			return status; // unuseable, to be redownloaded
}

bool fileitem::StoreHeader(const header & h)
{
	setLockGuard;

	ldbg("StoreHeader for " << m_sKey << ", current status: " << status);
	
	if(status > FIST_DLPENDING) // already started? error? whatever
		return false;
	
	m_nIncommingCount+=h.m_nEstimLength;
	
	// status will change, most likely... ie. BOUNCE action
	notifyAll();
	
#define BOUNCE(x) { m_head.frontLine="HTTP/1.1 "; m_head.frontLine+=x; status=FIST_ERROR; return false; }
	
	int code=h.getStatus();
	ldbg("http code: "<< code);
	bool bStoreData(false);
	
	if(code == 200)
	{
		bStoreData=true;
		m_nFileSize=0;
		m_head=h;
	}
	else if(code==206)
	{
		if(m_nFileSize<=0)
		{
			// wtf? Cannot have requested partial content
			BOUNCE("500 Unexpected Partial Response");
		}
		/*
		 * Range: bytes=453291-
		 * ...
		 * Content-Length: 7271829
		 * Content-Range: bytes 453291-7725119/7725120
		 */
		const char *p=h.h[header::CONTENT_RANGE];
		if(!p)
			BOUNCE("500 Missing Content-Range in Partial Response");
		long myfrom, myto, mylen;
		int n=sscanf(p, "bytes %ld-%ld/%ld", &myfrom, &myto, &mylen);
		if(n<=0)
			n=sscanf(p, "bytes=%ld-%ld/%ld", &myfrom, &myto, &mylen);
		
		ldbg("resuming? n: "<< n << " und myfrom: " <<myfrom << 
				" und myto: " << myto << " und mylen: " << mylen);
		if(n!=3  // check for nonsense
				|| myfrom != m_nFileSize-1
				|| myfrom<0 || mylen<0)
			BOUNCE("500 Server reports illegal range");
	
		m_nFileSize--;
		bStoreData=true;
		
		m_head=h;
		m_head.frontLine="HTTP/1.1 200 OK";
		m_head.del(header::CONTENT_RANGE);
		m_head.set(header::CONTENT_LENGTH, mylen);
	}
	else if(h.getStatus()!=304 && m_head.getStatus()==200 && m_head.type==header::ANSWER)
	{
		// just do nothing
		bStoreData=false;
	}
	else
	{
		bStoreData=false;
		// have a clear one with just the message
		m_head.frontLine=h.frontLine;
		m_head.type=header::ANSWER;
	}
	
	if(bStoreData)
	{
		string path=m_sPath+".head";
		mkbasedir(path);
		ldbg("Opening "+path);
		int count=m_head.StoreToFile(path);
		
		// unlink and retry
		if(count<0)
			unlink(path.c_str());
		count=m_head.StoreToFile(path);

		if(count<0)
		{
			if(-count!=ENOSPC)
			{
				aclog::err(string("Error writting header: ")+m_sPath+".head");
				BOUNCE("503 Cache storage error");
			}
			else 
				BOUNCE("503 OUT OF DISK SPACE");
		}

		int flags = O_WRONLY | O_CREAT;
		
		m_filefd=open(m_sPath.c_str(), flags, 00644);
		ldbg("file opened?! returned: " << m_filefd);
		
		// self-recovery from cache poisoned with files with wrong permissions
		if (m_filefd<0)
		{
			unlink(m_sPath.c_str());
			m_filefd=open(m_sPath.c_str(), flags, 00644);
			ldbg("file force-opened?! returned: " << m_filefd);
		}
		
		if (m_filefd<0)
			BOUNCE(errno==ENOSPC ? "503 OUT OF DISK SPACE" : 
			"503 Cache storage error, opening data file.");
		
		if(0!=ftruncate(m_filefd, m_nFileSize) || m_nFileSize!=lseek(m_filefd, m_nFileSize, SEEK_SET))
			BOUNCE("503 IO error, positioning");
	}
	
	status=FIST_DLGOTHEAD;
	return true;
}

bool fileitem::StoreFileData(const char *data, unsigned int size) {
	
	setLockGuard;

	// something might care, most likely... also about BOUNCE action
	notifyAll();
	
	m_nIncommingCount+=size;
	
	if(status >= FIST_ERROR || status < FIST_DLGOTHEAD)
		return false;
	
	if (size==0)
	{
		status=FIST_COMPLETE;

		// we are done! Fix header from chunked transfers?
		if (m_filefd>=0 && ! m_head.h[header::CONTENT_LENGTH])
		{
			m_head.set(header::CONTENT_LENGTH, m_nFileSize);
			m_head.StoreToFile(m_sPath+".head");
		}
	}
	else
	{
		status = FIST_DLRECEIVING;

		if (m_filefd>=0)
		{
			while(size>0)
			{
				int r=write(m_filefd, data, size);
				if (r<0) 
				{
					if(EINTR==errno || EAGAIN==errno)
						continue;
					if(ENOSPC==errno)
						BOUNCE("503 OUT OF DISK SPACE");
					BOUNCE("503 General storage error");
				}
				m_nFileSize+=r;
				size-=r;
				data+=r;
			}
			
		}
	}
	return true;
}

void fileitem::SetFailureMode(const MYSTD::string & message, FiStatus fist)
{
	ldbg("SetFailureMode: " << fist << ", " << message);
	setLockGuard;
	dbgline;
	if(m_filefd>=0)
	{
		::close(m_filefd);
		m_filefd=-1;
	}
	// maybe was already set internally before
	if(status<FIST_ERROR)
	{
		m_head.clear();
		m_head.frontLine=string("HTTP/1.1 ")+message;
		m_head.type=header::ANSWER;
		status=fist;
		notifyAll();
	}
}

fileitem::~fileitem() {

	ldbg("Destroying fitem, " << m_sKey);
	setLockGuard;
	
	m_head.clear();
	
	if(m_filefd>=0)
	{
		close(m_filefd);
	}
	ldbg("Fitem destroyed, " << m_sKey);
}





struct tFileRefEntry
{
	int nRefCount;
	tFileItemPtr ref;
	tFileRefEntry() : nRefCount(0) { };
};
static map<string, tFileRefEntry> mapItems;
//static pthread_mutex_t mapLck = PTHREAD_MUTEX_INITIALIZER;
lockable mapLck;

void DelUser(string sKey) {
   ldbg("fileitem::DelUser()");

   lockguard managementLock(mapLck); 
	
   tFileRefEntry & entry = mapItems[sKey];
   entry.nRefCount--;
   if(entry.nRefCount<=0)
   {
	   ASSERT(entry.ref.get());
	   // atomic and quick-and-dirty
	   lockguard g(*(entry.ref));
	   entry.ref->status=FIST_ERRNOUSER;
	   mapItems.erase(sKey);
   }
}

tFileItemPtr GetFileItem(MYSTD::string sPathKey)
{
	tFileItemPtr p;
	MYTRY
	{
		lockguard lockGlobalMap(mapLck);

		map<string, tFileRefEntry>::iterator it=mapItems.find(sPathKey);
		if(it!=mapItems.end())
		{
			it->second.nRefCount++;
			return it->second.ref;
		}
		p.reset(new fileitem(sPathKey));
		tFileRefEntry & entry = mapItems[sPathKey];
		if(!entry.ref)
		{
			entry.ref=p;
			entry.nRefCount++;
		}
		return entry.ref;
	}
	MYCATCH(std::bad_alloc)
	{
		if(p)
			p.reset();
		return p;
	}

}

#ifdef DEBUG

void DumpItems()
{
	lockguard lockGlobalMap(mapLck);

	map<string, tFileRefEntry>::iterator it=mapItems.begin();
	cout << "Remaining file items:\n";
	for(;it!=mapItems.end(); it++)
	{
		cout << it->second.ref->status << ": " << it->first <<endl;
	}
}
#endif
