
#define LOCAL_DEBUG
#include "debug.h"

#include "job.h"
#include <stdexcept>
#include <limits>
using namespace MYSTD;

#include "con.h"
#include "acfg.h"
#include "fileitem.h"
#include "dlcon.h"
#include "maintenance.h"

#include <sys/sendfile.h>
#include <errno.h>

// optimization on Linux; no-op for platforms that don't know it
#ifndef MSG_MORE
#define MSG_MORE 0
#endif

#define REPLEVEL (m_type==rechecks::FILE_INDEX ? 4 : (m_type==rechecks::FILE_PKG ? 5 : SPAMLEVEL))

//void DispatchAndRunMaintTask(const MYSTD::string &, int, const MYSTD::string &);


ssize_t sendfile_generic(int out_fd, int in_fd, off_t *offset, size_t count)
{
	char buf[4096];
	ssize_t cnt=0;
	
	if(!offset)
	{
		errno=EFAULT;
		return -1;
	}
	while(count>0)
	{
		int readcount=read(in_fd, buf, count>sizeof(buf)?sizeof(buf):count);
		if(readcount<=0)
		{
			if(errno==EINTR || errno==EAGAIN)
				continue;
			else
				return readcount;
		}
		
		*offset+=readcount;
		cnt+=readcount;
		count-=readcount;
		
		int nPos(0);
		while(nPos<readcount)
		{
			int r=write(out_fd, buf+nPos, readcount-nPos);
			if(r==0) continue; // not nice but needs to deliver it
			if(r<0)
			{
				if(errno==EAGAIN || errno==EINTR)
					continue;
				else
					return r;
			}
			nPos+=r;
		}
	}
	return cnt;
}

ssize_t linux_sendfile_with_fallback(int out_fd, int in_fd, off_t *offset, size_t count)
{
	ssize_t r=sendfile(out_fd, in_fd, offset, count);
	
	if(r<0 && (errno==ENOSYS || errno==EINVAL))
		return sendfile_generic(out_fd, in_fd, offset, count);
	else
		return r;
}


#ifndef __linux__
#define sendfile_linux(a,b,c,d) sendfile_generic(a,b,c,d)
#else
#define sendfile_linux(a,b,c,d) linux_sendfile_with_fallback(a,b,c,d)
#endif

job::job(header *h, con *pParent) :
	m_filefd(-1),
	m_pParentCon(pParent),
	m_bChunkMode(false),
	m_bCloseCon(false),
	m_bIsHttp11(false),
	m_state(STATE_FRESH),
	m_pReqHead(h),
	m_nSendPos(0),
	m_nRangeLast(OFFT_MAX),
	m_nCachedMaxSize(0),
	m_nChunkRemainingBytes(0)
{
	ldbg("job creating, " << m_pReqHead->frontLine << " and this: " << this);
	m_nAllDataCount=0;
}

job::~job()
{
	if (m_filefd>=0)
		close(m_filefd);

	if (m_pItem)
	{
		DelUser(m_pItem->m_sKey);
		m_pItem.reset();
	}
	
	if(m_pReqHead)
	{
		delete m_pReqHead;
		m_pReqHead=NULL;
	}

	ldbg("job destroyed");
}


const MYSTD::string & job::TakeCounts(off_t &nSizeIn, off_t &nSizeOut)
{
	nSizeIn=m_pItem ? m_pItem->GetTransferCount() : 0;
	nSizeOut=m_nAllDataCount;
	m_nAllDataCount=0;
	return m_sFileLoc;
}

void replaceChars(string &s, const char *szBadChars, char goodChar)
{
	for(string::iterator p=s.begin();p!=s.end();p++)
		for(const char *b=szBadChars;*b;b++)
			if(*b==*p)
			{
				*p=goodChar;
				break;
			}
}

void job::PrepareDownload() {

    ldbg("job prepare");
    
    string sTmp; // for raw uri and other tasks
    tHttpUrl tUrl; // parsed URL
        
    acfg::tHostiVec * pBackends(NULL); // appropriate backends
    const string * psVname(NULL); // virtual name for storage pool, if available
    
    FiStatus fistate;
    
    bool bForceFreshnessChecks; // force update of the file, i.e. on dynamic index files?

    if(!m_sErrorMsgBuffer.empty())
       	return ;
    
    if(m_pReqHead->type!=header::GET && m_pReqHead->type!=header::HEAD)
    	goto report_invpath;
    
    m_bIsHttp11=(m_pReqHead->frontLine.length()>8 && 
    		0 == m_pReqHead->frontLine.compare(
					m_pReqHead->frontLine.length()-8, 8, "HTTP/1.1"));
    

	{
		// persistent connection?
	    const char *p=m_pReqHead->h[header::CONNECTION];
		if (!p) // default: "close" IFF not http 1.1
			m_bCloseCon = !m_bIsHttp11; 
		else // normalize case
			m_bCloseCon = (0 == strcasecmp(p, "close"));
		
		tStrVec tmp;
		if(3 == Tokenize(m_pReqHead->frontLine, SPACECHARS, tmp))
			sTmp=tmp[1];
		else
			goto report_invpath; // invalid uri
	}
    
    MYTRY
	{
    	// filesystem browsing attempt?
		if(stmiss != sTmp.find("..")) goto report_notallowed;

		if (0==sTmp.compare(0, 12, "apt-cacher?/"))
		sTmp.erase(0, 12);
		if (0==sTmp.compare(0, 11, "apt-cacher/"))
		sTmp.erase(11);
		
		if(!tUrl.SetHttpUrl(sTmp))	goto report_info;

		if(!tUrl.sPort.empty() && tUrl.sPort!="80")
			goto report_invport;

		// make a shortcut
		string & sPath=tUrl.sPath;

		// kill multiple slashes
		for(tStrPos pos; stmiss != (pos = sPath.find("//")); )
			sPath.erase(pos, 1);

		ldbg("input uri: "<<tUrl.ToString());

		tStrPos nRepNameLen=acfg::reportpage.size();
		if(nRepNameLen>0 && 0==tUrl.sHost.compare(0, nRepNameLen, acfg::reportpage))
		{
			m_sMaintCmd=tUrl.sHost;
			return;
		}
		if(!tUrl.sPath.empty() && tUrl.sPath[tUrl.sPath.size()-1]=='/' )
			goto report_info;
		
		m_type = m_pParentCon->rex.getFiletype(sPath.c_str());

		if ( m_type == rechecks::FILE_INVALID ) goto report_notallowed;
		
		// got something valid, has type now, trace it
		USRDBG(REPLEVEL, "Processing new job, " << m_pReqHead->frontLine);

		// resolve to an internal location
		psVname = acfg::GetRepNameAndPathResidual(tUrl, sTmp);
		
		if(psVname)
			m_sFileLoc=*psVname+sPathSep+sTmp;
		else
			m_sFileLoc=tUrl.sHost+tUrl.sPath;
		
		// FIXME: this sucks, belongs into the fileitem
		if(acfg::stupidfs)
		{
			// fix weird chars that we don't like in the filesystem
			replaceChars(tUrl.sPath, ENEMIESOFDOSFS, '_');
			replaceChars(tUrl.sHost, ENEMIESOFDOSFS, '_');
#ifdef WIN32
			replaceChars(tUrl.sPath, "/", '\\');
#endif
		}
	}
	MYCATCH(out_of_range) // better safe...
	{
    	goto report_invpath;
    }
    
        
    bForceFreshnessChecks = ( ! acfg::offlinemode && m_type==rechecks::FILE_INDEX);
    m_pItem=GetFileItem(m_sFileLoc);
    
    if(!m_pItem)
    {
    	if(acfg::debug)
    		aclog::err(string("Error creating file item for ")+m_sFileLoc);
    	goto report_overload;
    }
    
    fistate=m_pItem->Setup(bForceFreshnessChecks);
    ldbg("Got initial file status: " << fistate);
    
    if(acfg::offlinemode) { // make sure there will be no problems later in SendData or prepare a user message
    	if(fistate==FIST_COMPLETE)
    		return; // perfect
    	if(fistate>=FIST_ERROR)
    		goto report_offlineconf;
    	return; // either way, we are done here
    }
    dbgline;
    if( fistate < FIST_DLGOTHEAD) // needs a downloader 
    {
    	dbgline;
    	if(!m_pParentCon->SetupDownloader())
    	{
    		if(acfg::debug)
    			aclog::err(string("Error creating download handler for ")+m_sFileLoc);
    		goto report_overload;
    	}
    	
    	dbgline;
    	MYTRY
		{

			if(psVname && NULL != (pBackends=acfg::GetBackendVec(psVname)))
			{
				ldbg("Backends found, using them with " + sTmp);
				m_pParentCon->m_pDlClient->AddJob(m_pItem, pBackends, sTmp);
			}
			else
			{
			    if(acfg::forcemanaged)
			    	goto report_notallowed;
			    
			    m_pParentCon->m_pDlClient->AddJob(m_pItem, tUrl);
			}
		}
		MYCATCH(std::bad_alloc) // OOM, may this ever happen here?
		{
			if(acfg::debug)
				aclog::err("Out of memory");			    		
			goto report_overload;
		};
	}
    
	return;
    
report_overload:
    m_sErrorMsgBuffer="503 Server overload, try later"; 
    return ;

report_notallowed:
    m_sErrorMsgBuffer="403 Forbidden file type or location";
    return ;

report_offlineconf:
	m_sErrorMsgBuffer="503 Unable to download in offline mode";
	return;

report_invpath:
    m_sErrorMsgBuffer="403 Invalid path specification"; 
    return ;

report_invport:
    m_sErrorMsgBuffer="403 Invalid or prohibited port"; 
    return ;

    report_info:
    m_sMaintCmd="/";
    return;
}

#define THROW_ERROR(x) { m_sErrorMsgBuffer=x; goto THROWN_ERROR; }
eJobResult job::SendData(int confd)
{
	if(!m_sMaintCmd.empty())
	{
		DispatchAndRunMaintTask(m_sMaintCmd, confd, m_pReqHead->h[header::AUTHORIZATION]);
			return R_DISCON; // just stop and close connection
	}
	
	ldbg("job::SendData");
	
	off_t nNewSize(0);
	FiStatus fistate(FIST_ERROR);

	if (confd<0|| m_state==STATE_FAILMODE)
		return R_DISCON; // shouldn't be here

	if (!m_sErrorMsgBuffer.empty()) 
	{  // should just report it
		m_state=STATE_FAILMODE;
	}
	else if(m_nCachedMaxSize>0)
	{  // short circuit the fitem check -- sendfile was interrupted, just continue there 
		nNewSize=m_nCachedMaxSize;
	}
	else if (m_pItem)
	{
		lockguard g(*m_pItem);
		
		while (true)
		{
			fistate=m_pItem->status;
			nNewSize=m_pItem->m_nFileSize;
			
			if (fistate>=FIST_ERROR)
			{
				m_sErrorMsgBuffer=m_pItem->GetHeader()->frontLine.c_str()+9;
				if(m_sErrorMsgBuffer.empty())
					m_sErrorMsgBuffer="500 Unknown error";
				// to send the error response
				return R_AGAIN;
			}
			
			if(fistate==FIST_COMPLETE || (m_nSendPos < nNewSize && fistate>=FIST_DLGOTHEAD))
				break;
			
			m_pItem->wait();
		}
		
		// left loop with usefull state of fileitem, get its header once
		if(m_RespHead.type == header::INVALID)
			m_RespHead = *(m_pItem->GetHeader());

	}
	else if(m_state != STATE_SEND_BUFFER)
	{
		ASSERT(!"no FileItem assigned and no sensible way to continue");
		return R_DISCON;
	}

	while (true) // left by returning
	{
		MYTRY // for bad_alloc in members
		{
			switch(m_state)
			{
				case(STATE_FRESH):
				{
					ldbg("STATE_FRESH");

					if(fistate<FIST_DLGOTHEAD) // be sure about that
						return R_AGAIN;
					
					m_state=STATE_SEND_BUFFER;
					m_backstate=STATE_HEADER_SENT; // might change without body
					
					if(m_RespHead.type != header::ANSWER)
						THROW_ERROR("500 Rotten Data");
					
					//h.del("Content-Length"); // TESTCASE: Force chunked mode
					
					m_sPrependBuf=m_RespHead.frontLine+"\r\n";
					
					bool bSendBody(true);
					// no data for error heads
					if(m_RespHead.getStatus() != 200)
						bSendBody=false;
					
					// make sure there is no junk inside
					// if missing completely, chunked mode is to be used below
					if(m_RespHead.h[header::CONTENT_LENGTH])
					{ 
						if(0==atol(m_RespHead.h[header::CONTENT_LENGTH]))
							bSendBody=false;
						
						/*
						if(!bSendBody && ! m_pReqHead->type==header::HEAD)
							m_RespHead.del(header::CONTENT_LENGTH);
							*/
					}
					
					if( ! bSendBody)
					{
						m_backstate=STATE_ALLDONE;
					}								
					else
					{
						
						if(! m_RespHead.h[header::CONTENT_LENGTH])
						{
							// unknown length but must have data, will have to improvise: prepare chunked transfer
							if(m_bIsHttp11 != 0)
							THROW_ERROR("505 HTTP version not supported for this file"); // you don't accept this, go away
							m_bChunkMode=true;
							m_RespHead.set(header::TRANSFER_ENCODING, "chunked");
						}
						else
						{ // has content length, can do optimizations?
							const char *pIfmo = m_pReqHead->h[header::RANGE] ? 
									m_pReqHead->h[header::IFRANGE] : m_pReqHead->h[header::IF_MODIFIED_SINCE];
							bool bGotLen(false);

							bool bFreshnessForced=(m_type!=rechecks::FILE_INDEX || m_pReqHead->h[header::XORIG]);
							bool bFreshnessTested=(pIfmo && m_RespHead.h[header::LAST_MODIFIED] &&
									0==strcmp(pIfmo, m_RespHead.h[header::LAST_MODIFIED]));
							
							// is it fresh? or is this relevant? or is range mode forced?
							if(  bFreshnessForced || bFreshnessTested)		                          
							{
								/*
								 * Range: bytes=453291-
								 * ...
								 * Content-Length: 7271829
								 * Content-Range: bytes 453291-7725119/7725120
								 */
								
								const char *pRange=m_pReqHead->h[header::RANGE];
								// work around BROKEN curl request
								if(!pRange)
									pRange=m_pReqHead->h[header::CONTENT_RANGE];
								if(pRange)
								{
									// extra check to avoid timeouts since the server
									// may need time to get there. Don't care if the user is acngfs...
									if(fistate==FIST_COMPLETE || m_pReqHead->h[header::XORIG])
									{
										int r=sscanf(pRange, "bytes=%ld-%ld", 
												&m_nSendPos, &m_nRangeLast);
										// work around BROKEN curl request
										if(r<=0)
											r=sscanf(pRange, "bytes %ld-%ld", 
												&m_nSendPos, &m_nRangeLast);
										
										if(r>0)
										{
											long nSize=atol(m_RespHead.h[header::CONTENT_LENGTH]);
											
											if(r==1) // not open-end, set the limit
												m_nRangeLast=nSize-1;
											
											if(m_nSendPos>=nSize || m_nRangeLast>=nSize)
												THROW_ERROR("416 Requested Range Not Satisfiable")
											
											m_sPrependBuf="HTTP/1.1 206 Partial Response\r\n";
											char buf[100];
											sprintf(buf, "Content-Length: %ld\r\n"
													"Content-Range: bytes %ld-%ld/%ld\r\n",
													m_nRangeLast-m_nSendPos+1,
													m_nSendPos,
													m_nRangeLast,
													nSize
													);
											m_sPrependBuf+=buf;
											bGotLen=true;
										}
									}
								}
								else if(!bFreshnessForced) 
									// no range request found, and freshness test was ok -> report 304
									THROW_ERROR("304 Not Modified");
							}
							// has cont.len but header was not set in the range stuff above
							if( !bGotLen)
							{
								m_sPrependBuf+=string("Content-Length: ")
								+m_RespHead.h[header::CONTENT_LENGTH]
								+"\r\n";
							}
						}
						
											
					}
					m_sPrependBuf+=header::GetInfoHeaders();
					if(bSendBody)
						m_sPrependBuf+="Content-Type: application/octet-stream\r\n";
					if(m_RespHead.h[header::XORIG])
						m_sPrependBuf+=string("X-Original-Source: ")+m_RespHead.h[header::XORIG]+"\r\n";
										
					m_sPrependBuf+=string("Connection: ")
					+(m_bCloseCon?"close\r\n\r\n":"Keep-Alive\r\n\r\n");
					
					
					if(m_pReqHead->type==header::HEAD)
						m_backstate=STATE_ALLDONE; // simulated head is prepared but don't send stuff
					
					USRDBG(REPLEVEL, "Prepared response header for user: \n" << m_sPrependBuf);

					if(m_nSendPos>0)
					{
						// maybe needs to wait in the prerequisites check in the beginning
						return R_AGAIN;
					}
					
					continue;
				}
				case(STATE_HEADER_SENT):
				{
					ldbg("STATE_HEADER_SENT");
					
					if( fistate < FIST_DLGOTHEAD)
					{
						ldbg("ERROR condition detected: starts activity while downloader not ready")
						return R_AGAIN;
					}

					m_filefd=m_pItem->GetFileFd();
					if(m_filefd<0)
						THROW_ERROR("503 IO error");

					m_state=m_bChunkMode ? STATE_SEND_CHUNK_HEADER : STATE_SEND_PLAIN_DATA;
					ldbg("next state will be: " << m_state);
					continue;
				}
				case(STATE_SEND_PLAIN_DATA):
				{
					ldbg("STATE_SEND_PLAIN_DATA untill " << nNewSize);

					// eof?
					if(m_nSendPos==m_pItem->m_nFileSize && fistate>=FIST_COMPLETE)
						return m_bCloseCon ? R_DISCON : R_DONE;
					
					if(nNewSize>m_nRangeLast)
						nNewSize=m_nRangeLast+1;
							
					ldbg("~sendfile: on "<< m_nSendPos << " up to : " << nNewSize-m_nSendPos );
					int n=sendfile_linux(confd, m_filefd, &m_nSendPos, nNewSize-m_nSendPos);
					ldbg("~sendfile: " << n << " new m_nSendPos: " << m_nSendPos);

					if(n>0)
						m_nAllDataCount+=n;
					
					 // sendfile was not done? Store that size to skip fitem polling next time, or reset it
					m_nCachedMaxSize = (nNewSize>m_nSendPos) ? nNewSize : 0;
					
					if(m_nSendPos>m_nRangeLast)
						return m_bCloseCon ? R_DISCON : R_DONE;
					
					if(n<0)
						THROW_ERROR("400 Client error");
					
					
					return R_AGAIN;
				}
				case(STATE_SEND_CHUNK_HEADER):
				{

					m_nChunkRemainingBytes=nNewSize-m_nSendPos;

					ldbg("STATE_SEND_CHUNK_HEADER for " << m_nChunkRemainingBytes);
					// if not on EOF then the chunk must have remaining size (otherwise the state would have been changed)
					//ASSERT( 0==(s&FIST_EOF) || 0==m_nChunkRemainingBytes);

					char buf[23];
					snprintf(buf, 23, "%x\r\n", m_nChunkRemainingBytes);
					m_sPrependBuf=buf;

					if(m_nChunkRemainingBytes==0)
						m_sPrependBuf+="\r\n";

					m_state=STATE_SEND_BUFFER;
					m_backstate=STATE_SEND_CHUNK_DATA;
					continue;
				}
				case(STATE_SEND_CHUNK_DATA):
				{
					ldbg("STATE_SEND_CHUNK_DATA");

					if(m_nChunkRemainingBytes==0)
						return m_bCloseCon ? R_DISCON : R_DONE; // yeah...

					int n=sendfile_linux(confd, m_filefd, &m_nSendPos, m_nChunkRemainingBytes);
					if(n<0)
						THROW_ERROR("400 Client error");
					m_nChunkRemainingBytes-=n;
					m_nAllDataCount+=n;
					if(m_nChunkRemainingBytes<=0)
					{ // append final newline
						m_sPrependBuf="\r\n";
						m_state=STATE_SEND_BUFFER;
						m_backstate=STATE_SEND_CHUNK_HEADER;
						continue;
					}
					return R_AGAIN;
				}
				case(STATE_SEND_BUFFER):
				{
					// special DELICATE state. Something fails here only when the connection is dead
					// and this state will be used by the error handler to send feedback.
					// ... or we send the report page
					MYTRY
					{
						ldbg("prebuf sende: "<< m_sPrependBuf);
						int r=send(confd, m_sPrependBuf.data(), m_sPrependBuf.length(), 
								(m_backstate == STATE_ERRORCONT || m_backstate == STATE_NOWAYOUT) ? 0 : MSG_MORE);
						if (r<0)
						{
							if (errno==EAGAIN || errno==EINTR || errno == ENOBUFS)
							{
								return R_AGAIN;
							}
							return R_DISCON;
						}
						m_nAllDataCount+=r;
						m_sPrependBuf.erase(0, r);
						
						USRDBG(REPLEVEL, "Sent " << r 
								<< " bytes of header data, remaining contents\n" 
								<< m_sPrependBuf);
						
						if(m_sPrependBuf.empty())
						{
							USRDBG(REPLEVEL, "Returning to last state, " << m_backstate);
							m_state=m_backstate;
							continue;
						}
					}
					MYCATCH(...)
					{
						return R_DISCON;
					}
					return R_AGAIN;
				}
				
				case (STATE_FAILMODE):
					goto THROWN_ERROR;
				
				case (STATE_ERRORCONT):
					if(m_bCloseCon)
					{
						USRDBG(REPLEVEL, "Job done, Connection:close found or failed -> trigger connection shutdown");
						return R_DISCON;
					}
					USRDBG(REPLEVEL, "Job done, connection will persist");
					return R_DONE;
				

				case(STATE_ALLDONE):
					return R_DONE;
				
				case(STATE_NOWAYOUT):
				default:
					return R_DISCON;
					
			}
			
			continue;
			
			
	THROWN_ERROR:
			ldbg("Processing error: " << m_sErrorMsgBuffer);
			if(m_nAllDataCount==0) // don't send errors when data header is underway
			{
				m_sPrependBuf=string("HTTP/1.1 ")+m_sErrorMsgBuffer
				+ "\r\nConnection: " + (m_bCloseCon?"close\r\n":"Keep-Alive\r\n")
				+ header::GetInfoHeaders() +
				+ "X-Original-Source: " + m_sFileLoc + "\r\n\r\n";
				m_backstate=STATE_ERRORCONT;
				m_sErrorMsgBuffer.clear();
				USRDBG(REPLEVEL, "Response error or message, will continue (for " << m_sFileLoc<<")");
				//USRDBG(REPLEVEL, h.as_string(true));
				m_state=STATE_SEND_BUFFER;

				continue;
			}
			ldbg("Headers already sent -> forcing abort")
			return R_DISCON;
		}
		MYCATCH(bad_alloc) {
			// TODO: report memory failure?
			return R_DISCON;
		}
		ASSERT(!"UNREACHED");
	}
}
