#ifndef CSMAPPING_H_
#define CSMAPPING_H_

#include <string.h>
#include "meta.h"
#include "filereader.h"

// mapping from symbolic names to preinitialized strings
typedef enum {
   CSTYPE_INVALID=0,
   CSTYPE_MD5=1,
   CSTYPE_SHA1=2,
   CSTYPE_SHA256=3
} CSTYPES;

/*
static MYSTD::string csNames[] =
{
   "UnknownChecksum",
   "MD5sum",
   "SHA1",
   "SHA256"
};
*/

// kind of file identity, compares by file size and checksum (MD5 or SHA1)
struct tFingerprint {
	off_t size;
	CSTYPES csType;
	bool bUnpack;
	uint8_t csum[20];
	
	tFingerprint() : csType(CSTYPE_MD5), bUnpack(false) {};
	bool ReadCsFromString(const MYSTD::string & in)
	{
		return CsAsciiToBin(in.c_str(), csum, CSTYPE_MD5==csType?16:20);
	}
	bool ReadCsFromFile(const MYSTD::string & path)
	{
		return (CSTYPE_MD5==csType)
		? filereader::GetMd5Sum(path, csum, bUnpack, size) 
		: filereader::GetSha1Sum(path, csum, bUnpack, size);
	}
	MYSTD::string GetCsAsString() const {
		return CsBinToString(csum, CSTYPE_MD5==csType?16:20);
	}
	bool operator==(const tFingerprint & other) const
	{
		if(other.csType!=csType || size!=other.size)
			return false;
		return 0==memcmp(csum, other.csum, csType==CSTYPE_MD5 ? 16 : 20);
	}
};
struct ltfingerprint
{
	bool operator()(const tFingerprint &a, const tFingerprint &b)
	{
		if(a.size!=b.size) 
			return a.size<b.size;
		if(a.csType!=b.csType)
			return a.csType<b.csType;
		return(memcmp(a.csum, b.csum, a.csType==CSTYPE_MD5 ? 16 : 20)<0);
	}
};

struct tRemoteFileInfo
{
	tFingerprint fpr;
	MYSTD::string sDirectory, sFileName;
	inline void SetInvalid() {
		sFileName.clear();
		fpr.csType=CSTYPE_INVALID;
		fpr.size=0;
	}
	inline bool IsUsable() {
		return (!sFileName.empty() && fpr.csType!=CSTYPE_INVALID && fpr.size>0);
	}
};


/** For IMPORT:
 * Helper class to store attributes for different purposes. 
 * They need to be stored somewhere, either in fingerprint or
 * another struct including fingerprint, or just here :-(.
 */
struct tImpFileInfo
{
    MYSTD::string sPath;
    
    time_t mtime;
    bool bFileUsed;
    
    inline tImpFileInfo(const MYSTD::string & s, time_t m) :
        sPath(s), mtime(m), bFileUsed(false) {};
    inline tImpFileInfo() : mtime(0), bFileUsed(false) {};
};
struct ltCacheKeyComp
{
  bool operator()(const tImpFileInfo &a, const tImpFileInfo &b) const
  {
#if defined(WINDOWS) || defined(WIN32)
#warning FIXME, make this use nocasestringmap
#endif
	  if(a.mtime!=b.mtime)
		  return a.mtime<b.mtime;
      return (a.sPath<b.sPath);
  }
};

typedef MYSTD::map<tFingerprint, tImpFileInfo, ltfingerprint> tImportMap;
typedef MYSTD::map<tImpFileInfo, tFingerprint, ltCacheKeyComp> tFprCacheMap;


#endif /*CSMAPPING_H_*/
